package mysqls

import (
	"context"
	"database/sql"
	"fmt"
	_ "github.com/go-sql-driver/mysql"
	"strings"
	"sync"
	"time"

	logger "a.yandex-team.ru/direct/infra/go-libs/pkg/logformat"
	"a.yandex-team.ru/direct/infra/go-libs/pkg/trylock"
	"golang.yandex/hasql/checkers"
)

const MaxUpdateTimeMaster = 5 * time.Second

type Options func()

//на основе списка мониторинговых групп, запускает потоки для проверки ролей баз
func RunCheckRolesMySQLThread(monGroup *MysqlMonitoringGroup, wg *sync.WaitGroup) {
	defer wg.Done()
	ticker := time.NewTicker(5 * time.Second)
	for range ticker.C {
		monGroup.lock.Lock()
		for _, istat := range *monGroup.InstancesStatus {
			if !istat.TryLock() {
				continue
			}
			go func(istat *MysqlInstanceStatus) {
				defer istat.Unlock()
				defer istat.Close()
				ticker := time.NewTicker(1 * time.Second)
				for range ticker.C {
					if !istat.CheckConnect() {
						istat.Reconnect()
					}
					select {
					case <-istat.cntx.Done():
						return
					default:
						istat.UpdateMySQLRole()
					}
				}
			}(istat)
		}
		monGroup.lock.Unlock()
	}
}

type MySQLGroup interface {
	Instances() []MySQLInstance
	TryLock() *trylock.Mutex
}

//MySQLInstance type
type MySQLInstance struct {
	User         string  `json:"user"`
	Password     *string `json:"-"`
	Host         string  `json:"host"`
	Port         int     `json:"port"`
	DB           string  `json:"db"`
	InstanceName string  `json:"instanceName"`
}

func NewMySQLInstance(host string, port int, user, passwd, db, name string) MySQLInstance {
	return MySQLInstance{
		User:         user,
		Host:         host,
		Port:         port,
		Password:     &passwd,
		DB:           db,
		InstanceName: name,
	}
}

func (myi MySQLInstance) Address() string {
	return fmt.Sprintf("%s:%s@(%s:%d)/%s",
		myi.User,
		*myi.Password,
		myi.Host,
		myi.Port,
		myi.DB)
}

func (myi MySQLInstance) Connect() (*sql.DB, error) {
	return sql.Open("mysql", myi.Address())
}

//MySQLInstanceStatus type
type MysqlInstanceStatus struct {
	MySQLInstance `json:"instance"`
	MySQLRole     *string       `json:"role"`
	MySQLReadOnly *string       `json:"readOnly"`
	Conn          *sql.DB       `json:"-"`
	LastError     *string       `json:"lastError"`
	LastUpdate    *time.Time    `json:"lastUpdate"`
	Locker        trylock.Mutex `json:"-"`
	Cancel        context.CancelFunc `json:"-"`
	cntx          context.Context
}

func NewMysqlInstanceStatus(instance MySQLInstance, options ...Options) *MysqlInstanceStatus {
	var role, ro, errmsg string
	var err error
	var conn *sql.DB
	for _, option := range options {
		option()
	}
	if conn, err = instance.Connect(); err != nil {
		errmsg = fmt.Sprint(err)
	}
	ctime := time.Now()
	cntx, cancel := context.WithCancel(context.Background())
	return &MysqlInstanceStatus{
		MySQLInstance: instance,
		Conn:          conn,
		LastError:     &errmsg,
		MySQLRole:     &role,
		MySQLReadOnly: &ro,
		LastUpdate:    &ctime,
		Locker:        trylock.Mutex{},
		cntx:          cntx,
		Cancel:        cancel,
	}
}

func (mis *MysqlInstanceStatus) Reconnect() {
	_ = mis.Conn.Close()
	conn, err := mis.Connect()
	mis.Conn = conn
	*mis.LastError = fmt.Sprint(err)
}

func (mis *MysqlInstanceStatus) TryLock() bool {
	return mis.Locker.TryLock()
}

func (mis *MysqlInstanceStatus) Unlock() {
	mis.Locker.Unlock()
}

func (mis *MysqlInstanceStatus) CheckConnect() bool {
	if mis.Conn == nil {
		logger.Crit("closed mysql connect %+v", mis)
		*mis.LastError = "closed connect"
		return false
	}
	if err := mis.Conn.Ping(); err != nil {
		logger.Crit("error mysql ping %+v: %s", mis, err)
		*mis.LastError = fmt.Sprint("error ping()", err)
		return false
	}
	return true
}

func (mis *MysqlInstanceStatus) IsMaster() bool {
	return strings.EqualFold(*mis.MySQLRole, "MASTER")
}

func (mis *MysqlInstanceStatus) CheckRole() string {
	return strings.ToUpper(*mis.MySQLRole)
}

func (mis *MysqlInstanceStatus) UpdateMySQLRole() {
	cntx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()
	ok, err := checkers.MySQL(cntx, mis.Conn)
	if err != nil {
		*mis.LastError = fmt.Sprintf("error checker: %s", err)
		logger.Warn("%s", *mis.LastError)
		*mis.MySQLRole = "UNKNOWN"
		return
	} else {
		if ok {
			*mis.MySQLRole = "MASTER"
		} else {
			*mis.MySQLRole = "SLAVE"
		}
	}
	ok, err = CheckMySQLReadOnlyFlag(cntx, mis.Conn)
	if err != nil {
		*mis.LastError = fmt.Sprintf("error checker: %s", err)
		logger.Warn("%s", *mis.LastError)
		*mis.MySQLReadOnly = "UNKNOWN"
		return
	} else {
		if ok {
			*mis.MySQLReadOnly = "ON"
		} else {
			*mis.MySQLReadOnly = "OFF"
		}
	}
	*mis.LastUpdate = time.Now()
	logger.Debug("instance: %+v, mysql_role: %s, read_only flag: %s, last error: %s",
		mis, *mis.MySQLRole, *mis.MySQLReadOnly, *mis.LastError)
}

func (mis *MysqlInstanceStatus) Close() {
	if err := mis.Conn.Close(); err != nil {
		logger.Crit("error close connect, %s", err)
		*mis.LastError = "connect already closed"
	}
}

func CheckMySQLReadOnlyFlag(cntx context.Context, db *sql.DB) (bool, error) {
	var readOnly int
	rows, err := db.QueryContext(cntx, "SELECT @@read_only AS read_only")
	if err != nil {
		return false, err
	}
	defer func() { _ = rows.Close() }()
	if hasRows := rows.Next(); !hasRows {
		return false, fmt.Errorf("empty row")
	}
	if err := rows.Scan(&readOnly); err != nil {
		return false, err
	}
	if readOnly > 0 {
		return true, nil
	}
	return false, nil
}

//MysqlMonitoringGroup
type MysqlMonitoringGroup struct {
	InstancesStatus *[]*MysqlInstanceStatus `json:"instancesStatus"`
	lock            *trylock.Mutex
	wg              *sync.WaitGroup
}

func NewMysqlMonitoringGroup() MysqlMonitoringGroup {
	var instances []*MysqlInstanceStatus
	return MysqlMonitoringGroup{
		InstancesStatus: &instances,
		lock:            &trylock.Mutex{},
	}
}

//Добавляем инстансы в мониторингову группу
func (msg *MysqlMonitoringGroup) Append(instnaces ...*MysqlInstanceStatus) {
	if lock := msg.Locker(); lock != nil {
		lock.Lock()
		defer lock.Unlock()
		for _, i := range instnaces {
			j := *msg.InstancesStatus
			*msg.InstancesStatus = append(j, i)
		}
	}
}

//Удаляем инстанс(хост + имя инстанса) из группы мониторинга
func (msg *MysqlMonitoringGroup) Delete(host, instanceName string) {
	var deleted []int
	for i, instance := range *msg.InstancesStatus {
		if strings.EqualFold(host, instance.Host) &&
			strings.EqualFold(instanceName, instance.InstanceName) {
			deleted = append(deleted, i)
			instance.Cancel()
		}
	}
	if len(deleted) > 0 {
		msg.lock.Lock()
		defer msg.lock.Unlock()
		var cnt, num int
		for cnt, num = range deleted {
			(*msg.InstancesStatus)[num] = (*msg.InstancesStatus)[len(*msg.InstancesStatus)-cnt-1]
		}
		*msg.InstancesStatus = (*msg.InstancesStatus)[:len(*msg.InstancesStatus)-cnt-1]
	}
}

//Проверяем наличие инстанса в мониторинговой группе
func (msg *MysqlMonitoringGroup) Has(instnace, host string) bool {
	if lock := msg.Locker(); lock != nil {
		lock.Lock()
		defer lock.Unlock()
		for _, i := range *msg.InstancesStatus {
			//msg.wg.Add(1)
			if strings.EqualFold(i.Host, host) && strings.EqualFold(i.InstanceName, instnace) {
				return true
			}
		}
	}
	return false
}

func (msg *MysqlMonitoringGroup) Print() {
	for _, instance := range *msg.InstancesStatus {
		fmt.Printf("%+v, status: %t, role: %s\n", instance, instance.IsMaster(), instance.CheckRole())
	}
}

func (msg *MysqlMonitoringGroup) Locker() *trylock.Mutex {
	if msg != nil {
		return msg.lock
	}
	return nil
}

//возвращает группу машин с указанным именем инстанса
func (msg *MysqlMonitoringGroup) GetGroupByInstanceName(name string) MysqlMonitoringGroup {
	result := NewMysqlMonitoringGroup()
	if lock := msg.Locker(); lock != nil {
		lock.Lock()
		defer lock.Unlock()
		if msg != nil {
			for _, i := range *msg.InstancesStatus {
				if strings.EqualFold(i.InstanceName, name) {
					result.Append(i)
				}
			}
		}
		result.lock = msg.lock
		result.wg = msg.wg
	}
	return result
}

//возвращает список всех имён инстансов, участвующих в переключении
func (msg *MysqlMonitoringGroup) GetActiveInstanceNames() []string {
	var instanceNames []string
	tmp := make(map[string]int)
	if lock := msg.Locker(); lock != nil {
		lock.Lock()
		defer lock.Unlock()
		if msg != nil {
			for _, i := range *msg.InstancesStatus {
				tmp[i.InstanceName] = 1
			}
		}
		for key := range tmp {
			instanceNames = append(instanceNames, key)
		}
	}
	return instanceNames
}

func (msg *MysqlMonitoringGroup) GetGroupByHost(host string) MysqlMonitoringGroup {
	result := NewMysqlMonitoringGroup()
	if lock := msg.Locker(); lock != nil {
		lock.Lock()
		defer lock.Unlock()
		if msg != nil {
			for _, i := range *msg.InstancesStatus {
				if strings.EqualFold(i.Host, host) {
					result.Append(i)
				}
			}
		}
		result.lock = msg.lock
		result.wg = msg.wg
	}
	return result
}

func (msg *MysqlMonitoringGroup) GetUpdateTimeByHost(host string) (time.Time, error) {
	if lock := msg.Locker(); lock != nil {
		lock.Lock()
		defer lock.Unlock()
		if msg != nil {
			for _, i := range *msg.InstancesStatus {
				if strings.EqualFold(i.Host, host) {
					return *i.LastUpdate, nil
				}
			}
			return time.Time{}, fmt.Errorf("not found host %s", host)
		}
	}
	return time.Time{}, fmt.Errorf("empty group")
}

func (msg *MysqlMonitoringGroup) GetMastersByInstanceName(name string) ([]string, error) {
	var readyMasters, roMasters []string
	if lock := msg.Locker(); lock != nil {
		lock.Lock()
		defer lock.Unlock()
		if msg != nil {
			for _, i := range *msg.InstancesStatus {
				if strings.EqualFold(i.InstanceName, name) && strings.EqualFold(*i.MySQLRole, "MASTER") {
					if s := *i.MySQLReadOnly; strings.EqualFold(s, "OFF") {
						readyMasters = append(readyMasters, i.Host)
					} else {
						roMasters = append(roMasters, i.Host)
					}
				}
			}
		}
		//если мастер один - то все впорядке
		if l := len(readyMasters); l == 1 {
			return readyMasters, nil
			//если мастеров не один, и ro сервера тоже отсутствуют
		} else if k := len(roMasters); l != 1 && k == 0 {
			return readyMasters, fmt.Errorf("we have %d masters for %s. Masters: %s", l, name, readyMasters)
			//если мастеров не один, и ro сервера присутствуют
		} else if k > 0 {
			return readyMasters, fmt.Errorf("we have %d masters for %s in read_only state. Masters: %s", k, name, roMasters)
		}
	}
	return readyMasters, fmt.Errorf("unknown error")
}

func (msg *MysqlMonitoringGroup) GetHostsByInstanceName(name string) ([]string, error) {
	var hosts []string
	if lock := msg.Locker(); lock != nil {
		lock.Lock()
		defer lock.Unlock()
		if msg != nil {
			for _, i := range *msg.InstancesStatus {
				if strings.EqualFold(i.InstanceName, name) {
					hosts = append(hosts, i.Host)
				}
			}
		}
		if l := len(hosts); l == 0 {
			return []string{}, fmt.Errorf("not found hosts for instance %s", name)
		}
	}
	return hosts, nil
}

//Проверяем, что имеем один мастер в группе и он имеет свежие данные
func (msg *MysqlMonitoringGroup) CheckMasterActiveGroups() (bool, error) {
	var errmsg []string
	errlog := func(group string, err error) {
		msg := fmt.Sprintf("%s for group %s", err, group)
		logger.Crit(msg)
		errmsg = append(errmsg, msg)
	}
	activeInstanceNames := msg.GetActiveInstanceNames()
	if len(activeInstanceNames) == 0 {
		return false, fmt.Errorf("empty active instance names(example: ppcdata1, ppcdata2 and etc)")
	}
	for _, name := range activeInstanceNames {
		masters, err := msg.GetMastersByInstanceName(name)
		if err == nil {
			logger.Debug("instance %s have master %s", name, masters)
		} else {
			errlog(name, err)
			continue
		}
		master := masters[0] //в GetMastersByInstanceName проверяется длина, этот вызов безопасен
		instanceHosts := msg.GetGroupByInstanceName(name)
		uptime, err := instanceHosts.GetUpdateTimeByHost(master)
		if err != nil {
			errlog(name, err)
			continue
		}
		if dur := time.Since(uptime); dur > MaxUpdateTimeMaster {
			err = fmt.Errorf("update time by host %s more %s. Update time %s", master, dur, uptime)
			errlog(name, err)
		}
	}
	if len(errmsg) > 0 {
		return false, fmt.Errorf("%s", strings.Join(errmsg, ","))
	}
	return true, nil
}

func (msg *MysqlMonitoringGroup) CheckAnyAliveMaster() (bool, error) {
	var errmsg []error
	for _, name := range msg.GetActiveInstanceNames() {
		if fqdns, err := msg.GetMastersByInstanceName(name); len(fqdns) == 1 {
			return true, nil
		} else {
			errmsg = append(errmsg, err)
		}
	}
	return false, fmt.Errorf("not found any good master by instances: %v", errmsg)
}
