package mythreads

import (
	"context"
	"fmt"
	"io/ioutil"
	"os"
	"os/signal"
	"strconv"
	"strings"
	"sync"
	"syscall"
	"time"

	"a.yandex-team.ru/direct/infra/go-libs/pkg/dbconfig"
	"a.yandex-team.ru/direct/infra/go-libs/pkg/juggler"
	logger "a.yandex-team.ru/direct/infra/go-libs/pkg/logformat"
	"a.yandex-team.ru/direct/infra/go-libs/pkg/trylock"
	"a.yandex-team.ru/direct/infra/go-libs/pkg/zklib"
)

var (
	ReadDurationZookeeper = 5 * time.Second
)

type Options func()

func SetReadDurationZookeeper(duration time.Duration) Options {
	return func() {
		ReadDurationZookeeper = duration
	}
}

type Job interface {
	New(interface{}) Job
	Do()
}

func RunJugglerSender(events juggler.JugglerMessages, wg *sync.WaitGroup) {
	defer wg.Done()
	for {
		select {
		case event := <-events:
			if err := event.Send(); err != nil {
				logger.Warn("error send event %s to juggler, %s", event.JugglerRequest, err)
			}
		default:
			time.Sleep(1 * time.Second)
		}
	}
}

type DatabaseWorkers map[dbconfig.NameShard]*DatabaseWorker

type DatabaseMeta struct {
	Hostname interface{}
	Port     interface{}
	User     interface{}
	Password interface{}
	Version  int
	Instance dbconfig.NameShard
}

type DatabaseWorker struct {
	DatabaseMeta
	Cancel      context.CancelFunc
	Lock        *trylock.Mutex
	Cntx        context.Context
	JugglerChan juggler.JugglerMessages
}

func NewDatabaseWorker(hostname, port, user, passfile interface{},
	version int, instance dbconfig.NameShard, jchan juggler.JugglerMessages) DatabaseWorker {
	cntx, cancel := context.WithCancel(context.Background())
	lock := trylock.Mutex{}
	return DatabaseWorker{
		DatabaseMeta: DatabaseMeta{
			Hostname: hostname,
			Port:     port,
			User:     user,
			Password: passfile,
			Version:  version,
			Instance: instance,
		},
		Cancel:      cancel,
		Lock:        &lock,
		Cntx:        cntx,
		JugglerChan: jchan,
	}
}

func (dw DatabaseWorker) GetHostname() string {
	return fmt.Sprint(dw.Hostname)
}

func (dw DatabaseWorker) GetPort() int {
	port, _ := strconv.ParseInt(fmt.Sprint(dw.Port), 10, 64)
	return int(port)
}

func (dw DatabaseWorker) GetUser() string {
	return fmt.Sprint(dw.User)
}

func (dw DatabaseWorker) GetPassword() string {
	switch v := dw.Password.(type) {
	case string:
		return v
	case map[string]interface{}:
		file, ok := v["file"]
		if !ok {
			fmt.Printf("not found key='file'")
			return ""
		}
		data, err := ioutil.ReadFile(fmt.Sprint(file))
		if err != nil {
			fmt.Printf("error open file %s: %s", file, err)
			return ""
		}
		return string(data)
	default:
		fmt.Printf("don't found type %t for %v", v, v)
	}
	return ""
}

func RunDatabaseWorkerThread(dbcnf *dbconfig.DBConfig, job Job, jchan juggler.JugglerMessages, wg *sync.WaitGroup) {
	defer wg.Done()
	ticker := time.NewTicker(10 * time.Second)
	defer ticker.Stop()
	runningWorkers := make(DatabaseWorkers)
	for range ticker.C {
		instanceList, err := dbcnf.AllInstances()
		if err != nil {
			logger.Printf("error get instances, %s\n", err)
			continue
		}
		instanceMap := instanceList.ByMap()

		for instance := range instanceMap {
			hostname, _ := dbcnf.GetParamForInstance(instance, "host")
			port, _ := dbcnf.GetParamForInstance(instance, "port")
			user, _ := dbcnf.GetParamForInstance(instance, "user")
			passwd, _ := dbcnf.GetParamForInstance(instance, "pass")
			newWorker := NewDatabaseWorker(hostname, port, user, passwd, (*dbcnf).GetVersion(), instance, jchan)
			if w, ok := runningWorkers[instance]; !ok {
				runningWorkers[instance] = &newWorker
			} else {
				if w.Version != newWorker.Version ||
					!strings.EqualFold(w.GetHostname(), newWorker.GetHostname()) ||
					!strings.EqualFold(w.GetUser(), newWorker.GetUser()) ||
					!strings.EqualFold(w.GetPassword(), newWorker.GetPassword()) ||
					w.GetPort() != newWorker.GetPort() {
					logger.Info("changed master hostname %s -> %s", w.GetHostname(), newWorker.GetHostname())
					logger.Info("changed master user %s -> %s", w.GetUser(), newWorker.GetUser())
					logger.Info("changed master port %s -> %s", fmt.Sprintln(w.GetPort(), newWorker.GetPort()))
					if w.Cancel != nil {
						logger.Info("dbconfig changes, reload workers")
						cancel := w.Cancel
						cancel()
					}
					runningWorkers[instance] = &newWorker
				}
			}
		}
		for instance, worker := range runningWorkers {
			if ok := worker.Lock.TryLock(); ok {
				j := job.New(*worker)
				go func(j Job, worker *DatabaseWorker) {
					defer worker.Lock.Unlock()
					j.Do()
				}(j, worker)
				if _, ok := instanceMap[instance]; !ok {
					worker.Cancel()
					delete(runningWorkers, instance)
				}
			}
		}
	}
}

type ZkConnInterface interface {
	CheckConnect() bool
	LoadNode(interface{}) (zklib.ZkNode, error)
	SaveNode(zklib.ZkNode) (zklib.ZkNode, error)
	ExistNode(interface{}) (bool, error)
}

func RunReadZookeeperThread(zkconn ZkConnInterface, zknode *zklib.ZkNode, wg *sync.WaitGroup) {
	defer wg.Done()
	sig := make(chan os.Signal, 1)
	signal.Notify(sig, syscall.SIGHUP)
	t := time.NewTicker(ReadDurationZookeeper)
	update := func() {
		if ok := zkconn.CheckConnect(); !ok {
			logger.Warn("reconnect to zookeeper")
			return
		}
		tmp, err := zkconn.LoadNode(zknode.GetPath())
		if err != nil {
			logger.Crit("error get config %s, error %s", zknode.GetPath(), err)
			return
		}
		if zknode.GetLastReadVersion() < tmp.GetNodeVersion() {
			zknode.Locker().Lock()
			*zknode.Data = *tmp.Data
			*zknode.Stat = *tmp.Stat
			*zknode.LastReadVersion = *tmp.LastReadVersion
			zknode.Locker().Unlock()
		}
		*zknode.LastReadTime = *tmp.LastReadTime
	}
	t2 := time.NewTicker(1 * time.Second)
	for range t2.C {
		select {
		case <-t.C:
			logger.Debug("cron load dbconfig")
			update()
		case s := <-sig:
			logger.Debug("signal %s load dbconfig", s)
			update()
		}
	}
}

func RunUpdateDBConfigThread(dbcnf *dbconfig.DBConfig, zknode *zklib.ZkNode, wg *sync.WaitGroup) {
	defer wg.Done()
	t3 := time.NewTicker(ReadDurationZookeeper)
	updater := func() {
		if nodeVersion := int(zknode.GetLastReadVersion()); dbcnf.GetVersion() != nodeVersion {
			zknode.Locker().Lock()
			defer zknode.Locker().Unlock()
			dbcnf.Locks.RWLock()
			defer dbcnf.Locks.RWUnlock()
			_ = dbcnf.LoadDBConfig(*zknode.Data)
			dbcnf.SetVersion(nodeVersion)
		}
	}
	for range t3.C {
		updater()
	}
}
