package oauthlib

import (
	"bytes"
	"context"
	"fmt"
	"io/ioutil"
	"os"
	"path/filepath"

	logger "a.yandex-team.ru/direct/infra/go-libs/pkg/logformat"
	"a.yandex-team.ru/library/go/yandex/oauth"
)

const (
	DefaultClientID     = "55610a34571f4f96b3899be05886fb94"
	DefaultClientSecret = "8e02445221f449c288ff836dd3dcc62c"
	DefaultTokenFile    = ".oauth.token"
)

type OAuthParams struct {
	ClientID     string
	ClientSecret string
	TokenFile    FilePath
	Login        string
}

func NewOAuthToken(clientID, clientSecret, tokenFile, login string) OAuthParams {
	if len(clientID) == 0 {
		clientID = DefaultClientID
	}
	if len(clientSecret) == 0 {
		clientSecret = DefaultClientSecret
	}
	if len(tokenFile) == 0 {
		tokenFile = DefaultTokenFile
	}
	if len(login) == 0 {
		login = os.Getenv("USER")
	}
	return OAuthParams{
		ClientID:     clientID,
		ClientSecret: clientSecret,
		TokenFile:    FilePath(tokenFile),
		Login:        login,
	}
}

func (oap OAuthParams) LoadOAuthToken() (string, error) {
	token, err := ReadOAuthTokenByFile(oap.TokenFile.FullPath())
	if err != nil {
		logger.Warn("error read oath token by file %s: %s", oap.TokenFile, err)
	}
	if len(token) == 0 {
		token, err = GetOAuthTokenBySSH(oap.Login, oap.ClientID, oap.ClientSecret)
		if err != nil {
			logger.Crit("error get oauth token by ssh: %s", err)
			return "", err
		}
		var b bytes.Buffer
		b.WriteString(token)
		if err := ioutil.WriteFile(oap.TokenFile.FullPath(), b.Bytes(), 0400); err != nil {
			logger.Warn("error write oauth token to %s: %s", oap.TokenFile.FullPath(), err)
		} else {
			logger.Info("save new oauth token to %s", oap.TokenFile.FullPath())
		}
	}
	return token, nil
}

func ReadOAuthTokenByFile(oauthFile string) (string, error) {
	token, err := ioutil.ReadFile(oauthFile)
	token = bytes.TrimSuffix(token, []byte("\n"))
	if err != nil {
		return "", err
	}
	return string(token), nil
}

func GetOAuthTokenBySSH(login, clientID, clientSecret string) (string, error) {
	if len(login) > 0 {
		return oauth.GetTokenBySSH(context.Background(), clientID, clientSecret, oauth.WithUserLogin(login))
	}
	return oauth.GetTokenBySSH(context.Background(), clientID, clientSecret)
}

type FilePath string

func (fp FilePath) FullPath() string {
	if ok := filepath.IsAbs(fmt.Sprint(fp)); ok {
		return fmt.Sprint(fp)
	}
	myHome := os.Getenv("HOME")
	return filepath.Join(myHome, filepath.Base(fmt.Sprint(fp)))
}
