package tmlib

import (
	"bytes"
	"fmt"
	"io/ioutil"
	"net/http"
	"net/url"
	"strings"

	logger "a.yandex-team.ru/direct/infra/go-libs/pkg/logformat"
	"a.yandex-team.ru/direct/infra/go-libs/pkg/sslmode"
	"a.yandex-team.ru/direct/infra/go-libs/pkg/trylock"
)

type Connection struct {
	Client *http.Client
	Lock   *trylock.Mutex
	Token  string
}

func NewConnection(token string) (Connection, error) {
	client, err := sslmode.NewHTTPSClient()
	if err != nil {
		return Connection{}, err
	}
	lock := trylock.Mutex{}
	connect := Connection{
		Client: client,
		Lock:   &lock,
		Token:  strings.Trim(token, "\n"),
	}
	return connect, nil
}

func (c *Connection) SetToken(newtoken string) {
	c.Token = newtoken
}

func (c Connection) Do(reqtype, address string, values interface{}, body []byte) (out []byte, err error) {
	var req *http.Request
	var myapi string

	switch vals := values.(type) {
	case map[string]string:
		if len(vals) > 0 {
			query := url.Values{}
			for k, v := range vals {
				query.Add(k, v)
			}
			myapi = fmt.Sprintf("%s?%s", address, query.Encode())
		} else {
			myapi = address
		}
	case map[string]interface{}:
		if len(vals) > 0 {
			query := url.Values{}
			for k, v := range vals {
				query.Add(k, fmt.Sprintf("%v", v))
			}
			myapi = fmt.Sprintf("%s?%s", address, query.Encode())
		} else {
			myapi = address
		}
	case MapStringInterface:
		if len(vals) > 0 {
			query := url.Values{}
			for k, v := range vals {
				query.Add(k, fmt.Sprintf("%s", v))
			}
			myapi = fmt.Sprintf("%s?%s", address, query.Encode())
		} else {
			myapi = address
		}
	default:
		myapi = address
	}

	if body != nil {
		req, err = http.NewRequest(reqtype, myapi, bytes.NewReader(body))
	} else {
		req, err = http.NewRequest(reqtype, myapi, nil)
	}

	if err != nil {
		return nil, err
	}

	req.Header.Set("Authorization", fmt.Sprintf("OAuth %s", c.Token))
	req.Header.Set("Content-Type", "application/json")
	//req.Header.Set("User-Agent", "curl")
	logger.Debug("load token %s with method %s url %s", req.Header.Get("Authorization"), reqtype, myapi)
	resp, err := (*c.Client).Do(req)
	if err != nil {
		return nil, err
	}
	defer func() { _ = resp.Body.Close() }()
	out, _ = ioutil.ReadAll(resp.Body)
	if resp.StatusCode != 200 {
		return nil, NewHTTPError(resp.StatusCode, resp.Status, myapi, out)
	}
	if len(out) == 0 {
		return nil, fmt.Errorf("empty body response %s", myapi)
	}
	return
}

type HTTPError struct {
	Code   int
	Status string
	URL    string
	Out    []byte
}

func NewHTTPError(code int, status, url string, out []byte) HTTPError {
	return HTTPError{
		Code:   code,
		Status: status,
		URL:    url,
		Out:    out,
	}
}

func (he HTTPError) Error() string {
	return fmt.Sprintf("http status: %s, req %s, data: %s", he.Status, he.URL, he.Out)
}
