package tmlib

import (
	"bytes"
	"encoding/json"
	"fmt"
	"strings"

	logger "a.yandex-team.ru/direct/infra/go-libs/pkg/logformat"
)

const (
	TransferAPI = "https://cdc.n.yandex-team.ru"

	OperationIDUnknown = iota
	OperationIDSuccess
	OperationIDProcess
	OperationIDFailed
)

var (
	TransferAvailibleStatus = []string{
		"TRANSFER_STATUS_UNSPECIFIED",
		"CREATING",
		"CREATED",
		"RUNNING",
		"STOPPING",
		"STOPPED",
		"ERROR",
		"SNAPSHOTTING",
		"DONE",
	}
)

func MergeBlocks(blocks ...interface{}) (MapStringInterface, error) {
	result := make(MapStringInterface)
	for _, t := range blocks {
		var b MapStringInterface
		d, err := json.Marshal(t)
		if err != nil {
			return nil, fmt.Errorf("error marshal source %s: %s", t, err)
		}
		if err := json.Unmarshal(d, &b); err != nil {
			return nil, fmt.Errorf("error unmarshal PrepareRequest %s: %s", d, err)
		}
		for key, val := range b {
			result[key] = val
		}
	}
	return result, nil
}

func (msi MapStringInterface) Marshal() ([]byte, error) {
	return json.Marshal(msi)
}

func UpdateMapKeys(srcMap, newKeys MapStringInterface) MapStringInterface {
	srcMap = DeleteEmptyKeys(srcMap)
	fmt.Printf("SRCMAP %+v\n", srcMap)
	for pk, pv := range newKeys {
		for k, v := range srcMap {
			if maps, ok := v.(map[string]interface{}); ok {
				srcMap[k] = UpdateMapKeys(maps, newKeys)
				continue
			}
			if k == pk {
				srcMap[k] = pv
			}
		}
	}
	return srcMap
}

func DeleteEmptyKeys(srcMap MapStringInterface) MapStringInterface {
	for k, v := range srcMap {
		if maps, ok := v.(map[string]interface{}); ok {
			if len(maps) > 0 {
				maps = DeleteEmptyKeys(maps)
			}
			if len(maps) == 0 {
				delete(srcMap, k)
			} else {
				srcMap[k] = maps
			}
		} else {
			if len(fmt.Sprint(v)) == 0 {
				delete(srcMap, k)
			}
		}
		if maps, ok := v.(MapStringInterface); ok {
			if len(maps) > 0 {
				maps = DeleteEmptyKeys(maps)
			}
			if len(maps) == 0 {
				delete(srcMap, k)
			} else {
				srcMap[k] = maps
			}
		} else {
			if len(fmt.Sprint(v)) == 0 {
				delete(srcMap, k)
			}
		}
	}
	return srcMap
}

//MysqlOnpremiseBlocks structure
func NewMysqlBlock(description, folderID, name, database, slotID, host,
	user, port string, pass Password, typeEndpoint, typeDelivery string, skipKeyChecks bool) MysqlBlock {
	var target interface{}
	switch typeEndpoint {
	case "mdb":
		switch typeDelivery {
		case "source":
			target = MysqlSourceBlock{
				MysqlSource{
					Database: database,
					Host:     host, //cluster-id
					User:     user,
					Port:     port,
					Password: pass,
					SlotID:   slotID,
				},
			}
		case "target":
			target = MysqlTargetBlock{
				MysqlTarget{
					Database:             database,
					Host:                 host, //cluster-id
					User:                 user,
					Port:                 port,
					Password:             pass,
					SlotID:               slotID,
					SkipConstraintChecks: skipKeyChecks,
				},
			}
		default:
		}
	case "hardware":
		switch typeDelivery {
		case "source":
			target = MysqlOnpremiseSourceBlock{
				MysqlOnpremiseSource{
					Database: database,
					Host:     host,
					User:     user,
					Port:     port,
					Password: pass,
					SlotID:   slotID,
				},
			}
		case "target":
			target = MysqlOnpremiseTargetBlock{
				MysqlOnpremiseTarget{
					Database:             database,
					Host:                 host,
					User:                 user,
					Port:                 port,
					Password:             pass,
					SlotID:               slotID,
					SkipConstraintChecks: skipKeyChecks,
				},
			}
		default:
		}
	default:
	}

	return MysqlBlock{
		Description: description,
		FolderID:    folderID,
		Name:        name,
		Target:      target,
	}
}

func (mb MysqlBlock) PrepareRequest() (MapStringInterface, error) {
	return MergeBlocks(mb, mb.Target)
}

func NewTransferBlock(description, folderID, name, srcEndpointID,
	dstEndpointID, typeTransport string) TransferBlock {
	return TransferBlock{
		FolderID:    folderID,
		SourceID:    srcEndpointID,
		TargetID:    dstEndpointID,
		Description: description,
		Type:        typeTransport,
		Name:        name,
	}
}

func (mb TransferBlock) PrepareRequest() (MapStringInterface, error) {
	return MergeBlocks(mb)
}

//ResponseTarget strusture
func (rt ResponseTarget) TargetID() string {
	return rt.ID
}

//Проверяет статус выполнения target
func (rt ResponseTarget) TargetDone() bool {
	return rt.Done
}

func (rt ResponseTarget) TargetError() error {
	return fmt.Errorf("%+v", rt.Error)
}

//Connect interface
type Connect interface {
	Do(reqtype, address string, values interface{}, body []byte) ([]byte, error)
}

//Methods API
func CreateEndpointTM(conn Connect, data []byte) (ResponseTarget, error) {
	var rt ResponseTarget
	myapi := fmt.Sprintf("%s/v1/endpoint", TransferAPI)
	logger.Debug("sending %s data %s", myapi, data)
	out, err := conn.Do("POST", myapi, nil, data)
	if err != nil {
		logger.Crit("error request %s, data %s, error %s, out: %s", myapi, data, err, out)
		return rt, err
	}
	if err := json.Unmarshal(out, &rt); err != nil {
		logger.Crit("error unmarshal %s, request %s", out, myapi)
		return rt, err
	}
	return rt, nil
}

func ListEndpointTM(conn Connect, folderID string) (Endpoints, error) {
	var endpoints Endpoints
	myapi := fmt.Sprintf("%s/v1/endpoints/list/%s", TransferAPI, folderID)
	logger.Debug("sending %s folder_id %s", myapi, folderID)
	out, err := conn.Do("GET", myapi, nil, nil)
	if err != nil {
		logger.Crit("error request %s, error %s, out: %s", myapi, err, out)
		return endpoints, err
	}
	if err := json.Unmarshal(out, &endpoints); err != nil {
		logger.Crit("error unmarshal %s, request %s", out, myapi)
		return endpoints, err
	}
	return endpoints, nil
}

func DetailEndpointTM(conn Connect, endpointID string) (Endpoint, error) {
	var endpoint Endpoint
	myapi := fmt.Sprintf("%s/v1/endpoint/%s", TransferAPI, endpointID)
	logger.Debug("sending %s folder_id %s", myapi, endpointID)
	out, err := conn.Do("GET", myapi, nil, nil)
	if err != nil {
		logger.Crit("error request %s, error %s, out: %s", myapi, err, out)
		return endpoint, err
	}
	if err := json.Unmarshal(out, &endpoint); err != nil {
		logger.Crit("error unmarshal %s, request %s", out, myapi)
		return endpoint, err
	}
	return endpoint, nil
}

func CreateTransferTM(conn Connect, data []byte) (ResponseTarget, error) {
	var rt ResponseTarget
	myapi := fmt.Sprintf("%s/v1/transfer", TransferAPI)
	logger.Debug("sending %s data %s", myapi, data)
	out, err := conn.Do("POST", myapi, nil, data)
	if err != nil {
		logger.Crit("error request %s, error %s, out: %s", myapi, err, out)
		return rt, err
	}
	if err := json.Unmarshal(out, &rt); err != nil {
		logger.Crit("error unmarshal %s, request %s", out, myapi)
		return rt, err
	}
	return rt, nil
}

func OperationIDTM(conn Connect, operationID string) (ResponseTarget, error) {
	var rt ResponseTarget
	myapi := fmt.Sprintf("%s/v1/operation/%s", TransferAPI, operationID)
	logger.Debug("sending %s operation_id %s", myapi, operationID)
	out, err := conn.Do("GET", myapi, nil, nil)
	if err != nil {
		logger.Crit("error request %s, error %s, out: %s", myapi, err, out)
		return rt, err
	}
	if err := json.Unmarshal(out, &rt); err != nil {
		logger.Crit("error unmarshal %s, request %s", out, myapi)
		return rt, err
	}
	return rt, nil
}

//Methods
func (eb Endpoints) FilterByDescription(description string) Endpoints {
	var result Endpoints
	for _, endpoint := range eb.Endpoints {
		if strings.Contains(endpoint.Description, description) {
			logger.Debug("FilterByDescription %+v", endpoint)
			result.Endpoints = append(result.Endpoints, endpoint)
		}
	}
	return result
}

func (eb Endpoints) FilterByName(name string) Endpoints {
	var result Endpoints
	for _, endpoint := range eb.Endpoints {
		if strings.Contains(endpoint.Name, name) {
			result.Endpoints = append(result.Endpoints, endpoint)
		}
	}
	return result
}

func (eb Endpoints) FilterByInstanceName(name string) Endpoints {
	return eb.FilterByName(fmt.Sprintf("%s_", name))
}

func (e Endpoint) Database() string {
	switch {
	case len(e.MysqlOnpremiseSource.Database) > 0:
		return e.MysqlOnpremiseSource.Database
	case len(e.MysqlOnpremiseTarget.Database) > 0:
		return e.MysqlOnpremiseTarget.Database
	default:
	}
	return ""
}

func UpdateEndpointTM(conn Connect, endpointID string, newVars MapStringInterface) (ResponseTarget, error) {
	var rt ResponseTarget
	oldEndpoint, err := DetailEndpointTM(conn, endpointID)
	if err != nil {
		return rt, fmt.Errorf("detail endpoint %s error: %s", endpointID, err)
	}
	updates, err := MergeBlocks(oldEndpoint)
	if err != nil {
		return rt, fmt.Errorf("convert endpoint %+v error: %s", oldEndpoint, err)
	}
	updates = UpdateMapKeys(updates, newVars)

	data, err := json.Marshal(updates)
	if err != nil {
		return rt, fmt.Errorf("error marshal %v: %s", data, err)
	}
	myapi := fmt.Sprintf("%s/v1/endpoint", TransferAPI)
	logger.Debug("sending %s data %s", myapi, data)
	out, err := conn.Do("PUT", myapi, nil, data)
	if err != nil {
		logger.Crit("error request %s, error %s, out: %s", myapi, err, out)
		return rt, err
	}
	if err := json.Unmarshal(out, &rt); err != nil {
		logger.Crit("error unmarshal %s, request %s", out, myapi)
		return rt, err
	}
	return rt, nil
}

func CreateMysqlEndpointTM(conn Connect, folderID, projectName, instanceName, databaseName, slotID, mysqlHostname,
	mysqlUser, mysqlPort string, mysqlPassword Password, typeDelivery string, skipKeyChecks bool) (ResponseTarget, error) {
	var rt ResponseTarget
	if len(folderID) == 0 || len(instanceName) == 0 || len(databaseName) == 0 || len(mysqlHostname) == 0 ||
		len(mysqlUser) == 0 || len(mysqlPort) == 0 || len(typeDelivery) == 0 || len(mysqlPassword.Key) == 0 ||
		len(mysqlPassword.VersionID) == 0 {
		msg := fmt.Errorf("found empty params. folderID: %s, instanceName: %s, databaseName: %s,"+
			" mysqlHostname %s,  mysqlUser %s, mysqlPort: %s, mysqlPassword: %+v,"+
			" typeDelivery %s, projectName: %s", folderID, instanceName, databaseName, mysqlHostname,
			mysqlUser, mysqlPort, mysqlPassword, typeDelivery, projectName)
		return rt, msg
	}

	typeEndpoint := "mdb"
	if strings.Contains(mysqlHostname, ".") {
		typeEndpoint = "hardware"
	}
	target := NewMysqlBlock(
		fmt.Sprintf("%s_%s_mysql_instance_%s", projectName, typeDelivery, instanceName),
		folderID,
		fmt.Sprintf("%s_%s_endpoint", instanceName, mysqlHostname),
		databaseName,
		slotID,
		mysqlHostname,
		mysqlUser,
		mysqlPort,
		mysqlPassword,
		typeEndpoint,
		typeDelivery,
		skipKeyChecks,
	)
	logger.Debug("target: %+v", target)

	request, err := target.PrepareRequest()
	if err != nil {
		return rt, fmt.Errorf("[createMysqlEndpoint] %s", err)
	}
	jsonData, err := request.Marshal()
	if err != nil {
		return rt, fmt.Errorf("[createMysqlEndpoint] error marshal %s: %s", jsonData, err)
	}
	return CreateEndpointTM(conn, jsonData)
}

func CreateMysqlTransferTM(conn Connect, folderID, projectName, instanceName, sourceMySQL, destinationMySQL,
	srcEndpointID, dstEndpointID, typeTransport string) (ResponseTarget, error) {
	transfer := NewTransferBlock(
		fmt.Sprintf("%s mysql tranfer %s cluster/host %s -> %s", projectName, instanceName, sourceMySQL, destinationMySQL),
		folderID,
		fmt.Sprintf("%s_%s_to_%s_transfer", instanceName, sourceMySQL, destinationMySQL),
		srcEndpointID,
		dstEndpointID,
		typeTransport,
	)
	var rt ResponseTarget
	request, err := transfer.PrepareRequest()
	if err != nil {
		return rt, fmt.Errorf("[CreateMysqlTransfer] %s", err)
	}
	jsonData, err := request.Marshal()
	if err != nil {
		return rt, fmt.Errorf("[CreateMysqlTransfer] error marshal %s: %s", jsonData, err)
	}
	return CreateTransferTM(conn, jsonData)
}

func NewTransferID(transferID string) TransferID {
	return TransferID{
		TransferID: transferID,
	}
}

func ActivateTransferTM(conn Connect, transferID string) (ResponseTarget, error) {
	var rt ResponseTarget
	tid := NewTransferID(transferID)
	data, err := json.Marshal(tid)
	if err != nil {
		return rt, fmt.Errorf("error marshal %v: %s", tid, err)
	}
	myapi := fmt.Sprintf("%s/v1/transfer/activate", TransferAPI)
	logger.Debug("sending %s data %s", myapi, data)
	out, err := conn.Do("PATCH", myapi, nil, data)
	if err != nil {
		logger.Crit("error request %s, tid %s, error %s, out: %s", myapi, tid, err, out)
		return rt, err
	}
	if err := json.Unmarshal(out, &rt); err != nil {
		logger.Crit("error unmarshal: %s, request: %s, tid: %s", out, myapi, transferID)
		return rt, err
	}
	return rt, nil
}

func LogsTransferTM(conn Connect, transferID string) ([]byte, error) {
	myapi := fmt.Sprintf("%s/v1/transfers/logs", TransferAPI)
	values := make(map[string]interface{})
	values["log_group_id"] = transferID
	values["page_size"] = 200
	out, err := conn.Do("GET", myapi, values, nil)
	if err != nil {
		logger.Crit("error request %s, error %s, out: %s", myapi, err, out)
		return out, err
	}
	return out, nil
}

func OperationsByTransferTM(conn Connect, transferID string) (TransferOperationsStatus, error) {
	var ts TransferOperationsStatus
	myapi := fmt.Sprintf("%s/v1/transfers/operations", TransferAPI)
	values := make(map[string]string)
	values["transfer_id"] = transferID
	out, err := conn.Do("GET", myapi, values, nil)
	if err != nil {
		logger.Crit("error request %s, error %s, out: %s", myapi, err, out)
		return ts, err
	}
	if err := json.Unmarshal(out, &ts); err != nil {
		logger.Crit("error unmarshal %s, request %s", out, myapi)
		return ts, err
	}
	return ts, nil
}

func (tos TransferOperationsStatus) Success() []ResponseTarget {
	var r []ResponseTarget
	for _, tos := range tos.Operations {
		if tos.Done {
			r = append(r, tos)
		}
	}
	return r
}

func (tos TransferOperationsStatus) All() []ResponseTarget {
	var r []ResponseTarget
	return append(r, tos.Operations...)
}

func AbortOperationTM(conn Connect, operationID string) (ResponseTarget, error) {
	var rt ResponseTarget
	myapi := fmt.Sprintf("%s/v1/operation/%s", TransferAPI, operationID)
	logger.Debug("delete %s", myapi)
	out, err := conn.Do("DELETE", myapi, nil, nil)
	if err != nil {
		if e, ok := err.(HTTPError); ok && e.Code != 404 {
			logger.Crit("error request %s, tid %s, error %s, out: %s", myapi, operationID, err, out)
			return rt, err
		} else {
			logger.Debug("error request %s, tid %s, error %s, out: %s", myapi, operationID, err, out)
			return rt, err
		}
	}
	if err := json.Unmarshal(out, &rt); err != nil {
		logger.Crit("error unmarshal: %s, request: %s, tid: %s", out, myapi, operationID)
		return rt, err
	}
	return rt, nil
}

func DeactivateTransferTM(conn Connect, transferID string) (ResponseTarget, error) {
	var rt ResponseTarget
	tid := NewTransferID(transferID)
	data, err := json.Marshal(tid)
	if err != nil {
		return rt, fmt.Errorf("error marshal %v: %s", tid, err)
	}
	myapi := fmt.Sprintf("%s/v1/transfer/deactivate", TransferAPI)
	logger.Debug("sending %s data %s", myapi, data)
	out, err := conn.Do("PATCH", myapi, nil, data)
	if err != nil {
		logger.Crit("error request %s, tid %s, error %s, out: %s", myapi, transferID, err, out)
		return rt, err
	}
	if err := json.Unmarshal(out, &rt); err != nil {
		logger.Crit("error unmarshal %s, request %s", out, myapi)
		return rt, err
	}
	return rt, nil
}

func CheckOperationDoneTM(conn Connect, operationID string) OperationStatus {
	rt, err := OperationIDTM(conn, operationID)
	if err != nil {
		return OperationStatus{false, fmt.Sprint(err), OperationIDFailed}
	}
	switch {
	case rt.Done && rt.Code == 0:
		msg := fmt.Sprintf("id %s success, message: %s, details: %s", rt.ID, rt.Message, rt.Details)
		return OperationStatus{true, msg, OperationIDSuccess}
	case !rt.Done && rt.Code == 0:
		msg := fmt.Sprintf("id %s process, message: %s/%s", rt.ID, rt.Message, rt.Details)
		return OperationStatus{false, msg, OperationIDProcess}
	case rt.Done && rt.Code > 0:
		msg := fmt.Sprintf("id %s failed, message: %s, details: %s, error: %s",
			rt.ID, rt.Message, rt.Details, rt.Error)
		return OperationStatus{false, msg, OperationIDFailed}
	default:
	}
	msg := fmt.Sprintf("failed parse status %+v", rt)
	return OperationStatus{false, msg, OperationIDFailed}
}

func (ost OperationStatus) GetOperationIDStatus() OperationIDStatus {
	switch {
	case ost.OperationIDCode == OperationIDSuccess:
		return OperationIDStatus{"SUCCESS"}
	case ost.OperationIDCode == OperationIDFailed:
		return OperationIDStatus{"FAILED"}
	case ost.OperationIDCode == OperationIDProcess:
		return OperationIDStatus{"PROCESS"}
	default:
	}
	return OperationIDStatus{"UNKOWN"}
}

func (ost OperationStatus) Marshal() ([]byte, error) {
	data, err := MergeBlocks(ost, ost.GetOperationIDStatus())
	if err != nil {
		return nil, fmt.Errorf("error merge OperationStatus %s: %s", data, err)
	}
	return json.Marshal(data)
}

func NewYavPassword(yavKey, yavVersionID string) (Password, error) {
	var passwd Password
	if len(yavKey) == 0 || len(yavVersionID) == 0 {
		return passwd, fmt.Errorf("empty yavKey(%d) or yavVersionID(%d)", len(yavKey), len(yavVersionID))
	}
	return Password{
		Secret{
			Key:       yavKey,
			VersionID: yavVersionID,
		},
	}, nil
}

func TransfersByStatus(conn Connect, folderID, status string) (TransfersStatus, error) {
	var transfersStatus TransfersStatus
	myapi := fmt.Sprintf("%s/v1/transfers/list/%s/%s", TransferAPI, folderID, status)
	logger.Debug("sending %s folder_id %s status %s", myapi, folderID, status)
	out, err := conn.Do("GET", myapi, nil, nil)
	if err != nil {
		logger.Crit("error request %s, error %s, out: %s", myapi, err, out)
		return transfersStatus, err
	}
	if err := json.Unmarshal(out, &transfersStatus); err != nil {
		logger.Crit("error unmarshal %s, request %s", out, myapi)
		return transfersStatus, err
	}
	return transfersStatus, nil
}

func TransferDetailByID(conn Connect, transferID string) (TransferStatus, error) {
	var transferStatus TransferStatus
	myapi := fmt.Sprintf("%s/v1/transfer/%s", TransferAPI, transferID)
	logger.Debug("sending %s transfer_id %s", myapi, transferID)
	out, err := conn.Do("GET", myapi, nil, nil)
	if err != nil {
		logger.Crit("error request %s, error %s, out: %s", myapi, err, out)
		return transferStatus, err
	}
	if err := json.Unmarshal(out, &transferStatus); err != nil {
		logger.Crit("error unmarshal %s, request %s", out, myapi)
		return transferStatus, err
	}
	return transferStatus, nil
}

func (tss TransfersStatus) Len() int {
	return len(tss.Transfers)
}

func (tss TransfersStatus) Swap(i, j int) {
	tss.Transfers[i], tss.Transfers[j] = tss.Transfers[j], tss.Transfers[i]
}

func (tss TransfersStatus) Less(i, j int) bool {
	curr := tss.Transfers[i].Instance()
	next := tss.Transfers[j].Instance()
	//logger.Debug("sort transfer status: %s %s\n", curr, next)
	if len(curr) < len(next) {
		return true
	} else if len(curr) == len(next) && curr < next {
		return true
	}
	return false
}

func (ts TransferStatus) Instance() string {
	if strings.Contains(ts.Name, "_") {
		instance := strings.SplitN(ts.Name, "_", 2)
		return instance[0]
	}
	return ts.Name
}

func TransfersByAllStatus(conn Connect, folderID string) (map[string]TransfersStatus, error) {
	var errmsg bytes.Buffer
	ids := make(map[string]int)
	transfersStatusAll := make(map[string]TransfersStatus)
	for _, status := range TransferAvailibleStatus {
		transfers, err := TransfersByStatus(conn, folderID, status)
		if err != nil {
			logger.Crit("error get transfers by status %s: %s", status, err)
			errmsg.WriteString(fmt.Sprint(err))
		}
		for _, t := range transfers.Transfers {
			if _, ok := ids[t.ID]; !ok {
				ids[t.ID] = 1
			}
		}
	}
	for id := range ids {
		transferStatus, err := TransferDetailByID(conn, id)
		if err != nil {
			logger.Crit("error get transfer details by id %s: %s", id, err)
			errmsg.WriteString(fmt.Sprint(err))
		}
		if val, ok := transfersStatusAll[transferStatus.Status]; !ok {
			transfersStatusAll[transferStatus.Status] = TransfersStatus{
				[]TransferStatus{transferStatus}}
		} else {
			val.Transfers = append(val.Transfers, transferStatus)
			transfersStatusAll[transferStatus.Status] = val
		}
	}
	return transfersStatusAll, fmt.Errorf("%s", errmsg.String())
}

func CreateMysqlTransportTM(conn Connect, folderID, projectName, instanceName, databaseName, slotID, sourceMySQL,
	destinationMySQL, sourceMySQLPort, destinationMySQLPort, user,
	typeTransport string, mysqlPassword Password, skipKeyChecks bool) (ResponseTarget, error) {
	switch {
	case len(sourceMySQL) > 0 && len(destinationMySQL) == 0:
		errmsg := fmt.Errorf("empty target mysql for replication: %s ----> UNKNOWN", sourceMySQL)
		logger.Crit("%s", errmsg)
		return ResponseTarget{}, errmsg
	case len(destinationMySQL) > 0 && len(sourceMySQL) == 0:
		errmsg := fmt.Errorf("empty source mysql for replication: UNKNOWN ---> %s", destinationMySQL)
		logger.Crit("%s", errmsg)
		return ResponseTarget{}, errmsg
	default:
	}

	srcEndpoint, err := CreateMysqlEndpointTM(conn,
		folderID,
		projectName,
		instanceName,
		databaseName,
		slotID,
		sourceMySQL, //hostname or clusterID
		user,
		sourceMySQLPort,
		mysqlPassword,
		"source",
		false,
	)

	if err != nil {
		errmsg := fmt.Errorf("error create source mysql endpoint: %s", err)
		logger.Crit("%s", errmsg)
		return srcEndpoint, errmsg
	}

	dstEndpoint, err := CreateMysqlEndpointTM(conn,
		folderID,
		projectName,
		instanceName,
		databaseName,
		slotID,
		destinationMySQL, //hostname or clusterID
		user,
		destinationMySQLPort,
		mysqlPassword,
		"target",
		skipKeyChecks,
	)
	if err != nil {
		errmsg := fmt.Errorf("error create target mysql endpoint: %s", err)
		logger.Crit("%s", errmsg)
		return dstEndpoint, errmsg
	}
	//Проверяем статус выполнения создания endpoints источников и потребителей
	switch {
	case srcEndpoint.TargetDone() && dstEndpoint.TargetDone():
		errmsg := fmt.Errorf("created srcEndpoint(id: %s) and dstEndpoint(id: %s)",
			srcEndpoint.TargetID(), dstEndpoint.TargetID())
		logger.Info("%s", errmsg)
	case !srcEndpoint.TargetDone() || !dstEndpoint.TargetDone():
		if !srcEndpoint.TargetDone() {
			errmsg := fmt.Errorf("failed src operationID %s: %+v", srcEndpoint.TargetID(), srcEndpoint)
			logger.Crit("%s", errmsg)
			return srcEndpoint, errmsg
		}
		if !dstEndpoint.TargetDone() {
			errmsg := fmt.Errorf("failed dst operationID %s: %+v", dstEndpoint.TargetID(), dstEndpoint)
			logger.Crit("%s", errmsg)
			return dstEndpoint, errmsg
		}
	default:
	}
	//Создаём транспорт между endpoints источников и потребителей
	logger.Info("%s mysql transport %s cluster/host %s", projectName, instanceName, sourceMySQL)
	transfer, err := CreateMysqlTransferTM(conn,
		folderID,
		projectName,
		instanceName,
		sourceMySQL,
		destinationMySQL,
		srcEndpoint.TargetID(),
		dstEndpoint.TargetID(),
		typeTransport,
	)

	if err != nil {
		errmsg := fmt.Errorf("error create transport %s -- > %s", srcEndpoint.TargetID(), dstEndpoint.TargetID())
		logger.Crit("%s", errmsg)
		return transfer, errmsg
	}

	switch {
	case transfer.TargetDone():
		logger.Info("created transportEndpoint(id: %s)", transfer.TargetID())
	case !transfer.TargetDone():
		msg := fmt.Errorf("transfer status id transport %s: %s", transfer.TargetID(), err)
		logger.Crit("%s", msg)
	default:
	}

	activate, err := ActivateTransferTM(conn, transfer.TargetID())
	if err != nil {
		errmsg := fmt.Errorf("error activate %s: %s", transfer.TargetID(), err)
		logger.Crit("%s", errmsg)
	}
	logger.Info("activted %+v", activate)
	return activate, nil
}

func PauseTransferTM(conn Connect, transferID string) (ResponseTarget, error) {
	var rt ResponseTarget
	tid := NewTransferID(transferID)
	data, err := json.Marshal(tid)
	if err != nil {
		return rt, fmt.Errorf("error marshal %v: %s", tid, err)
	}
	myapi := fmt.Sprintf("%s/v1/transfer/pause", TransferAPI)
	logger.Debug("sending %s data %s", myapi, data)
	out, err := conn.Do("PATCH", myapi, nil, data)
	if err != nil {
		logger.Crit("error request %s, tid %s, error %s, out: %s", myapi, tid, err, out)
		return rt, err
	}
	if err := json.Unmarshal(out, &rt); err != nil {
		logger.Crit("error unmarshal: %s, request: %s, tid: %s", out, myapi, transferID)
		return rt, err
	}
	return rt, nil
}

func UnpauseTransferTM(conn Connect, transferID string) (ResponseTarget, error) {
	var rt ResponseTarget
	tid := NewTransferID(transferID)
	data, err := json.Marshal(tid)
	if err != nil {
		return rt, fmt.Errorf("error marshal %v: %s", tid, err)
	}
	myapi := fmt.Sprintf("%s/v1/transfer/start", TransferAPI)
	logger.Debug("sending %s data %s", myapi, data)
	out, err := conn.Do("PATCH", myapi, nil, data)
	if err != nil {
		logger.Crit("error request %s, tid %s, error %s, out: %s", myapi, tid, err, out)
		return rt, err
	}
	if err := json.Unmarshal(out, &rt); err != nil {
		logger.Crit("error unmarshal: %s, request: %s, tid: %s", out, myapi, transferID)
		return rt, err
	}
	return rt, nil
}

func ExecuteTM(conn Connect, data []byte) (ResponseTarget, error) {
	var rt ResponseTarget
	myapi := fmt.Sprintf("%s/v1/transfers/execute", TransferAPI)
	logger.Debug("sending %s data %s", myapi, data)
	out, err := conn.Do("POST", myapi, nil, data)
	if err != nil {
		logger.Crit("error request %s, error %s, out: %s", myapi, err, out)
		return rt, err
	}
	if err := json.Unmarshal(out, &rt); err != nil {
		logger.Crit("error unmarshal %s, request %s", out, myapi)
		return rt, err
	}
	return rt, nil
}

func (otm OperationTM) Marshal() ([]byte, error) {
	return json.Marshal(otm)
}

func RunMySQLChecksum(conn Connect, transferID string) (ResponseTarget, error) {
	operation := OperationTM{
		Operation:  "CHECKSUM",
		TransferID: transferID,
	}
	logger.Debug("operation: %+v", operation)

	jsonData, err := operation.Marshal()
	if err != nil {
		return ResponseTarget{}, fmt.Errorf("[RunMySQLChecksum] error marshal %s: %s", jsonData, err)
	}
	return ExecuteTM(conn, jsonData)
}

func DeleteEndpointTM(conn Connect, endpointID string) (ResponseTarget, error) {
	var rt ResponseTarget
	eid := NewEndpointID(endpointID)
	data, err := json.Marshal(eid)
	if err != nil {
		return rt, fmt.Errorf("error marshal %v: %s", eid, err)
	}
	myapi := fmt.Sprintf("%s/v1/endpoint/%s", TransferAPI, endpointID)
	logger.Debug("delete endpoint %s", myapi)
	out, err := conn.Do("DELETE", myapi, nil, data)
	if err != nil {
		logger.Crit("error request %s, tid %s, error %s, out: %s", myapi, endpointID, err, out)
		return rt, err
	}
	if err := json.Unmarshal(out, &rt); err != nil {
		logger.Crit("error unmarshal: %s, request: %s, tid: %s", out, myapi, endpointID)
		return rt, err
	}
	return rt, nil
}

func DeleteTransferTM(conn Connect, transferID string) (ResponseTarget, error) {
	var rt ResponseTarget
	tid := NewTransferID(transferID)
	values, err := MergeBlocks(tid)
	if err != nil {
		return rt, fmt.Errorf("error merge %v: %s", tid, err)
	}
	myapi := fmt.Sprintf("%s/v1/transfer", TransferAPI)
	logger.Debug("delete %s values %s", myapi, values)
	out, err := conn.Do("DELETE", myapi, values, nil)
	if err != nil {
		logger.Crit("error request %s, tid %s, error %s, out: %s", myapi, tid, err, out)
		return rt, err
	}
	if err := json.Unmarshal(out, &rt); err != nil {
		logger.Crit("error unmarshal: %s, request: %s, tid: %s", out, myapi, transferID)
		return rt, err
	}
	return rt, nil
}

func NewEndpointID(endpointID string) EndpointID {
	return EndpointID{
		EndpointID: endpointID,
	}
}

func DeleteMysqlTransportTM(conn Connect, transferID string) error {
	transfer, err := TransferDetailByID(conn, transferID)
	if err != nil {
		logger.Crit("[DeleteMysqlTransportTM] detail %v: %s", transfer, err)
		return err
	}
	sourceEndpointID := transfer.SourceTM.ID
	targetEndpointID := transfer.TargetTM.ID
	logger.Info("start delete: %s(transer), %s(source endpoint), %s(destination endpoint)",
		transfer.ID, sourceEndpointID, targetEndpointID)
	rt, err := DeleteTransferTM(conn, transfer.ID)
	if err != nil {
		logger.Crit("[DeleteMysqlTransportTM] transfer %v: %s", rt, err)
		return err
	}
	var errmsg []error
	rt, err = DeleteEndpointTM(conn, sourceEndpointID)
	if err != nil {
		logger.Crit("[DeleteMysqlTransportTM] endpoint %v: %s", rt, err)
		errmsg = append(errmsg, err)
	}
	rt, err = DeleteEndpointTM(conn, sourceEndpointID)
	if err != nil {
		logger.Crit("[DeleteMysqlTransportTM] endpoint %v: %s", rt, err)
		errmsg = append(errmsg, err)
	}
	if len(errmsg) > 0 {
		return fmt.Errorf("%s", errmsg)
	}
	return nil
}

func PrintEndpoints(conn Connection, folderID, projectName, instanceName string) error {
	endpoints, err := ListEndpointTM(conn, folderID)
	if err != nil {
		logger.Crit("[PrintEndpoints] list %v: %s", endpoints, err)
		return err
	}
	endpoints = endpoints.FilterByDescription(projectName)
	if len(instanceName) > 0 {
		endpoints = endpoints.FilterByName(instanceName)
	}
	for _, e := range endpoints.Endpoints {
		fmt.Printf("Name: %s, ID: %s, Direction: %s, Database: %s\n", e.Name, e.ID, e.Direction, e.Database())
	}
	return nil
}
