package ytlib

import (
	"context"
	"fmt"
	"io/ioutil"
	"os"
	"strings"
	"time"

	logger "a.yandex-team.ru/direct/infra/go-libs/pkg/logformat"
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yt"
	"a.yandex-team.ru/yt/go/yt/ythttp"
	"a.yandex-team.ru/yt/go/ytlock"
)

var (
	lockPath        = "//home/direct-np/testing/apps/binlogage"
	lockClusterName = "locke"
	ytConfig        yt.Client
	lockToken       string
)

type Option func()

func WithLockClusterName(name string) Option {
	return func() {
		lockClusterName = name
	}
}

func WithLockPath(path string) Option {
	return func() {
		lockPath = path
	}
}

func WithLockToken(token interface{}) Option {
	return func() {
		lockToken = valueByFile(token)
	}
}

func valueByFile(name interface{}) string {
	logger.Debug("valueByFile %s %t", name, name)
	switch value := name.(type) {
	case string:
		return value
	case *string:
		return *value
	case os.File:
		_, _ = value.Seek(0, os.SEEK_SET)
		data, err := ioutil.ReadAll(&value)
		if err != nil {
			logger.Warn("error read file %s, %s", value.Name(), err)
		} else {
			return strings.TrimSuffix(string(data), "\n")
		}
	case *os.File:
		_, _ = (*value).Seek(0, os.SEEK_SET)
		data, err := ioutil.ReadAll(value)
		if err != nil {
			logger.Warn("error read file %s, %s", value.Name(), err)
		} else {
			return strings.TrimSuffix(string(data), "\n")
		}
	}
	return ""
}

func NewYtClient(options ...Option) (yt.Client, error) {
	for _, option := range options {
		option()
	}
	ytcnf := yt.Config{
		Proxy: lockClusterName,
		Token: lockToken,
	}
	logger.Debug("ytconfig %+v", ytcnf)
	return ythttp.NewClient(&ytcnf)
}

func LockTransaction(yc yt.Client) (*ytlock.Lock, yt.Tx, error) {
	ytpath := ypath.Path(lockPath)
	initialOpt := ytlock.Options{
		CreateIfMissing: true,
		LockMode:        yt.LockExclusive,
	}
	lock := ytlock.NewLockOptions(yc, ytpath, initialOpt)
	cntx := context.Background()
	tx, err := lock.AcquireTx(cntx)
	return lock, tx, err
}

func UnlockTransaction(lock *ytlock.Lock) error {
	cntx, cancel := context.WithTimeout(context.Background(), 60*time.Second)
	defer cancel()
	return lock.Release(cntx)
}

func MustLockTransaction(client yt.Client) (lock *ytlock.Lock, err error) {
	for {
		lock, tx, err := LockTransaction(client)
		if err != nil && tx == nil {
			logger.Crit("error get yt lock %s, %s", lockPath, err)
			return nil, err
		} else if !lock.IsLocked() {
			cntx := context.Background()
			_ = lock.Release(cntx)
			logger.Warn("yt lock %s already used", lockPath)
			time.Sleep(10 * time.Second)
			continue
		} else {
			return lock, nil
		}
	}
}

func MustMultiLockTransaction(clients []yt.Client) (lock []*ytlock.Lock, err error) {
L:
	for {
		var locks []*ytlock.Lock
		var errmsg []string
		for _, client := range clients {
			logger.Debug("client config: %s", client)
			lock, tx, err := LockTransaction(client)
			if err != nil && tx == nil {
				logger.Crit("error get yt lock %s, %s", lockPath, err)
				errmsg = append(errmsg, err.Error())
				continue
			} else if !lock.IsLocked() {
				cntx := context.Background()
				_ = lock.Release(cntx)
				logger.Warn("yt lock %s already used", lockPath)
				for _, l := range locks {
					if err := UnlockTransaction(l); err != nil {
						logger.Warn("error unlock transaction, %s", err)
					}
				}
				//locks = []*ytlock.Lock{}
				time.Sleep(10 * time.Second)
				continue L
			} else {
				locks = append(locks, lock)
			}
		}
		if len(locks) > 0 {
			return locks, nil
		}
		return nil, fmt.Errorf("%s", strings.Join(errmsg, ","))
	}
}
