package yttransfer

import (
	"bytes"
	"context"
	"fmt"
	"io/ioutil"
	"math"
	"strings"
	"sync"

	"a.yandex-team.ru/yt/go/mapreduce"
	"a.yandex-team.ru/yt/go/mapreduce/spec"
	"a.yandex-team.ru/yt/go/schema"
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yt"
	"a.yandex-team.ru/yt/go/yt/ythttp"
)

type YtConnect struct {
	Client  yt.Client
	Cntx    context.Context `json:"-"`
	Account string
}

func NewYtConnect(cluster, account, tokenFile string) (YtConnect, error) {
	ytch, err := connect(cluster, tokenFile)
	return YtConnect{ytch, context.Background(), account}, err
}

func connect(cluster, tokenFile string) (ytch yt.Client, err error) {
	var ytconf *yt.Config
	if len(tokenFile) == 0 {
		ytconf = &yt.Config{}
	} else {
		token, err := ioutil.ReadFile(tokenFile)
		if err != nil {
			return nil, fmt.Errorf("error read token %s: %s", tokenFile, err)
		}
		ytconf = &yt.Config{
			Proxy: "",
			Token: string(bytes.TrimRight(token, "\n")),
		}
	}
	if len(cluster) > 0 {
		ytconf.Proxy = cluster
	}
	if ytch, err = ythttp.NewClient(ytconf); err != nil {
		return nil, fmt.Errorf("error connect to YT %v: %s", ytconf, err)
	}
	//ytch.Stop()
	return ytch, err
}

func (y YtConnect) ListYtNode(startDir string) (response []string, err error) {
	err = y.Client.ListNode(y.Cntx, ypath.Path(startDir), &response, nil)
	return
}

func (y YtConnect) RecurseListYtNode(startDir string) (result []string, err error) {
	var response interface{}
	err = y.Client.GetNode(y.Cntx, ypath.Path(startDir), &response, nil)
	_ = filepath(response, "", &result)
	return
}

func convertYtPath(path interface{}) (result ypath.Path, err error) {
	switch v := path.(type) {
	case string:
		result = ypath.Path(v)
	case *ypath.Rich:
		result = v.Path
	case ypath.Path:
		result = v
	case *ypath.Path:
		result = *v
	default:
		err = fmt.Errorf("not found type for %s type %T", v, v)
	}
	return
}

func (y YtConnect) RemoveYtNode(path interface{}, force, recurse bool) (err error) {
	p, err := convertYtPath(path)
	if err != nil {
		return fmt.Errorf("[RemoveYtNode/convertYtPath] %s: %s", path, err)
	}
	optsRemove := yt.RemoveNodeOptions{
		Force:     force,
		Recursive: recurse,
	}
	return y.Client.RemoveNode(y.Cntx, p, &optsRemove)
}

func (y YtConnect) MoveYtNode(srcPath, dstPath interface{}, force, recurse bool) (id yt.NodeID, err error) {
	src, err := convertYtPath(srcPath)
	if err != nil {
		return id, fmt.Errorf("[RemoveYtNode/convertYtPath] %s: %s", srcPath, err)
	}
	dst, err := convertYtPath(dstPath)
	if err != nil {
		return id, fmt.Errorf("[RemoveYtNode/convertYtPath] %s: %s", dstPath, err)
	}

	optsMove := yt.MoveNodeOptions{
		Force:     force,
		Recursive: recurse,
	}
	return y.Client.MoveNode(y.Cntx, src, dst, &optsMove)
}

func filepath(value interface{}, path string, result *[]string) (err error) {
	v, ok := value.(map[string]interface{})
	if !ok {
		return fmt.Errorf("error convert map: %v", value)
	}
	for name, value := range v {
		newpath := fmt.Sprintf("%s/%s", path, name)
		switch value.(type) {
		case map[string]interface{}:
			_ = filepath(value, newpath, result)
		default:
			*result = append(*result, newpath)
		}
	}
	return
}

func (y YtConnect) GetAttribute(startDir, nameAttr string, attributes interface{}) error {
	path := ypath.Path(fmt.Sprintf("%s/@%s", startDir, nameAttr))
	return y.Client.GetNode(y.Cntx, path, attributes, nil)
}

func (y YtConnect) IsDynamic(startDir string) (ok bool, err error) {
	err = y.GetAttribute(startDir, "dynamic", &ok)
	return ok, err
}

func (y YtConnect) IsTableOrFile(startDir string) (bool, error) {
	var nodeType string
	err := y.GetAttribute(startDir, "type", &nodeType)
	if strings.Contains(nodeType, "table") || strings.Contains(nodeType, "file") {
		return true, err
	}
	return false, err
}

func (y YtConnect) GetLinkAttribute(startDir, nameAttr string, attributes interface{}) error {
	path := ypath.Path(fmt.Sprintf("%s&/@%s", startDir, nameAttr))
	return y.Client.GetNode(y.Cntx, path, attributes, nil)
}

func (y YtConnect) SetAttribute(startDir, nameAttr string, attributes interface{}) error {
	path := ypath.Path(fmt.Sprintf("%s/@%s", startDir, nameAttr))
	return y.Client.SetNode(y.Cntx, path, attributes, nil)
}

func (y YtConnect) NodeHasType(path, typeName string) (ok bool, err error) {
	var attr string
	err = y.GetAttribute(path, "type", &attr)
	ok = strings.Contains(typeName, attr)
	return
}

func (y YtConnect) NodeYtExists(path interface{}) (ok bool, err error) {
	p, err := convertYtPath(path)
	if err != nil {
		return false, err
	}
	return y.Client.NodeExists(y.Cntx, p, nil)
}

//удаляет старый линк на дирректорию и добавляет новый
func (y YtConnect) UpdateYtLink(link, target string) (err error) {
	if ok, _ := y.NodeYtExists(link); ok {
		err = y.RemoveYtNode(link, true, false)
		if err != nil {
			return fmt.Errorf("[UpdateYtLink/RemoveYtNode] %s", err)
		}
	}
	ylink, err := convertYtPath(link)
	if err != nil {
		return fmt.Errorf("[UpdateYtLink/convertYtPath] %s: %s", link, err)
	}
	ytarget, err := convertYtPath(target)
	if err != nil {
		return fmt.Errorf("[UpdateYtLink/convertYtPath] %s: %s", target, err)
	}
	_, err = y.Client.LinkNode(y.Cntx, ytarget, ylink, nil)
	return
}

type ResourceDisk struct {
	DiskSpace  int `yson:"disk_space"`
	NodeCount  int `yson:"node_count"`
	ChunkCount int `yson:"chunk_count"`
}

type ReplicatorVersion string

func (y YtConnect) ResourceUsage(startDir string) (resource ResourceDisk, err error) {
	err = y.GetAttribute(startDir, "recursive_resource_usage", &resource)
	return
}

func (y YtConnect) DatabaseVersion(file string) (version ReplicatorVersion, err error) {
	err = y.GetAttribute(file, "sync_version", &version)
	return
}

//
func (y YtConnect) ResourceLimit() (resource ResourceDisk, err error) {
	path := fmt.Sprintf("//sys/accounts/%s", y.Account)
	err = y.GetAttribute(path, "resource_limits", &resource)
	return
}

func (y YtConnect) CreateTable(dstpath string, data interface{}) (err error) {
	client := y.Client
	dstYtPath := ypath.NewRich(dstpath)
	optsCreate := yt.CreateNodeOptions{
		IgnoreExisting: true,
		Attributes: map[string]interface{}{
			"schema": schema.MustInfer(data),
		},
	}
	fmt.Printf("%+v\n", optsCreate)
	_, err = client.CreateNode(y.Cntx, dstYtPath.Path, yt.NodeType("table"), &optsCreate)
	return
}

func (y YtConnect) InsertTable(dstpath string, data []interface{}) (err error) {
	client := y.Client
	dstYtPath := ypath.NewRich(dstpath)

	optsInsert := yt.InsertRowsOptions{}
	return client.InsertRows(y.Cntx, dstYtPath.Path, data, &optsInsert)
}

func (y YtConnect) WriteTable(dstpath string, data []interface{}) (err error) {
	client := y.Client
	dstYtPath := ypath.NewRich(dstpath)

	optsWrite := yt.WriteTableOptions{}
	w, err := client.WriteTable(y.Cntx, dstYtPath.Path, &optsWrite)
	if err != nil {
		return err
	}
	for _, row := range data {
		_ = w.Write(row)
	}
	return w.Commit()
}

func (y YtConnect) ConvertStaticTable(srcpath, dstpath string, lock *sync.WaitGroup) (result ConvertTableStatus) {
	defer lock.Done()
	result = NewConvertStatus(srcpath, dstpath)
	client := y.Client
	result.SetStart()
	columns, err := y.GetKeyColumns(srcpath)
	if err != nil {
		result.SetFailed(err)
		return
	}

	schema, err := y.GetSchema(srcpath)
	if err != nil {
		return
	}

	srcYtPath := ypath.NewRich(srcpath)
	dstYtPath := ypath.NewRich(dstpath)

	fmt.Printf("start convert %s to %s\n", srcpath, dstpath)

	ok, err := client.NodeExists(y.Cntx, dstYtPath.Path, nil)
	if ok && err == nil {
		err = y.RemoveYtNode(dstYtPath.Path, true, false)
		//err = client.RemoveNode(y.Cntx, dstYtPath.Path, &optsRemove)
		if err != nil {
			result.SetFailed(fmt.Errorf("error remove node %s: %s", dstYtPath.Path, err))
			return
		}
		fmt.Printf("remove directory %s done", dstYtPath.Path)
	}

	//create new table
	optsCreate := yt.CreateNodeOptions{
		IgnoreExisting: true,
		Attributes: map[string]interface{}{
			"optimize_for":   "scan",
			"schema":         schema,
			"primary_medium": "ssd_blobs",
		},
	}

	_, err = client.CreateNode(y.Cntx, dstYtPath.Path, yt.NodeType("table"), &optsCreate)
	if err != nil {
		result.SetFailed(fmt.Errorf("error create node %s: %s", dstYtPath.Path, err))
		return
	}

	myspec := spec.Merge()
	myspec.JobIO = &spec.JobIO{
		TableWriter: map[string]int{
			"block_size":         256 * int(math.Pow(2, 10)),
			"desired_chunk_size": 100 * int(math.Pow(2, 20)),
		},
	}
	myspec.MergeBy = columns
	myspec.ForceTransform = true
	myspec.MergeMode = "sorted"
	_ = myspec.AddInput(srcYtPath.Path)
	_ = myspec.SetOutput(dstYtPath.Path)

	//fmt.Printf("%+v\n", myspec) debug

	mr := mapreduce.New(client, mapreduce.WithContext(y.Cntx))
	op, err := mr.Merge(myspec)
	if err != nil {
		result.SetFailed(fmt.Errorf("error merge %+v: %s", myspec, err))
		return
	}

	id := op.ID()
	//fmt.Printf("start task %s", id) debug
	if err := op.Wait(); err != nil {
		result.SetFailed(fmt.Errorf("error operation task %s: %s", id, err))
		return
	}
	result.OperationID = id.String()

	if ok, err := client.NodeExists(y.Cntx, dstYtPath.Path, nil); !ok || err != nil {
		result.SetFailed(fmt.Errorf("table dont exists %s error: %s", dstYtPath.Path, err))
		return
	}

	dynamic := true
	optsAlter := yt.AlterTableOptions{Dynamic: &dynamic}
	if err = client.AlterTable(y.Cntx, dstYtPath.Path, &optsAlter); err != nil {
		result.SetFailed(fmt.Errorf("error alter table %s: %s", dstYtPath.Path, err))
	}

	pivotPath := ypath.NewRich(dstYtPath.Path.String() + "_pivot")
	fmt.Printf("INFO. start reshard %s with pivot %s\n", dstYtPath.Path.String(), pivotPath.Path)

	if ok, err := client.NodeExists(y.Cntx, pivotPath.Path, nil); ok {
		r, err := client.ReadTable(y.Cntx, pivotPath.Path, nil)
		if err == nil {
			defer func() { _ = r.Close() }()
			var pivot YtPivotRow
			r.Next()
			if err = r.Scan(&pivot); err != nil {
				fmt.Printf("WARN. Error scan 'pivot' keys, error %s", err)
			}
			fmt.Printf("INFO. pivot keys: %+v\n", pivot)
			if len(pivot.Value) > 1 {
				optsReshard := yt.ReshardTableOptions{
					PivotKeys: pivot.Value,
				}

				err = client.ReshardTable(y.Cntx, dstYtPath.Path, &optsReshard)
				if err != nil {
					fmt.Printf("ERROR. failed reshard %v with pivot %+v, error %s", dstYtPath, pivot, err)
				}
			}
		} else {
			fmt.Printf("ERROR. failed read pivot table %s, error %s", pivotPath.Path, err)
		}
	} else {
		fmt.Printf("ERROR. failed check exists %s, error %s", pivotPath.Path, err)
	}

	//
	attrMount := yt.MountTableOptions{}
	if err := client.MountTable(y.Cntx, dstYtPath.Path, &attrMount); err != nil {
		result.SetFailed(fmt.Errorf("error mount %s: %s", dstYtPath.Path, err))
	}
	result.SetDone()
	return
}

func (y YtConnect) ConvertDynamicTable(srcpath, dstpath string, lock *sync.WaitGroup) (result ConvertTableStatus) {
	defer lock.Done()
	result = NewConvertStatus(srcpath, dstpath)
	client := y.Client
	result.SetStart()
	columns, err := y.GetKeyColumns(srcpath)
	if err != nil {
		result.SetFailed(err)
		return
	}

	schema, err := y.GetSchema(srcpath)
	if err != nil {
		return
	}

	srcYtPath := ypath.NewRich(srcpath)
	dstYtPath := ypath.NewRich(dstpath)

	fmt.Printf("start convert %s to %s\n", srcpath, dstpath)

	ok, err := client.NodeExists(y.Cntx, dstYtPath.Path, nil)
	if ok && err == nil {
		err = y.RemoveYtNode(dstYtPath.Path, true, false)
		if err != nil {
			result.SetFailed(fmt.Errorf("error remove node %s: %s", dstYtPath.Path, err))
			return
		}
		fmt.Printf("INFO. success remove old directory %s\n", dstYtPath.Path)
	}

	//create new table
	optsCreate := yt.CreateNodeOptions{
		IgnoreExisting: true,
		Attributes: map[string]interface{}{
			"optimize_for":   "scan",
			"schema":         schema,
			"primary_medium": "ssd_blobs",
		},
	}

	_, err = client.CreateNode(y.Cntx, dstYtPath.Path, yt.NodeType("table"), &optsCreate)
	if err != nil {
		result.SetFailed(fmt.Errorf("error create node %s: %s", dstYtPath.Path, err))
		return
	}

	err = client.FreezeTable(y.Cntx, srcYtPath.Path, nil)
	if err != nil {
		result.SetFailed(fmt.Errorf("error freeze %s: %s", srcYtPath.Path, err))
		return
	}
	defer func() {
		_ = client.UnfreezeTable(y.Cntx, srcYtPath.Path, nil)
	}()
	myspec := spec.Merge()
	myspec.JobIO = &spec.JobIO{
		TableWriter: map[string]int{
			"block_size":         10240 * int(math.Pow(2, 10)),
			"desired_chunk_size": 300 * int(math.Pow(2, 20)),
		},
	}
	myspec.MergeBy = columns
	myspec.ForceTransform = true
	myspec.MergeMode = "ordered"
	_ = myspec.AddInput(srcYtPath.Path)
	_ = myspec.SetOutput(dstYtPath.Path)

	mr := mapreduce.New(client, mapreduce.WithContext(y.Cntx))
	op, err := mr.Merge(myspec)
	if err != nil {
		result.SetFailed(fmt.Errorf("error merge %+v: %s", myspec, err))
		return
	}

	id := op.ID()
	if err := op.Wait(); err != nil {
		result.SetFailed(fmt.Errorf("error operation task %s: %s", id, err))
		return
	}

	result.OperationID = id.String()

	if ok, err := client.NodeExists(y.Cntx, dstYtPath.Path, nil); !ok || err != nil {
		result.SetFailed(fmt.Errorf("table dont exists %s error: %s", dstYtPath.Path, err))
		return
	}

	result.SetDone()
	return
}

type ConvertTableStatus struct {
	SrcPath     string
	DstPath     string
	Status      string
	OperationID string
	Error       error
}

type ConvertTablesStatus []ConvertTableStatus

func NewConvertStatus(src, dst string) ConvertTableStatus {
	return ConvertTableStatus{
		SrcPath:     src,
		DstPath:     dst,
		Status:      "NEW",
		OperationID: "",
		Error:       nil,
	}
}

func (cts *ConvertTableStatus) SetFailed(err error) {
	cts.Error = err
	cts.Status = "FAILED"
}

func (cts *ConvertTableStatus) SetDone() {
	cts.Status = "DONE"
}

func (cts *ConvertTableStatus) SetStart() {
	cts.Status = "PROCESS"
}

func (ctss ConvertTablesStatus) ConvertsFailed() (result ConvertTablesStatus) {
	for _, status := range ctss {
		if strings.Contains(status.Status, "FAILED") {
			result = append(result, status)
		}
	}
	return
}

func (y YtConnect) RemoveConvertedTables(convertsStatus ConvertTablesStatus) (err error) {
	var errmsg bytes.Buffer
	for _, s := range convertsStatus {
		if strings.Contains(s.Status, "DONE") {
			removed := []string{s.SrcPath, strings.ReplaceAll(s.SrcPath, "_static", "_pivot")}
			for _, path := range removed {
				ok, err := y.NodeYtExists(path)
				if err != nil {
					errmsg.WriteString(err.Error())
					continue
				}
				if ok {
					err := y.RemoveYtNode(path, false, false)
					if err != nil {
						errmsg.WriteString(fmt.Sprintf("%s", err))
					}
					fmt.Printf("removed converted %s\n", path)
				}
			}
		}
	}
	if errmsg.Len() > 0 {
		err = fmt.Errorf(errmsg.String())
	}
	return
}

func (y YtConnect) UpdateSyncVersion(convertsStatus ConvertTablesStatus) (err error) {
	for _, s := range convertsStatus {
		var version ReplicatorVersion
		if err = y.GetAttribute(s.SrcPath, "sync_version", &version); err != nil {
			return
		}
		if err = y.SetAttribute(s.DstPath, "sync_version", &version); err != nil {
			return
		}
	}
	return
}

func (y YtConnect) StartConvertStaticTable(tables []string) (result ConvertTablesStatus) {
	var lock sync.WaitGroup
	for _, table := range tables {
		lock.Add(1)
		srcpath := table
		dstpath := strings.ReplaceAll(srcpath, "_static", "")
		go func() {
			convertStatus := y.ConvertStaticTable(srcpath, dstpath, &lock)
			result = append(result, convertStatus)
		}()
	}
	lock.Wait()
	return
}

func (y YtConnect) StartConvertDynamicTable(tables []string) (result ConvertTablesStatus) {
	var lock sync.WaitGroup
	for _, table := range tables {
		lock.Add(1)
		srcpath := table
		dstpath := srcpath + "_static"
		go func() {
			convertStatus := y.ConvertDynamicTable(srcpath, dstpath, &lock)
			result = append(result, convertStatus)
		}()
	}
	lock.Wait()
	return
}

func (y YtConnect) GetKeyColumns(startDir string) (columns []string, err error) {
	err = y.GetAttribute(startDir, "key_columns", &columns)
	//fmt.Println(columns) debug
	return
}

func (y YtConnect) GetSchema(startDir string) (schema interface{}, err error) {
	err = y.GetAttribute(startDir, "schema", &schema)
	//fmt.Println(schema) debug
	return
}

type YtPivotRow struct {
	Value []interface{} `yson:"value"`
}

func (y YtConnect) GetPivotKeys(ytPath string) (keys interface{}, err error) {
	var pivotKeys YtPivotRow
	err = y.GetAttribute(ytPath, "pivot_keys", &pivotKeys.Value)
	return pivotKeys, err
}
