package zklib

import (
	"bytes"
	"errors"
	"fmt"
	//"os"
	"path/filepath"
	"strings"
	"sync"
	"time"

	logger "a.yandex-team.ru/direct/infra/go-libs/pkg/logformat"
	"github.com/go-zookeeper/zk"
)

type ZkAddress struct {
	Servers  []string
	User     string
	Password *string
}

func NewZkAddress(user, pass string, servers ...string) ZkAddress {
	return ZkAddress{
		User:     user,
		Password: &pass,
		Servers:  servers,
	}
}

func NewZkAddressWithToken(token string, servers ...string) ZkAddress {
	userPass := strings.SplitN(token, ":", 2)
	user := userPass[0]
	var pass string
	if len(userPass) > 1 {
		pass = userPass[1]
	}
	return ZkAddress{
		User:     user,
		Password: &pass,
		Servers:  servers,
	}
}

type ZkConnect struct {
	*zk.Conn
	Event <-chan zk.Event
	ZkAddress
}

type Options func()

type Logger struct{}

func (l Logger) Printf(msg string, args ...interface{}) {
	logger.Debug(msg, args)
}

func NewZkConnect(zxzk ZkAddress, opts ...Options) (ZkConnect, error) {
	var conn *zk.Conn
	var session <-chan zk.Event
	var err error
	//если logger передается, то выставляем его
	for _, option := range opts {
		option()
	}
	var newlogger Logger
	for i := 0; i < 3; i++ {
		conn, session, err = zk.Connect(zxzk.Servers, 10*time.Second, zk.WithLogger(newlogger))
		if err != nil {
			logger.Crit("error connect to zookeeper %s, error %s, try %d", zxzk.Servers, err, i)
			continue
		}
		if len(zxzk.User) > 0 && len(*zxzk.Password) > 0 {
			userPass := fmt.Sprintf("%s:%s", zxzk.User, *zxzk.Password)
			if err = conn.AddAuth("digest", []byte(userPass)); err != nil {
				logger.Crit("error auth to zookeeper %s for user %s, error %s", zxzk.Servers, zxzk.User, err)
				continue
			}
		}

		logger.Info("success connect to zookeeper %s", zxzk.Servers)
		return ZkConnect{conn, session, zxzk}, nil
	}
	return ZkConnect{}, err
}

func (zc *ZkConnect) CheckConnect() bool {
	if zc.State() == zk.State(-1) {
		zkaddr := zc.ZkAddress
		logger.Info("zookeeper status are invalid, reconnect")
		if newconn, err := NewZkConnect(zkaddr); err != nil {
			logger.Crit("error reconnect %v, error %s", zkaddr, err)
			return false
		} else {
			zc = &newconn
		}
	}
	return true
}

type ZkNode struct {
	Path            string
	Stat            *zk.Stat
	Data            *[]byte
	Deleted         int
	locker          *sync.Mutex
	LastReadVersion *int32
	LastReadTime    *time.Time
	LastWriteTime   *time.Time
}

func NewZkNode(path string) ZkNode {
	stat := zk.Stat{}
	data := []byte{}
	var rv int32
	var rt, wt time.Time
	return ZkNode{
		Path:            path,
		Stat:            &stat,
		Data:            &data,
		Deleted:         0,
		locker:          &sync.Mutex{},
		LastReadVersion: &rv,
		LastReadTime:    &rt, //время последнего удачного чтения
		LastWriteTime:   &wt, //время последней удачной записи
	}
}

func NodePath(node interface{}) string {
	switch v := node.(type) {
	case ZkNode:
		return v.Path
	case string:
		return v
	}
	logger.Warn("NodePath not found type %+v(%T)", node, node)
	return ""
}

func (n ZkNode) GetNodeVersion() int32 {
	if n.Stat == nil {
		return -1
	}
	return n.Stat.Version
}

func (n ZkNode) GetLastReadTime() time.Time {
	if n.LastReadTime == nil {
		return time.Time{}
	}
	return *n.LastReadTime
}

func (n ZkNode) GetLastWriteTime() time.Time {
	if n.LastWriteTime == nil {
		return time.Time{}
	}
	return *n.LastWriteTime
}

func (n ZkNode) GetLastReadVersion() int32 {
	if n.LastReadVersion == nil {
		return -1
	}
	return *n.LastReadVersion
}

func (n ZkNode) GetPath() string {
	return n.Path
}

func (n *ZkNode) Locker() *sync.Mutex {
	return n.locker
}

func (n *ZkNode) UpdateNode(data []byte, version int32) error {
	n.Locker().Lock()
	defer n.Locker().Unlock()
	if lastvers := n.GetLastReadVersion(); version != lastvers {
		return fmt.Errorf("cannot update zknode, current version %d != last read version %d", version, lastvers)
	}
	*n.Data = data
	(*n.Stat).Version++
	return nil
}

type WriteLagError string

func (we WriteLagError) Error() string {
	return string(we)
}

//Проверяем, что процесс с чтением из zk жив и может записать данные в zookeeper
func (n *ZkNode) CheckZookeeperNode(maxReadDuration, maxWriteDuration time.Duration) (bool, error) {
	now := time.Now()
	uptime := n.GetLastReadTime()
	if readDuration := now.Sub(uptime); readDuration > maxReadDuration {
		err := fmt.Errorf("last read zknode %s more %s", n.Path, readDuration)
		logger.Crit("%s. Last zookeeper read time %s more current time %s", err, uptime, now)
		return false, err
	}

	if writeDuration := now.Sub(n.GetLastWriteTime()); writeDuration > maxWriteDuration && //если zookeeper давно не обновлялся
		n.GetNodeVersion() > n.GetLastReadVersion() { //и текущая версия ноды больше последней прочитанной
		errmsg := fmt.Sprintf("last write zknode %s more %s", n.Path, writeDuration)
		logger.Crit("%s. Last zookeeper write time %s, current time %s", errmsg, writeDuration, now)
		return false, WriteLagError(errmsg)
	}
	return true, nil
}

//ZkNodes
type ZkNodes []*ZkNode

//можно передать путь до ноды в виде string, либо в виде ZkNode
func (ns ZkNodes) NodeVersion(zkpath interface{}) int32 {
	for _, node := range ns {
		if node.Path == NodePath(zkpath) {
			return node.Stat.Version
		}
	}
	return 0
}

func (ns *ZkNodes) DeleteNode(zkpath interface{}) {
	var indexDeleted []int
	for num, node := range *ns {
		if node.Path == NodePath(zkpath) {
			indexDeleted = append(indexDeleted, num)
		}
	}
	if len(indexDeleted) > 0 {
		var cnt, num int
		for cnt, num = range indexDeleted {
			(*ns)[num] = (*ns)[len(*ns)-cnt-1]
		}
		*ns = (*ns)[:len(*ns)-cnt-1]
	}
}

func (ns *ZkNodes) AddNode(newnode ZkNode) {
	*ns = append(*ns, &newnode)
}

func (ns *ZkNodes) UpdateNode(newnode ZkNode) bool {
	if !ns.ExistNode(newnode) {
		ns.AddNode(newnode)
		return true
	}
	for _, node := range *ns {
		if NodePath(node.Path) == NodePath(newnode.Path) {
			node.Data = newnode.Data
			node.Stat.Version++
		}
	}
	return true
}

func (ns ZkNodes) ExistNode(zkpath interface{}) bool {
	for _, node := range ns {
		if node.Path == NodePath(zkpath) {
			return true
		}
	}
	return false
}

//сообщает найдена ли указаная нода по пути path
func (zc *ZkConnect) ExistNode(path interface{}) (bool, error) {
	if ok, _, err := zc.Exists(NodePath(path)); err != nil {
		return false, err
	} else if !ok {
		return false, nil
	}
	return true, nil
}

//возвращает список путей нод, найденных в дирректории path
func (zc *ZkConnect) ListNodePath(path interface{}) ([]string, error) {
	var zklist []string
	if ok, err := zc.ExistNode(NodePath(path)); err == nil && ok {
		zklist, _, err := zc.Children(NodePath(path))
		logger.Debug("list zkpath %s childrens: %s, error: %s", path, zklist, err)
		return zklist, err
	} else {
		return zklist, err
	}
}

//вычищает дирректорию path от вложенных нод
func (zc *ZkConnect) CleanNodePath(path interface{}) (bool, error) {
	msg := bytes.NewBuffer([]byte(""))
	var removedNodes []string
	cleanNodePath := NodePath(path)
	if len(cleanNodePath) == 0 {
		return false, fmt.Errorf("empty directory for clean. Skip")
	}
	baseZkPath := filepath.Dir(cleanNodePath)
	childsPath, err := zc.ListNodePath(baseZkPath)
	if err != nil {
		return false, err
	}
	for _, p := range childsPath {
		removedNodes = append(removedNodes, filepath.Join(baseZkPath, p))
	}
	removedNodes = append(removedNodes, baseZkPath)
	logger.Info("start clean zookeeper nodes: %s", removedNodes)
	for _, rp := range removedNodes {
		_, stat, _ := zc.Get(rp)
		logger.Info("remove node %s", path)
		if err := zc.Delete(rp, stat.Version); err != nil {
			msg.WriteString(fmt.Sprintf("%s, ", err))
		}
	}
	return msg.Len() == 0, errors.New(msg.String())
}

func (zc *ZkConnect) LoadNode(path interface{}) (ZkNode, error) {
	var (
		ok   bool
		err  error
		data []byte
		stat *zk.Stat
	)
	loadNodePath := NodePath(path)
	zknode := NewZkNode(loadNodePath)
	logger.Debug("start load zknode %s", path)
	if ok, err = zc.ExistNode(loadNodePath); err == nil && ok {
		if data, stat, err = zc.Get(loadNodePath); err != nil {
			logger.Debug("error load zknode %s", loadNodePath)
			return zknode, fmt.Errorf("error load zknode %s: %s", loadNodePath, err)
		}
		zknode.Path = loadNodePath
		zknode.Stat = stat
		*zknode.Data = data
		*zknode.LastReadVersion = stat.Version
		*zknode.LastReadTime = time.Now()
	}
	return zknode, err
}

func (zc *ZkConnect) CreateNode(node ZkNode) (ZkNode, error) {
	var errmsg []string
	tree := strings.Split(node.Path, "/")
	for i := range tree {
		zpath := "/" + filepath.Join(tree[:i+1]...)
		if ok, err := zc.ExistNode(zpath); len(zpath) > 1 && !ok && err == nil {
			if _, err := zc.Create(zpath, []byte{}, 0, zk.WorldACL(zk.PermAll)); err != nil {
				errmsg = append(errmsg, fmt.Sprintf("error save/create node %s: %s", zpath, err))
			}
		}
	}
	if len(errmsg) > 0 {
		return node, fmt.Errorf("%s", strings.Join(errmsg, ","))
	}
	*node.LastWriteTime = time.Now()
	return node, nil
}

func (zc *ZkConnect) SaveNode(node ZkNode) (ZkNode, error) {
	var err error
	logger.Info("start save zknode %s", node.Path)
	logger.Debug("zknode %s current/last read version %d/%d", node.Path, node.GetLastReadVersion(), node.GetNodeVersion())
	if node.GetLastReadVersion() != -1 && node.GetLastReadVersion() < node.GetNodeVersion() {
		if _, err = zc.Set(node.Path, *node.Data, node.GetLastReadVersion()); err == nil {
			logger.Info("success save zknode %s", node.Path)
			*node.LastWriteTime = time.Now()
			return node, nil
		} else {
			return node, fmt.Errorf("error save zknode %s with version %d/%d(current/last read): error %s", node.Path,
				node.GetNodeVersion(), node.GetLastReadVersion(), err)
		}
	}
	return node, fmt.Errorf("error save zknode %s with version %d/%d(current/last read)", node.Path,
		node.GetNodeVersion(), node.GetLastReadVersion())
}

func (zc *ZkConnect) DeleteNode(node ZkNode) (ZkNode, error) {
	var (
		err  error
		stat *zk.Stat
	)
	deleteNodePath := NodePath(node.Path)
	_, stat, _ = zc.Get(deleteNodePath)
	logger.Debug("start delete zknode %s", deleteNodePath)
	if stat.Version == 0 || stat.Version < node.GetNodeVersion() {
		if err = zc.Delete(deleteNodePath, stat.Version); err != nil {
			return node, fmt.Errorf("delete node %s: %s", deleteNodePath, err)
		}
	}
	logger.Debug("done delete zknode %s", node.Path)
	*node.LastWriteTime = time.Now()
	return node, nil
}
