from yalibrary.upload import mds_uploader
from library.python import resource

import os
import click
import dateutil.parser
from clickhouse_driver import Client
import re
import tempfile
import shutil

from direct.infra.java_threaddump_flamegraph import db_config


class Flamer(object):
    def __init__(self, service, trace_like, datetime_from, datetime_to, where, config_file, ttl, no_omit_waiting):
        self.service = service
        self.trace_like = trace_like
        self.datetime_from = datetime_from
        self.datetime_to = datetime_to
        self.where = where
        self.ttl = ttl
        self.config_file = config_file
        self.no_omit_waiting = no_omit_waiting

        self.clh = self.clickhouse()

    def clickhouse(self):
        cfg = db_config.DbConfig(config_file=self.config_file).get_db_config("ppchouse:cloud")
        return Client(host=cfg['host'], port=9440, database=cfg['db'],
                      user=cfg['user'], password=cfg['pass'],
                      secure=bool(cfg.get('ssl', False)), verify=False)

    def run(self):
        temp_dir = tempfile.mkdtemp()
        try:
            self.load_data(temp_dir)
            self.make_graph(temp_dir)
            self.upload_to_mds(temp_dir)
        finally:
            shutil.rmtree(temp_dir)

    def load_data(self, temp_dir):
        print('executing query...')
        where = [
            """log_time BETWEEN %(datetime_from)s AND %(datetime_to)s
                 AND like(thread.stacktrace, '_at%%')
                 AND not like(thread.stacktrace, '%%EPoll.wait%%')
                 AND not like(thread.stacktrace, '%%epollWait%%')
                 AND not like(thread.header, '%%Reference Handler%%')
                 AND service like %(service)s
                 AND thread.stacktrace like %(trace_like)s""",
        ]
        if self.where:
            where.append("(" + self.where + ")")
        rows = self.clh.execute("""SELECT thread.header, thread.state, thread.stacktrace
                 FROM java_threaddump
                 ARRAY JOIN thread
                 WHERE """ + ' AND '.join(where) + """
                 ORDER BY log_time ASC
                 LIMIT 0, 10000
                 """, {'service': self.service, 'trace_like': self.trace_like, 'datetime_from': self.datetime_from,
                       'datetime_to': self.datetime_to})

        print('write file...')
        with open(temp_dir + '/spaced.txt', 'w') as f:
            for i, row in enumerate(rows):
                f.write(self.format_stacktrace(*row))
                f.write("\n")

    def format_stacktrace(self, header, state, stacktrace):
        ans = '{header}\n' \
              '   java.lang.Thread.State: {state}\n' \
              '{stacktrace}\n'.format(**locals())
        # trim some functions
        ans = re.sub(r'GeneratedMethodAccessor\d+', 'GeneratedMethodAccessor', ans)
        ans = re.sub(r'\$\$EnhancerBySpringCGLIB\$\$.+\.', '$$EnhancerBySpringCGLIB$$.', ans)
        ans = re.sub(r'\$\$Lambda\$.+\.', 'Lambda.', ans)
        return ans

    def make_graph(self, temp_dir):
        print('generate flamegraph...')
        open(temp_dir + "/stackcollapse-jstack.pl", 'wb').write(resource.find("/flamegraph/stackcollapse-jstack.pl"))
        open(temp_dir + "/stackcollapse-java-exceptions.pl", 'wb').write(resource.find("/flamegraph/stackcollapse-java-exceptions.pl"))
        open(temp_dir + "/flamegraph.pl", 'wb').write(resource.find("/flamegraph/flamegraph.pl"))

        if self.no_omit_waiting:
            os.system("perl {0}/stackcollapse-java-exceptions.pl {0}/spaced.txt > {0}/collapsed.txt".format(temp_dir))
        else:
            # stackcollapse-jstack.pl use only RUNNABLE threads
            os.system("perl {0}/stackcollapse-jstack.pl {0}/spaced.txt > {0}/collapsed.txt".format(temp_dir))
        os.system("perl {0}/flamegraph.pl {0}/collapsed.txt > {0}/flamegraph.svg".format(temp_dir))

    def upload_to_mds(self, temp_dir):
        print('upload to mds...')
        if self.ttl is None:
            ttl = 14
        elif self.ttl == 'inf':
            ttl = None
        else:
            ttl = int(self.ttl)

        ret = mds_uploader.do(
            [temp_dir + "/flamegraph.svg"],
            ttl=int(ttl) if ttl is not None and ttl != 'inf' else None
        )
        url = mds_uploader.make_mds_url(ret)
        print("")
        print(url)


def validate_datetime(ctx, param, value):
    if value is None:
        return None
    try:
        return dateutil.parser.parse(value)
    except ValueError:
        raise click.BadParameter("Incorrect datetime format, see dateutil.parser documentation")


@click.command(help='Script for drawing flamegraphs from threaddumps in logviewer. '
                    'Usage example: '
                    'java-threaddump-flamegraph -s %web% -from "2020-03-20 16:50:00" -to "2020-03-20 16:59:59"')
@click.option('--service', '-s', type=str, default='%',
              help="like pattern for service name")
@click.option('--trace-like', '-l', 'trace_like', type=str, default='%',
              help="like pattern for stacktrace")
@click.option('--from', '-f', 'datetime_from', default=None, callback=validate_datetime, required=True,
              help="from datetime")
@click.option('--to', '-t', 'datetime_to', default=None, callback=validate_datetime, required=True,
              help="to datetime")
@click.option('--where', '-w', 'where', type=str, default=None,
              help="arbitrary sql where expression, you can use log_time, host, thread.state, thread.stacktrace, ...")
@click.option('--ttl', type=str,
              help="Time to live in days or inf")
@click.option('--config-file', '-c', 'config_file', type=str, default=db_config.DEFAULT_CONFIG,
              help=f"Path to db-config file, default: {db_config.DEFAULT_CONFIG}")
@click.option('--no-omit-waiting', '-O', 'no_omit_waiting', is_flag=True, default=False,
              help="Do not omit waiting threads. Useful for external call profiling")
def cli(service, trace_like, datetime_from, datetime_to, where, config_file, ttl, no_omit_waiting):
    Flamer(service, trace_like, datetime_from, datetime_to, where, config_file, ttl, no_omit_waiting).run()


if __name__ == '__main__':
    cli()
