#!/usr/bin/env python
# -*- coding:utf-8 -*-

from flask import Flask, request
import json
import datetime
import requests
import urllib
import logging
import hashlib
from redis.sentinel import Sentinel
from logging.handlers import SysLogHandler

requests.packages.urllib3.disable_warnings()
logging.getLogger("requests").setLevel(logging.WARNING)

redis_hosts = ['myt-4oi0grct5j1i24x8.db.yandex.net', 'sas-35ttmjsao32a8zch.db.yandex.net', 'vla-azc461nwux8kp3nw.db.yandex.net']
redis_sentinel = Sentinel([(h, 26379) for h in redis_hosts], socket_timeout=0.1)
with open('/etc/direct-tokens/redis_juggler_history_password') as redis_pass_fh:
    redis_password = redis_pass_fh.read().strip()
redis_master = redis_sentinel.master_for('juggler_history_redis', password=redis_password)
REDIS_TTL = 86400

app = Flask(__name__)
HOST = "ppchouse-cloud.direct.yandex.net:8443"
USER = "direct_writer"
DB = "directdb"
TABLE = "juggler_history_buffer"

with open("/etc/direct-tokens/clickhouse_direct_writer") as ch_pass_fh:
    CH_PASSWORD = ch_pass_fh.read().strip()

logging.getLogger('werkzeug').disabled = True
app.logger.disabled = True

logger = logging.getLogger()
logger.setLevel(logging.INFO)
syslog = SysLogHandler('/dev/log', facility="local0")
syslog.setFormatter(logging.Formatter('juggler-history.log: (%(levelname)s) %(message)s'))
logger.handlers = []
logger.addHandler(syslog)


AGGR_PROPS = {
    'names': ['host_name', 'service_name', 'status', 'hash', 'flags'],
    'prefix': 'aggr_'
}

CHILD_PROPS = {
    'names': ['host_name', 'service_name', 'instance_name', 'status', 'status_mtime', 'description', 'flags'],
    'prefix': 'child_'
}

ACTUAL_PROPS = {
    'names': ['status', 'status_mtime', 'description'],
    'prefix': 'actual_'
}

ALL_PROPS = [group['prefix'] + name for group in [AGGR_PROPS, CHILD_PROPS, ACTUAL_PROPS] for name in group['names']]
EVENT_PROPS = [prop for prop in ALL_PROPS if prop != AGGR_PROPS['prefix'] + "hash"]
SELECTOR_PROPS = [
    group['prefix'] + name
    for group, names in [
        (AGGR_PROPS, ['host_name', 'service_name']), (CHILD_PROPS, ['host_name', 'service_name', 'instance_name'])
    ] for name in names
]


def get_event_id(event):
    return hashlib.md5(
        u";".join(
            event.get(prop, u"") if not isinstance(event.get(prop, u""), list) else u",".join(event.get(prop, [])).decode("utf-8")
            for prop in EVENT_PROPS
        ).encode("utf-8")
    ).hexdigest()


def get_selector_id(event):
    return hashlib.md5(u";".join(event.get(prop, u"") for prop in SELECTOR_PROPS)).hexdigest()


@app.route('/alive', methods=['GET', 'POST', 'OPTIONS'])
def alive():
    return ('', 200)


@app.route('/save_notifications', methods=['GET', 'POST', 'OPTIONS'])
def receive_notifications():
    global logger

    if not request.data:
        return ('', 200)

    rows_to_send = []
    checks = json.loads(request.data)['checks']
    cur_dt = datetime.datetime.now()
    time_received = cur_dt.strftime("%Y-%m-%d %H:%M:%S").decode("utf-8")
    date_received = cur_dt.strftime("%Y-%m-%d").decode("utf-8")

    for event in checks:
        try:
            flat_event = {'time_received': time_received, 'date_received': date_received}
            flat_event.update({AGGR_PROPS['prefix'] + prop: event[prop] for prop in AGGR_PROPS['names']})

            if event.get('children'):
                for child in event['children']:
                    flat_event_ext = flat_event.copy()
                    flat_event_ext.update({
                        CHILD_PROPS['prefix'] + prop: child[prop] for prop in CHILD_PROPS['names'] if child.get(prop)
                    })
                    if 'actual' in child:
                        flat_event_ext.update({
                            ACTUAL_PROPS['prefix'] + prop: child['actual'][prop]
                            for prop in ACTUAL_PROPS['names'] if child['actual'].get(prop)
                        })

                    for lvl in [CHILD_PROPS['prefix'], ACTUAL_PROPS['prefix']]:
                        cur_key = lvl + 'status_mtime'
                        if cur_key in flat_event_ext:
                            flat_event_ext[cur_key] = datetime.datetime.fromtimestamp(
                                float(flat_event_ext[cur_key])
                            ).strftime("%Y-%m-%d %H:%M:%S").decode("utf-8")

                    flat_event_ext['event_id'] = get_event_id(flat_event_ext)
                    flat_event_ext['selector_id'] = get_selector_id(flat_event_ext)
                    rows_to_send.append(flat_event_ext)

            else:
                flat_event['event_id'] = get_event_id(flat_event)
                flat_event['selector_id'] = get_selector_id(flat_event)
                rows_to_send.append(flat_event)

        except Exception as e:
            logger.exception("unexpected exception: %s %s" % (type(e), e))

    rows_to_send_filtered = []
    for event in rows_to_send:
        try:
            last_selector_value = redis_master.get(event['selector_id'])
            if not last_selector_value:
                last_selector_value = ""

            if last_selector_value != event['event_id']:
                redis_master.setex(event['selector_id'], REDIS_TTL, event['event_id'])
                rows_to_send_filtered.append(event)

        except Exception as e:
            logger.exception("unexpected exception: %s %s" % (type(e), e))

    rows_to_send = rows_to_send_filtered

    try:
        # при нескольких запущенных воркерах не хочется потерять данные на самом редко используемом воркере
        if len(rows_to_send) >= 1:
            data_to_send = u"\n".join([json.dumps(el) for el in rows_to_send])

            req = requests.post(
                "https://%s/?query=%s" % (
                    HOST, urllib.quote("INSERT INTO %s.%s FORMAT JSONEachRow" % (DB, TABLE))
                ),
                headers={"X-ClickHouse-User": USER, "X-ClickHouse-Key": CH_PASSWORD},
                data=data_to_send,
                verify=False
            )

            if req.status_code != 200:
                logger.critical("error while sending rows, status code: %d; response: %s" % (req.status_code, req.text))

    except Exception as e:
        logger.exception("unexpected exception: %s %s" % (type(e), e))

    return ('', 200)


if __name__ == "__main__":
    app.run(host="::", port=8041)
