package ru.yandex.market.clickhouse.dealer;

import com.google.common.annotations.VisibleForTesting;
import io.grpc.ManagedChannel;
import io.grpc.netty.shaded.io.grpc.netty.NettyChannelBuilder;
import ru.yandex.market.request.trace.Module;
import ru.yandex.market.tsum.clients.iam.IamClientAuthInterceptor;
import src.ru.yandex.market.grpc.trace.TraceClientInterceptor;
import yandex.cloud.iam.v1.IamTokenServiceGrpc;
import yandex.cloud.mdb.clickhouse.v1.ClusterOuterClass;
import yandex.cloud.mdb.clickhouse.v1.ClusterServiceGrpc;
import yandex.cloud.mdb.clickhouse.v1.ClusterServiceOuterClass;

import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @author Aleksei Malygin <a href="mailto:Malygin-Me@yandex-team.ru"></a>
 * Date: 2018-12-06
 */
public class DealerMdbService implements AutoCloseable {

    private static final int UUID_COMPONENTS_COUNT = 5;

    private ClusterServiceGrpc.ClusterServiceBlockingStub clusterService;
    private ManagedChannel iamChannel;
    private ManagedChannel grpcChannel;

    @VisibleForTesting
    DealerMdbService() {
    }

    public DealerMdbService(String mdbToken, String mdbEndpoint) {
        iamChannel = NettyChannelBuilder.forTarget(mdbEndpoint)
            .intercept(new TraceClientInterceptor(Module.IAM_YANDEX_CLOUD))
            .build();

        grpcChannel = NettyChannelBuilder.forTarget(mdbEndpoint)
            .intercept(
                new TraceClientInterceptor(Module.DBAAS),
                new IamClientAuthInterceptor(mdbToken, IamTokenServiceGrpc.newBlockingStub(iamChannel))
            )
            .build();

        clusterService = ClusterServiceGrpc.newBlockingStub(grpcChannel);
    }

    private ClusterOuterClass.Cluster getCluster(String clusterId) {
        return clusterService.get(
            ClusterServiceOuterClass.GetClusterRequest.newBuilder()
                .setClusterId(clusterId)
                .build()
        );
    }

    public String getClusterName(String clusterId) {
        return getCluster(clusterId).getName();
    }

    public List<String> getClickhouseHosts(String clusterId) {
        return clusterService
            .listHosts(
                ClusterServiceOuterClass.ListClusterHostsRequest.newBuilder()
                    .setClusterId(clusterId)
                    .build()
            )
            .getHostsList().stream()
            .filter(h -> (h.getType() == ClusterOuterClass.Host.Type.CLICKHOUSE))
            .map(ClusterOuterClass.Host::getName)
            .collect(Collectors.toList());
    }

    /**
     * @param clusterId - Since 2018.12.11 clusterId format has been changed for a new one (e.g. "mdbi2ha8ttbhpus8k5o0")
     *                  previous clusterId format is matched by UUID type. Now both clusterId formats are available.
     * @return true/false
     */
    private boolean isUUIDClusterId(String clusterId) {
        return UUID_COMPONENTS_COUNT == clusterId.split("-").length;
    }

    /**
     * @param clusterId - clusterId
     * @return cluster(Name / Id) - system.clusters table contains column 'cluster'
     * for new mdb clusters - this column contains cluster id,
     * for old mdb clusters - this column contains cluster name
     */
    public String getClickHouseSystemClusterName(String clusterId) {
        return isUUIDClusterId(clusterId) ? getClusterName(clusterId) : clusterId;
    }

    @Override
    public void close() throws Exception {
        closeChannel(iamChannel);
        closeChannel(grpcChannel);
    }

    private void closeChannel(ManagedChannel channel) throws InterruptedException {
        if (channel != null) {
            channel.shutdownNow();
            channel.awaitTermination(2, TimeUnit.SECONDS);
        }
    }
}
