package ru.yandex.market.clickhouse.dealer;

import com.google.common.annotations.VisibleForTesting;
import com.mongodb.MongoInterruptedException;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.recipes.leader.LeaderLatch;
import org.apache.curator.framework.recipes.leader.LeaderLatchListener;
import org.apache.curator.framework.state.ConnectionState;
import org.apache.curator.framework.state.ConnectionStateListener;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.ThreadContext;
import org.springframework.dao.OptimisticLockingFailureException;
import org.springframework.data.mongodb.UncategorizedMongoDbException;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.market.application.monitoring.ComplicatedMonitoring;
import ru.yandex.market.application.monitoring.MonitoringUnit;
import ru.yandex.market.clickhouse.dealer.config.DealerConfig;
import ru.yandex.market.clickhouse.dealer.config.DealerConfigurationService;
import ru.yandex.market.clickhouse.dealer.config.DealerGlobalConfig;
import ru.yandex.market.clickhouse.dealer.state.DealerDao;
import ru.yandex.market.clickhouse.dealer.state.DealerState;
import ru.yandex.market.clickhouse.dealer.tm.TransferManagerClient;

import java.io.IOException;
import java.net.InetAddress;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 09/04/2018
 */
public class DealerService implements ConnectionStateListener, LeaderLatchListener, Runnable {

    private static final Logger log = LogManager.getLogger();

    private static final int MONITORING_CHECK_INTERVAL_SECONDS = 60;

    private final DealerDao dealerDao;
    private final Yt yt;
    private final TransferManagerClient tmClient;
    private final DealerConfigurationService configurationService;
    private final ComplicatedMonitoring monitoring;
    private final LeaderLatch leaderLatch;

    private final MonitoringUnit zkUnit;
    private final MonitoringUnit serviceUnit;
    private final MonitoringUnit dataMismatchUnit;

    private final int threadCount;
    private final DealerGlobalConfig globalConfig;

    private final AtomicReference<LeaderContext> leaderContextReference = new AtomicReference<>();

    public DealerService(DealerDao dealerDao, Yt yt, TransferManagerClient tmClient,
                         DealerConfigurationService configurationService,
                         CuratorFramework curatorFramework, ComplicatedMonitoring monitoring) {
        this.dealerDao = dealerDao;
        this.yt = yt;
        this.tmClient = tmClient;
        this.monitoring = monitoring;
        this.configurationService = configurationService;
        zkUnit = monitoring.createUnit("zk");
        serviceUnit = monitoring.createUnit("service");
        dataMismatchUnit = monitoring.createUnit("dataMismatch");
        leaderLatch = (curatorFramework == null) ? null : createLeaderLatch(curatorFramework);
        threadCount = configurationService.getThreadCount();
        globalConfig = configurationService.getGlobalConfig();
    }

    public void start() throws Exception {
        if (leaderLatch == null) {
            zkUnit.ok("Leadership election disabled");
            createLeaderContext();
        } else {
            try {
                leaderLatch.start();
                Runtime.getRuntime().addShutdownHook(
                    new Thread(() -> {
                        try {
                            leaderLatch.close();
                        } catch (IOException e) {
                            log.error("Cannon release leadership", e);
                        }
                    })
                );
                leaderLatch.await(1, TimeUnit.SECONDS);
                reportLeader();
            } catch (Exception e) {
                log.error("Failed to start leader latch");
                throw e;
            }
        }
        new Thread(this, "monitoring").start();
    }

    @Override
    public void run() {
        while (!Thread.interrupted()) {
            try {
                updateServiceMonitoring();
            } catch (Exception e) {
                log.error("Failed to update service monitoring");
                serviceUnit.critical("Monitoring exception");
            }
            try {
                TimeUnit.SECONDS.sleep(MONITORING_CHECK_INTERVAL_SECONDS);
            } catch (InterruptedException ignored) {
            }
        }
    }

    private void updateServiceMonitoring() {
        if (leaderLatch != null && !leaderLatch.hasLeadership()) {
            serviceUnit.ok();
            dataMismatchUnit.ok();
            return;
        }
        LeaderContext leaderContext = leaderContextReference.get();

        if (leaderContext == null) {
            log.error("No leader context, but I am dictator. Service is not working");
            serviceUnit.critical("Service is not working");
            return;
        }

        //TODO status from leaderContext;
        serviceUnit.ok();
        checkDataMismatches();
    }

    @VisibleForTesting
    void checkDataMismatches() {
        long dataMismatchCount = dealerDao.getDataMismatchCount(
            Instant.now().minus(globalConfig.getDataMismatchMonitoringPeriodHours(), ChronoUnit.HOURS)
        );

        String message = "Data mismatch count > %d ---> ";
        if (dataMismatchCount > globalConfig.getDataMismatchesForCritical()) {
            dataMismatchUnit.critical(
                String.format(message, globalConfig.getDataMismatchesForCritical()) + dataMismatchCount
            );
        } else if (dataMismatchCount > globalConfig.getDataMismatchesForWarning()) {
            dataMismatchUnit.warning(
                String.format(message, globalConfig.getDataMismatchesForWarning()) + dataMismatchCount
            );
        } else {
            dataMismatchUnit.ok();
        }
    }

    private LeaderLatch createLeaderLatch(CuratorFramework curatorFramework) {
        try {
            curatorFramework.getConnectionStateListenable().addListener(this);
            String host = InetAddress.getLocalHost().getCanonicalHostName();
            log.info("I am " + host);
            LeaderLatch latch = new LeaderLatch(curatorFramework, "/leader", host);
            latch.addListener(this);
            return latch;
        } catch (Exception e) {
            log.error("Failed to create leader latch", e);
            throw new RuntimeException("Failed to create leader latch", e);
        }
    }

    @Override
    public void stateChanged(CuratorFramework client, ConnectionState newState) {
        if (!newState.isConnected()) {
            log.error("Zk connection lost. State {}", newState.name());
            zkUnit.critical("Lost connection to ZK");
        } else {
            log.info("Zk connection state {}", newState.name());
            reportLeader();
        }
    }

    public void createLeaderContext() {
        leaderContextReference.updateAndGet(leaderContext -> {
            if (leaderContext != null) {
                log.warn("Leader context already exists. This is strange...");
            } else {
                log.info("Creating new leader context");
                leaderContext = new LeaderContext(configurationService.getConfigs());
            }
            return leaderContext;
        });
    }

    @Override
    public void isLeader() {
        log.info("Now I'm the dictator here!");
        createLeaderContext();
        reportLeader();
    }

    @Override
    public void notLeader() {
        log.warn("Leadership lost.");
        LeaderContext context = leaderContextReference.getAndSet(null);
        if (context != null) {
            log.info("Stopping all processes (not a leader)");
            context.shutdown();
        }
        reportLeader();
    }

    private void reportLeader() {
        try {
            String leader = leaderLatch.getLeader().getId();
            log.info("Current leader: {}", leader);
            if (leaderLatch.hasLeadership()) {
                zkUnit.ok("I am the dictator!");
            } else {
                zkUnit.ok("Not a leader. Current: " + leader);
            }
        } catch (Exception e) {
            log.error("Failed to get current leader", e);
            zkUnit.critical("Failed to get current leader", e);
        }
    }

    @VisibleForTesting
    class LeaderContext {
        private final List<MonitoringUnit> monitoringUnits = new ArrayList<>();
        private final ScheduledExecutorService executorService;
        private volatile boolean active = true;

        private LeaderContext(List<DealerConfig> configs) {
            this.executorService = Executors.newScheduledThreadPool(threadCount);

            for (DealerConfig config : configs) {
                MonitoringUnit configUnit = monitoring.getOrCreateUnit(config.getConfigName());
                monitoringUnits.add(configUnit);
                executorService.execute(new DealerRunnable(this, config, configUnit));
                wait(2);
            }
        }

        private void wait(int seconds) {
            try {
                TimeUnit.SECONDS.sleep(seconds);
            } catch (InterruptedException e) {
                throw new RuntimeException("Can't wait " + seconds + " seconds between processing configs", e);
            }
        }

        private void shutdown() {
            active = false;
            executorService.shutdownNow();
            for (MonitoringUnit monitoringUnit : monitoringUnits) {
                monitoringUnit.ok();
            }
        }

        private void reschedule(Runnable runnable, int delayMinutes) {
            if (active) {
                executorService.schedule(runnable, delayMinutes, TimeUnit.MINUTES);
            }
        }
    }

    private class DealerRunnable implements Runnable {

        private final LeaderContext leaderContext;
        private final DealerConfig config;
        private final MonitoringUnit configUnit;
        private DealerWorker worker;
        private DealerState state;

        private DealerRunnable(LeaderContext leaderContext, DealerConfig config, MonitoringUnit configUnit) {
            this.leaderContext = leaderContext;
            this.config = config;
            this.configUnit = configUnit;
        }

        @Override
        public void run() {
            try {
                configUnit.ok();
                doRun();
                leaderContext.reschedule(this, config.getCheckIntervalMinutes());
            } catch (Throwable e) {
                log.error("Unhandled exception in thread for config {}.", config.getKey(), e);
                if (!leaderContext.active) {
                    return;
                }
                configUnit.critical("Unhandled exception. Message: " + e.getMessage(), e);
                leaderContext.reschedule(
                    new DealerRunnable(leaderContext, config, configUnit),
                    config.getRetryOnErrorIntervalMinutes()
                );
            } finally {
                ThreadContext.clearAll();
            }
        }

        private void doRun() {
            try {
                log.info("Processing config {}", config.getKey());
                init();
                worker.work();
                state.cleanError();
                dealerDao.save(state);
                configUnit.ok();
            } catch (InterruptedException | MongoInterruptedException e) {
                log.warn("Thread interrupted {}", config.getKey());
                configUnit.ok();
            } catch (UncategorizedMongoDbException e) {
                if (e.getCause() instanceof MongoInterruptedException) {
                    log.warn("Thread interrupted {}", config.getKey());
                    configUnit.ok();
                } else {
                    throw e;
                }
            } catch (OptimisticLockingFailureException e) {
                log.warn("OptimisticLockException occurred", e);
                configUnit.warning(e.getMessage());
            } catch (DealerException e) {
                log.error("Exception in thread for config {}.", config.getKey(), e);
                int errorNumber = state.getError() == null ? 1 : state.getError().getNumber() + 1;
                state.setError(new DealerState.Error(e.getMessage(), errorNumber));
                dealerDao.save(state);

                if (errorNumber >= worker.getGlobalConfig().getTheNumberOfConfigErrorsForWarning()) {
                    configUnit.warning(errorNumber + " times failed", e);
                }
            }
        }

        private void init() {
            if (worker != null) {
                return;
            }
            log.info("Initializing worker for config {}", config.getKey());
            ThreadContext.put("config", config.getConfigName());
            ThreadContext.put("cluster", config.getKey().getClickHouseClusterId());
            ThreadContext.put("table", config.getKey().getClickHouseFullTableName());
            state = dealerDao.loadState(config.getKey()).orElseGet(() -> new DealerState(config.getKey()));
            dealerDao.save(state); //to increment version
            worker = new DealerWorker(config, yt, tmClient, dealerDao, state);
        }
    }
}
