package ru.yandex.market.clickhouse.dealer;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import ru.yandex.clickhouse.settings.ClickHouseProperties;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.ytree.YTreeStringNodeImpl;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeStringNode;
import ru.yandex.market.clickhouse.ddl.ClickHouseCluster;
import ru.yandex.market.clickhouse.ddl.ClickHouseDdlService;
import ru.yandex.market.clickhouse.ddl.ClickHouseTableDefinition;
import ru.yandex.market.clickhouse.ddl.ClickHouseTableDefinitionImpl;
import ru.yandex.market.clickhouse.ddl.TableDdlState;
import ru.yandex.market.clickhouse.ddl.TableName;
import ru.yandex.market.clickhouse.ddl.engine.DistributedEngine;
import ru.yandex.market.clickhouse.dealer.clickhouse.SingletonClickHousePartitionExtractor;
import ru.yandex.market.clickhouse.dealer.config.DealerClusterConfig;
import ru.yandex.market.clickhouse.dealer.config.DealerConfig;
import ru.yandex.market.clickhouse.dealer.config.DealerGlobalConfig;
import ru.yandex.market.clickhouse.dealer.operation.CopyOperation;
import ru.yandex.market.clickhouse.dealer.operation.DealerOperation;
import ru.yandex.market.clickhouse.dealer.operation.OperationContext;
import ru.yandex.market.clickhouse.dealer.operation.UpdateOperation;
import ru.yandex.market.clickhouse.dealer.state.DataMismatch;
import ru.yandex.market.clickhouse.dealer.state.DealerDao;
import ru.yandex.market.clickhouse.dealer.state.DealerState;
import ru.yandex.market.clickhouse.dealer.state.DealerStateProcessor;
import ru.yandex.market.clickhouse.dealer.state.PartitionState;
import ru.yandex.market.clickhouse.dealer.tm.TmClickHouseClusterOptions;
import ru.yandex.market.clickhouse.dealer.tm.TmTaskState;
import ru.yandex.market.clickhouse.dealer.tm.TmYt2ClickHouseCopyTask;
import ru.yandex.market.clickhouse.dealer.tm.TransferManagerClient;
import ru.yandex.market.clickhouse.dealer.util.Retrier;
import ru.yandex.market.monitoring.MonitoringStatus;
import ru.yandex.market.rotation.DataRotationService;
import ru.yandex.market.rotation.DataRotationTask;
import ru.yandex.market.rotation.ObsoletePartition;

import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.OptionalInt;
import java.util.Set;
import java.util.SortedMap;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;
import java.util.stream.Collectors;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 15/02/2018
 */
public class DealerWorker implements OperationContext {

    private static final Logger log = LogManager.getLogger();
    private static final int TM_POLL_INTERVAL_SECONDS = 30;
    private static final int DBAAS_CLICKHOUSE_HTTPS_PORT = 8443;

    private static final int TRUNCATE_TABLE_RETRIES = 10;
    private static final int TRUNCATE_TABLE_SLEEP_SECONDS = 120;
    private static final int MIN_NUMBER_TO_MAJORITY_QUORUM = 3;
    private static final int CLICKHOUSE_READ_ATTEMPTS = 10;

    private final DealerConfig dealerConfig;

    private final Yt yt;
    private final TransferManagerClient tmClient;
    private final DealerDao dealerDao;
    private final DealerState state;

    private ClickHouseRuntimeParams runtimeOptions;
    private ClickHouseDdlService ddlService;
    private ClickHouseDdlService ddlServiceForClusterNext;
    private DataRotationService dataRotationService;

    private static final Set<PartitionState.Status> TRANSFER_STATUSES = ImmutableSet.of(
        PartitionState.Status.NEW,
        PartitionState.Status.TRANSFERRED_NEED_UPDATE,
        PartitionState.Status.TRANSFERRED_DATA_MISMATCH
    );

    private static final Predicate<PartitionState.Status> SHOULD_BE_UPDATED = (status) ->
        status != PartitionState.Status.SKIPPED &&
            status != PartitionState.Status.YT_DELETED &&
            status != PartitionState.Status.ROTATED;

    private Instant lastValidationTime;

    public DealerWorker(DealerConfig dealerConfig, Yt yt, TransferManagerClient tmClient,
                        DealerDao dealerDao, DealerState state) {
        this.dealerConfig = dealerConfig;
        this.yt = yt;
        this.tmClient = tmClient;
        this.dealerDao = dealerDao;
        this.state = state;
    }

    public void work() throws InterruptedException, DealerException {
        reloadRuntimeParams();
        applyDdl();
        runNewCycle();
    }

    private void runNewCycle() throws InterruptedException, DealerException {
        processCurrentOperation();
        processYtPartitions();
        validateYtPartitionsFromClickHouse();
        runRotation();
        saveState();
        process();
    }

    private void processCurrentOperation() throws InterruptedException, DealerException {
        DealerOperation currentOperation = state.getCurrentOperation();
        if (currentOperation != null) {
            doOperation(currentOperation);
        }
    }

    @VisibleForTesting
    void runRotation() throws DealerException {

        if (dealerConfig.getRotationPeriodDays() <= 0) {
            log.info("Rotation is off for table: {}", dealerConfig.getDataTable());
            return;
        }
        log.info("Rotation is on for table: {}, dataRotationDays='{}'",
            dealerConfig.getDataTable(), dealerConfig.getRotationPeriodDays());

        DataRotationTask dataRotationTask = dataRotationService.findObsoletePartitions(
            dealerConfig.getDataTable(),
            dealerConfig.getRotationPeriodDays()
        );

        if (dataRotationTask.getObsoletePartitions().isEmpty()) {
            return;
        }

        DealerStateProcessor.checkForPartitionsRequiredBeRotated(state, dataRotationTask.getPivotPartition());
        DealerStateProcessor.checkForNewPartitionsRequiredRotation(state, dataRotationTask.getPivotPartition());
        DealerStateProcessor.checkForRotatedPartitionsRequiredBeTransferred(
            state, dataRotationTask.getPivotPartition()
        );

        logRotation(dataRotationTask);
        markRotationProcessPartitions(dataRotationTask, PartitionState.Status.ROTATING);
        deleteObsoletePartitions(dataRotationTask);
        markRotationProcessPartitions(dataRotationTask, PartitionState.Status.ROTATED);
    }

    private void deleteObsoletePartitions(DataRotationTask dataRotationTask) throws DealerException {
        try {
            log.info(
                "Clickhouse partitions: '{}' marked as rotated for '{}' config ",
                dataRotationService.deleteObsoletePartitions(dataRotationTask).stream()
                    .map(ObsoletePartition::getPartition)
                    .collect(Collectors.toList()),
                dealerConfig.getConfigName()
            );
        } catch (Exception e) {
            throw new DealerException("Rotation process failed", e);
        }
    }

    private void markRotationProcessPartitions(DataRotationTask dataRotationTask, PartitionState.Status status) {
        Set<String> shouldBeRotatedChPartitions = dataRotationTask.getDistinctObsoletePartitions();
        log.info("Marking '{}' partitions as {}", shouldBeRotatedChPartitions.size(), status);
        shouldBeRotatedChPartitions.forEach(
            chPartition -> state.getPartitionStates(chPartition).forEach(
                ytPartition -> ytPartition.setStatus(status)
            )
        );
        saveState();
    }

    private void logRotation(DataRotationTask dataRotationTask) {
        Collection<ObsoletePartition> obsoletePartitions = dataRotationTask.getObsoletePartitions();
        TableName table = obsoletePartitions.stream().findFirst().get().getTableName();
        log.info(
            "'{}' obsolete partitions, which are older than '{}' days (less than {} partition), found in table '{}'",
            dataRotationTask.getDistinctObsoletePartitions().size(),
            dataRotationTask.getRotationPeriodDays(),
            dataRotationTask.getPivotPartition(),
            table
        );
    }

    private void process() throws InterruptedException, DealerException {
        Multimap<String, PartitionState> transferPartitionStates = getTransferStates();

        List<String> clickHousePartitions = transferPartitionStates.keySet().stream()
            .sorted(Comparator.reverseOrder())
            .collect(Collectors.toList());

        for (String clickHousePartition : clickHousePartitions) {
            process(
                clickHousePartition,
                transferPartitionStates.get(clickHousePartition)
            );
        }
    }

    private void process(String clickHousePartition, Collection<PartitionState> partitionStates)
        throws InterruptedException, DealerException {

        boolean isUpdateOperation = partitionStates.stream().anyMatch(
            ps -> ps.getStatus() == PartitionState.Status.TRANSFERRED_NEED_UPDATE
                || ps.getStatus() == PartitionState.Status.TRANSFERRED_DATA_MISMATCH
        );

        if (isUpdateOperation) {
            runUpdateProcess(clickHousePartition, partitionStates);
        } else {
            runCopyProcess(partitionStates);
        }
    }

    @VisibleForTesting
    Multimap<String, PartitionState> getTransferStates() {

        Set<String> clickHousePartitionsForUpdate = state.getPartitionStates().stream()
            .filter(ps -> TRANSFER_STATUSES.contains(ps.getStatus()))
            .map(PartitionState::getClickHousePartition)
            .distinct()
            .sorted(Comparator.reverseOrder())
            .limit(dealerConfig.getMaxClickHousePartitionsPerIteration())
            .collect(Collectors.toSet());

        return Multimaps.index(
            state.getPartitionStates().stream()
                .filter(ps -> clickHousePartitionsForUpdate.contains(ps.getClickHousePartition()))
                .collect(Collectors.toList()),
            PartitionState::getClickHousePartition
        );
    }

    /* Copy operation expects all partitions with only NEW statuses */
    private void runCopyProcess(Collection<PartitionState> partitionStatesToCopy)
        throws InterruptedException, DealerException {

        partitionStatesToCopy = partitionStatesToCopy.stream()
            .filter(ps -> ps.getStatus() == PartitionState.Status.NEW)
            .sorted(Comparator.comparing(PartitionState::getYtPartition).reversed())
            .collect(Collectors.toList());

        for (PartitionState partitionState : partitionStatesToCopy) {
            long clickHouseRowCount = getClickHousePartitionRowCount(partitionState.getClickHousePartition());
            DealerOperation operation = new CopyOperation(partitionState, clickHouseRowCount);
            state.setCurrentOperation(operation);
            doOperation(operation);
        }
    }

    /* Update operation expects all partitions with any statuses, but SKIPPED AND YT_DELETED statuses will be ignored */
    @VisibleForTesting
    void runUpdateProcess(String clickHousePartition, Collection<PartitionState> partitionStates)
        throws InterruptedException, DealerException {

        partitionStates = partitionStates.stream()
            .filter(ps -> SHOULD_BE_UPDATED.test(ps.getStatus()))
            .collect(Collectors.toList());

        DealerOperation operation = new UpdateOperation(clickHousePartition, partitionStates);
        state.setCurrentOperation(operation);
        doOperation(operation);

        /* Once it's successful we could remove yt-deleted partitions in dealerState for mentioned CH-partition */
        state.removeYtDeletedPartitionStates(clickHousePartition);
        saveState();
    }

    @VisibleForTesting
    void doOperation(DealerOperation operation) throws InterruptedException, DealerException {
        try {
            operation.runOperation(this);
            log.info("Operation {} successfully finished.", operation);
            state.setCurrentOperation(null);
            saveState();
        } catch (InterruptedException e) {
            log.warn("Operation interrupted");
            throw e;
        } catch (Exception e) {
            log.error("Exception while running operation: {}", operation, e);
            if (operation.canBeCanceled()) {
                log.info("Canceling operation {}.", operation);
                state.setCurrentOperation(null);
                saveState();
            } else {
                log.warn("Operation cant be canceled. Will retry.");
            }
            throw new DealerException(
                "Exception while running operation: " + operation + System.lineSeparator() + e.getMessage(), e
            );
        }
    }

    @Override
    public void applyDdl() {
        List<ClickHouseTableDefinition> tables = createTableDefinitions();
        for (ClickHouseTableDefinition table : tables) {
            checkDdlState(ddlServiceForClusterNext.applyDdl(table));
        }
    }

    private void checkDdlState(TableDdlState tableDdlState) {
        Preconditions.checkState(
            tableDdlState.getGlobalStatus() != MonitoringStatus.CRITICAL,
            "Failed to apply DDL" + tableDdlState.toString()
        );
    }

    @VisibleForTesting
    List<ClickHouseTableDefinition> createTableDefinitions() {
        ClickHouseTableDefinition dataTable = createReplicatedTableDefinition(dealerConfig.getDataTable());
        ClickHouseTableDefinition distributedTable = createDistributedTableDefinition(
            dealerConfig.getDistributedTable(), dealerConfig.getDataTable()
        );
        ClickHouseTableDefinition tempDataTable = createReplicatedTableDefinition(
            dealerConfig.getTempDataTable()
        );
        ClickHouseTableDefinition tempDistributedTable = createDistributedTableDefinition(
            dealerConfig.getTempDistributedTable(), dealerConfig.getTempDataTable()
        );
        return Arrays.asList(dataTable, distributedTable, tempDataTable, tempDistributedTable);
    }

    @Override
    public void clearTempTable() throws InterruptedException {
        truncateTable(dealerConfig.getTempDataTable());
    }

    @Override
    public String startTmCopyOperation(String ytPartition) {
        TmYt2ClickHouseCopyTask copyTask = createCopyTask(ytPartition);
        return tmClient.createTask(copyTask);
    }

    @Override
    public void retryTmCopyOperation(String tmTaskId) {
        tmClient.restartTask(tmTaskId);
    }

    @Override
    public TmTaskState pollTmTask(String tmTaskId) throws InterruptedException {
        Preconditions.checkState(tmTaskId != null);
        TmTaskState taskState;
        do {
            taskState = tmClient.getTaskState(tmTaskId);
            log.info(
                "Tm task {} status {}. ClickHouse table {}",
                tmTaskId, taskState.getStatus(),
                dealerConfig.getDistributedTable().getFullName()
            );
            TimeUnit.SECONDS.sleep(TM_POLL_INTERVAL_SECONDS);
        } while (!taskState.isFinished());

        log.info(
            "Tm task {} finished with status {}. ClickHouse table {}",
            tmTaskId, taskState.getStatus(), dealerConfig.getDistributedTable().getFullName()
        );
        return taskState;
    }

    private long countTableRows(TableName tableName) {
        return countTableRows(tableName, "");
    }

    private long countTableRows(TableName tableName, String where) {
        String query = String.format("SELECT count() FROM %s %s", tableName.getFullName(), where);
        log.info("countTableRows: " + query);
        return ddlService.getSeedJdbcTemplate().queryForObject(query, Long.class);
    }

    @VisibleForTesting
    Map<String, Long> getYtPartitionCountsFromClickHouse() {
        String ytPartition = (SingletonClickHousePartitionExtractor.SINGLETON_PARTITION_BY.equals(
            dealerConfig.getMergeTree().getPartitionBy()
        )) ? String.format("'%s'", PartitionState.SINGLE_PARTITION_ID) : dealerConfig.getYtPartitionNameColumn();


        return Retrier.INSTANCE.retryOnException(() -> getYtPartitionCountFromClickHouse(ytPartition),
            CLICKHOUSE_READ_ATTEMPTS,
            Arrays.asList("Timeout exceeded while reading from socket", "Connection reset")
        );
    }

    private Map<String, Long> getYtPartitionCountFromClickHouse(String ytPartition) {
        Map<String, Long> results = new HashMap<>();
        ddlService.getSeedJdbcTemplate().query(
            String.format("SELECT partition, count() as partitionCount from %s group by %s as partition",
                dealerConfig.getDistributedTable().getFullName(),
                ytPartition),
            (rs) -> {
                results.put(rs.getString("partition"), rs.getLong("partitionCount"));
            }
        );
        return results;
    }

    @Override
    public DealerGlobalConfig getGlobalConfig() {
        return dealerConfig.getGlobalConfig();
    }

    @Override
    public void cleanError() {
        state.cleanError();
    }

    @Override
    public void saveState() {
        dealerDao.save(state);
    }

    @Override
    public long countTempTableRows() {
        return countTableRows(dealerConfig.getTempDistributedTable());
    }

    @Override
    public long countTargetTableRows(String clickHousePartition) {
        String where = (SingletonClickHousePartitionExtractor.SINGLETON_PARTITION_BY.equals(clickHousePartition)) ? ""
            : String.format("WHERE %s = %s", dealerConfig.getMergeTree().getPartitionBy(), clickHousePartition);
        return countTableRows(dealerConfig.getDistributedTable(), where);
    }

    @Override
    public void replacePartitionFromTempTable(String clickHousePartition) throws InterruptedException {
        replacePartition(dealerConfig.getTempDataTable(), dealerConfig.getDataTable(), clickHousePartition);
    }

    @Override
    public void replacePartitionFromTargetTable(String clickHousePartition) throws InterruptedException {
        replacePartition(dealerConfig.getDataTable(), dealerConfig.getTempDataTable(), clickHousePartition);
    }

    private void replacePartition(TableName fromTable, TableName toTable, String clickHousePartition)
        throws InterruptedException {

        ddlService.replacePartitionFromTable(fromTable, toTable, clickHousePartition);
        TimeUnit.SECONDS.sleep(dealerConfig.getTimeForReplicationSeconds());

    }

    @Override
    public void updatePartitionStates(List<PartitionState> partitionStates) {
        for (PartitionState partitionState : partitionStates) {
            state.putPartitionState(partitionState);
        }
        saveState();
    }

    private long getClickHousePartitionRowCount(String clickHousePartition) {
        return state.getPartitionStates(clickHousePartition).stream()
            .filter(p -> p.getStatus() == PartitionState.Status.TRANSFERRED)
            .mapToLong(p -> p.getYtState().getRowCount())
            .sum();
    }

    @VisibleForTesting
    TmYt2ClickHouseCopyTask createCopyTask(String yaPartition) {
        DealerConfig.ClickHouseCluster clickHouseCluster = dealerConfig.getClickHouseCluster();
        String ytPath = dealerConfig.getYtPath();
        if (!yaPartition.equals(PartitionState.SINGLE_PARTITION_ID)) {
            ytPath += "/" + yaPartition;
        }

        TmYt2ClickHouseCopyTask.Builder builder = TmYt2ClickHouseCopyTask.newBuilder()
            .withYtCluster(dealerConfig.getYtCluster())
            .withYtPath(ytPath)
            .withClickHouseTable(dealerConfig.getTempDataTable().getFullName())
            .withTmQueueName(dealerConfig.getTmQueueName())
            .withCredentials(clickHouseCluster.getUser(), clickHouseCluster.getPassword())
            .withShardingKey(dealerConfig.getShardingKeys())
            .withDateColumn(dealerConfig.getYtPartitionNameColumn())
            .withLabel(clickHouseCluster.getCluster())
            .withResetState(true)
            .withQuorum(getQuorum());

        if (clickHouseCluster.isDbaas()) {
            builder.withDbaasToken(clickHouseCluster.getDbaasToken());
            builder.withDbaasClusterAddress(clickHouseCluster.getCluster());
        } else {
            builder.withClickHouseCluster(clickHouseCluster.getCluster());
        }

        return builder.build();
    }

    @VisibleForTesting
    TmYt2ClickHouseCopyTask.Quorum getQuorum() {
        OptionalInt minReplicaSizePerShard = ddlService.getCluster().getServers().stream()
            .collect(Collectors.groupingBy(ClickHouseCluster.Server::getShardNumber, Collectors.toSet()))
            .values()
            .stream()
            .mapToInt(Set::size)
            .min();

        if (minReplicaSizePerShard.isPresent() && minReplicaSizePerShard.getAsInt() < MIN_NUMBER_TO_MAJORITY_QUORUM) {
            return TmYt2ClickHouseCopyTask.Quorum.AT_LEAST_ONE;
        }

        return TmYt2ClickHouseCopyTask.Quorum.MAJORITY;
    }

    private void reloadRuntimeParams() {
        DealerConfig.ClickHouseCluster clickHouseCluster = dealerConfig.getClickHouseCluster();
        String clusterId = clickHouseCluster.getCluster();

        if (clickHouseCluster.isDbaas()) {
            reloadMdbRuntimeParams(clickHouseCluster, clusterId);
        } else {
            Retrier.INSTANCE.retryOnException(() -> reloadHealthRuntimeParams(clickHouseCluster, clusterId),
                CLICKHOUSE_READ_ATTEMPTS,
                Collections.singletonList("failed to respond")
            );
        }
        setDataRotationService(new DataRotationService(ddlService));
    }

    private void reloadHealthRuntimeParams(DealerConfig.ClickHouseCluster clickHouseCluster, String clusterId) {
        SortedMap<String, DealerClusterConfig> clusterConfigs = dealerConfig.getGlobalConfig().getClusterConfigs();
        TmClickHouseClusterOptions options = tmClient.getClusterOptions(clusterId);

        runtimeOptions = generateHealthRuntimeOptions(options, options.getClusterName());
        setDdlService(createDdlService(clickHouseCluster, runtimeOptions));

        if (clusterConfigs.containsKey(clusterId)) {
            ddlServiceForClusterNext = createDdlService(
                clickHouseCluster,
                generateHealthRuntimeOptions(options, clusterConfigs.get(clusterId).getClusterForDdlApply())
            );
        } else {
            ddlServiceForClusterNext = ddlService;
        }
    }

    private void reloadMdbRuntimeParams(DealerConfig.ClickHouseCluster clickHouseCluster, String clusterId) {
        try (DealerMdbService mdbService = new DealerMdbService(
            clickHouseCluster.getDbaasToken(), dealerConfig.getMdbEndpoint()
        )) {
            runtimeOptions = new ClickHouseRuntimeParams(
                mdbService.getClickhouseHosts(clusterId),
                mdbService.getClickHouseSystemClusterName(clusterId),
                DBAAS_CLICKHOUSE_HTTPS_PORT,
                true
            );
        } catch (Exception e) {
            throw new RuntimeException(e);
        }

        setDdlService(createDdlService(clickHouseCluster, runtimeOptions));
        ddlServiceForClusterNext = ddlService;
    }


    private ClickHouseRuntimeParams generateHealthRuntimeOptions(
        TmClickHouseClusterOptions options, String clusterName) {
        return new ClickHouseRuntimeParams(options.getSeedHosts(), clusterName, options.getHttpPort(), false);
    }

    private ClickHouseDdlService createDdlService(DealerConfig.ClickHouseCluster clickHouseCluster,
                                                  ClickHouseRuntimeParams runtimeParams) {
        ClickHouseProperties properties = new ClickHouseProperties();
        properties.setUser(clickHouseCluster.getUser());
        properties.setPassword(clickHouseCluster.getPassword());
        properties.setPort(runtimeParams.getPort());
        properties.setSsl(runtimeParams.isSsl());
        properties.setSslMode("none");
        //Доступ к system есть всегда. А если указанная база не существует - мы получим ошибку и не сможем её создать.
        properties.setDatabase("system");
        properties.setSocketTimeout(secondsToMillis(dealerConfig.getClickHouseSocketTimeoutSeconds()));
        properties.setKeepAliveTimeout(secondsToMillis(dealerConfig.getClickHouseKeepAliveTimeoutSeconds()));

        return ClickHouseDdlService.create(runtimeParams.getSeedHosts(), properties, runtimeParams.getClusterName());
    }

    private int secondsToMillis(int seconds) {
        return (int) Duration.ofSeconds(seconds).toMillis();
    }

    private void processYtPartitions() {
        log.info("Checking YT partitions for path {}", dealerConfig.getYtPath());
        YPath path = YPath.simple(dealerConfig.getYtPath());
        YTreeNode yTreeNode = yt.cypress().get(path, YtAttributes.ATTRIBUTES);

        if (yTreeNode.getAttributeOrThrow(YtAttributes.TYPE).stringValue().equals(YtAttributes.TYPE_DIR)) {
            List<YTreeStringNode> nodes = yt.cypress().list(path, YtAttributes.ATTRIBUTES);
            log.info("Found {} partitions in path {}", nodes.size(), dealerConfig.getYtPath());
            DealerStateProcessor.processYtNodes(state, dealerConfig.getPartitionExtractor(), nodes, Instant.now());

        } else if (yTreeNode.getAttributeOrThrow(YtAttributes.TYPE).stringValue().equals(YtAttributes.TYPE_TABLE)) {
            log.info("Found table in path {}. Assuming it is single yt partition table.", dealerConfig.getYtPath());
            YTreeStringNode node = new YTreeStringNodeImpl(
                PartitionState.SINGLE_PARTITION_ID, yTreeNode.getAttributes()
            );
            DealerStateProcessor.processYtNodes(
                state, dealerConfig.getPartitionExtractor(), Collections.singletonList(node), Instant.now()
            );
        }
    }

    private void validateYtPartitionsFromClickHouse() {

        if (lastValidationTime != null &&
            lastValidationTime.isAfter(
                Instant.now().minus(Duration.ofSeconds(dealerConfig.getYtPartitionStateValidationPeriodSeconds()))
            )) {
            return;
        }
        lastValidationTime = Instant.now();
        log.info("Checking YT partitions transferred to clickHouse, Validation time: '{}'", lastValidationTime);

        List<PartitionState> rotatingPartitions = getPartitions(PartitionState.Status.ROTATING);
        List<PartitionState> transferredPartitions = getPartitions(PartitionState.Status.TRANSFERRED);

        if (transferredPartitions.isEmpty() && rotatingPartitions.isEmpty()) {
            log.info("No transferred or rotatiting partitions - no validation");
            return;
        }

        Map<String, Long> ytPartitionCountsFromCh = getYtPartitionCountsFromClickHouse();
        validateTransferredPartitionsFromClickHouse(ytPartitionCountsFromCh, transferredPartitions);
        validateRotatingPartitionsFromClickHouse(ytPartitionCountsFromCh, rotatingPartitions);
        log.info("Validation finished");
    }

    @VisibleForTesting
    void validateRotatingPartitionsFromClickHouse(Map<String, Long> ytPartitionCountsFromCh,
                                                  List<PartitionState> rotatingPartitions) {

        log.info("Validation: '{}' rotating YT-partitions found in dealer's state", rotatingPartitions.size());

        rotatingPartitions.stream()
            .filter(st -> !ytPartitionCountsFromCh.containsKey(st.getYtPartition()))
            .forEach(st -> {
                log.info(
                    "YT Partition '{}', (part of CH Partition '{}') has rotating status," +
                        " but it's not found in clickhouse, marking it as rotated",
                    st.getYtPartition(),
                    st.getClickHousePartition()
                );
                st.setStatus(PartitionState.Status.ROTATED);
            });

    }

    @VisibleForTesting
    void validateTransferredPartitionsFromClickHouse(Map<String, Long> ytPartitionCountsFromCh,
                                                     List<PartitionState> transferredPartitions) {

        log.info("Validation: '{}' transferred YT-partitions found in dealer's state", transferredPartitions.size());

        transferredPartitions.stream()
            .filter(st -> !ytPartitionCountsFromCh.containsKey(st.getYtPartition()) ||
                ytPartitionCountsFromCh.get(st.getYtPartition()) !=
                    st.getClickHouseState()
                        .getTransferredYtState()
                        .getRowCount()
            )
            .forEach(st -> markAsDataMismatch(ytPartitionCountsFromCh, st));
    }

    private void markAsDataMismatch(Map<String, Long> ytPartitionCountsFromCh, PartitionState st) {
        st.setStatus(PartitionState.Status.TRANSFERRED_DATA_MISMATCH);
        log.info(
            "Found {} - transferred partition with different rowCount " +
                "between dealerState - {} rows and clickHouse - {} rows",
            st.getYtPartition(), st.getYtState().getRowCount(), ytPartitionCountsFromCh.get(st.getYtPartition())
        );

        persistDataMismatch(st, ytPartitionCountsFromCh.get(st.getYtPartition()), lastValidationTime);
    }

    private List<PartitionState> getPartitions(PartitionState.Status status) {
        return state.getPartitionStates().stream()
            .filter(st -> st.getStatus() == status)
            .collect(Collectors.toList());
    }

    private void persistDataMismatch(PartitionState st, Long ytPartitionRowsInClickHouse, Instant lastValidationTime) {
        DataMismatch dataMismatch = DataMismatch.newBuilder()
            .withConfig(dealerConfig.getKey())
            .withChPartition(st.getClickHousePartition())
            .withYtPartition(st.getYtPartition())
            .withClickHouseRowCount(ytPartitionRowsInClickHouse)
            .withDealerStateRowCount(st.getClickHouseState().getTransferredYtState().getRowCount())
            .withValidationTime(lastValidationTime)
            .build();

        dealerDao.insert(dataMismatch);
    }

    private ClickHouseTableDefinition createReplicatedTableDefinition(TableName tableName) {
        return new ClickHouseTableDefinitionImpl(
            tableName, dealerConfig.getColumns(),
            dealerConfig.getMergeTree().replicated(tableName, dealerConfig.getZookeeperPrefix())
        );
    }

    private ClickHouseTableDefinition createDistributedTableDefinition(TableName distributedTable,
                                                                       TableName dataTable) {
        return new ClickHouseTableDefinitionImpl(
            distributedTable, dealerConfig.getColumns(),
            new DistributedEngine(runtimeOptions.getClusterName(), dataTable)
        );
    }

    @VisibleForTesting
    void setRuntimeOptions(ClickHouseRuntimeParams runtimeOptions) {
        this.runtimeOptions = runtimeOptions;
    }

    private void truncateTable(TableName tableName) throws InterruptedException {
        log.info("Truncating table {}", tableName.getFullName());
        int attemptNumber = 1;
        while (countTableRows(dealerConfig.getTempDistributedTable()) > 0) {
            Preconditions.checkState(
                attemptNumber <= TRUNCATE_TABLE_RETRIES,
                "Failed to truncate table. Too many attempts: %d (max %d)", attemptNumber, TRUNCATE_TABLE_RETRIES
            );
            ddlService.truncateTable(tableName);
            log.info(
                "Table {} cleared, Waiting {} seconds, to check that it is empty. Attempt number {}.",
                tableName.getFullName(), TRUNCATE_TABLE_SLEEP_SECONDS, attemptNumber
            );
            TimeUnit.SECONDS.sleep(TRUNCATE_TABLE_SLEEP_SECONDS);
            attemptNumber++;
        }
        log.info("Table {} is empty", tableName.getFullName());
    }

    @VisibleForTesting
    void setDdlService(ClickHouseDdlService ddlService) {
        this.ddlService = ddlService;
    }

    @VisibleForTesting
    void setDataRotationService(DataRotationService dataRotationService) {
        this.dataRotationService = dataRotationService;
    }

    @VisibleForTesting
    static class ClickHouseRuntimeParams {
        private final List<String> seedHosts;
        private final String clusterName;
        private final int port;
        private final boolean ssl;

        ClickHouseRuntimeParams(List<String> seedHosts, String clusterName, int port, boolean ssl) {
            this.seedHosts = new ArrayList<>(seedHosts);
            this.clusterName = clusterName;
            this.port = port;
            this.ssl = ssl;
        }

        public List<String> getSeedHosts() {
            return new ArrayList<>(seedHosts);
        }

        public String getClusterName() {
            return clusterName;
        }

        public int getPort() {
            return port;
        }

        public boolean isSsl() {
            return ssl;
        }
    }

}
