package ru.yandex.market.clickhouse.dealer.config;

import com.google.common.base.Preconditions;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import org.springframework.util.StringValueResolver;
import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnTypeBase;
import ru.yandex.market.clickhouse.ddl.ColumnTypeUtils;

import java.io.File;
import java.io.FileReader;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;

/**
 * @author Aleksei Malygin <a href="mailto:Malygin-Me@yandex-team.ru"></a>
 * Date: 2019-01-18
 */
public abstract class DealerCommonConfigParser<T> {

    private static final JsonParser JSON_PARSER = new JsonParser();
    private static final Pattern FIELD_PATTERN = Pattern.compile("\\w+\\((?<column>\\w+)[\\w, ]*\\)");

    private final StringValueResolver resolver;

    public DealerCommonConfigParser(StringValueResolver resolver) {
        this.resolver = resolver;
    }

    public T parseConfig(File file) throws Exception {
        JsonObject configObject = JSON_PARSER.parse(new FileReader(file)).getAsJsonObject();
        return parseConfig(configObject, file.getName());
    }

    public abstract T parseConfig(JsonObject configObject, String name);

    protected void checkColumn(String column, Set<String> tableColumns, String configKey, boolean allowFunctions) {
        checkColumns(Collections.singletonList(column), tableColumns, configKey, allowFunctions);
    }

    protected void checkColumns(List<String> columns, Set<String> tableColumns,
                                String configKey, boolean allowFunctions) {
        Preconditions.checkState(!columns.isEmpty(), "Config key %s is empty", configKey);
        for (String column : columns) {
            //Поиск имени колонки в функции достаточно грубый, но покрывает большинство кейсов
            Matcher matcher = FIELD_PATTERN.matcher(column);
            if (allowFunctions && matcher.matches()) {
                column = matcher.group("column");
            }
            Preconditions.checkState(
                tableColumns.contains(column),
                "Column %s from config key %s don't exists in table columns. Allowed: %s",
                column, configKey, String.join(", ", tableColumns)
            );
        }
    }

    protected List<String> parseResolvedList(JsonObject jsonObject, String key) {
        JsonElement jsonElement = jsonObject.get(key);
        Preconditions.checkState(
            jsonElement != null && (jsonElement.isJsonPrimitive() || jsonElement.isJsonArray()),
            "Config key %s should be json string or json array", key
        );
        if (jsonElement.isJsonPrimitive()) {
            return Collections.singletonList(getResolvedString(jsonObject, key));
        } else {
            JsonArray jsonArray = jsonElement.getAsJsonArray();
            return StreamSupport.stream(jsonArray.spliterator(), false)
                .map(JsonElement::getAsString)
                .map(resolver::resolveStringValue)
                .collect(Collectors.toList());
        }
    }

    protected Column parseColumn(Map.Entry<String, JsonElement> columnEntry) {
        JsonElement columnElement = columnEntry.getValue();
        if (columnElement.isJsonPrimitive()) {
            ColumnTypeBase type = ColumnTypeUtils.fromClickhouseDDL(columnElement.getAsString());
            return new Column(columnEntry.getKey(), type);
        }
        JsonObject columnObject = columnElement.getAsJsonObject();
        String defaultExpr = null;
        if (columnObject.has("defaultExpr")) {
            defaultExpr = getResolvedString(columnObject, "defaultExpr");
        }
        ColumnTypeBase type = ColumnTypeUtils.fromClickhouseDDL(columnObject.get("type").getAsString());
        return new Column(columnEntry.getKey(), type, defaultExpr);
    }

    protected int getOptionalResolvedInt(JsonObject jsonObject, String key, int defaultValue) {
        if (!jsonObject.has(key)) {
            return defaultValue;
        }
        return Integer.valueOf(getResolvedString(jsonObject, key));
    }

    protected String getOptionalResolvedString(JsonObject jsonObject, String key, String defaultValue) {
        if (!jsonObject.has(key)) {
            return defaultValue;
        }
        return getResolvedString(jsonObject, key);
    }

    protected String getResolvedString(JsonObject jsonObject, String key) {
        Preconditions.checkArgument(jsonObject.has(key), "No required parameter %s in config", key);
        String value = jsonObject.get(key).getAsString();
        return resolver.resolveStringValue(value);
    }
}
