package ru.yandex.market.clickhouse.dealer.config;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.TableName;
import ru.yandex.market.clickhouse.ddl.engine.MergeTree;
import ru.yandex.market.clickhouse.dealer.clickhouse.ClickHousePartitionExtractor;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 18/01/2018
 */
public class DealerConfig {

    private final DealerGlobalConfig globalConfig;
    private final String configName;
    private final String ytPath;
    private final String ytCluster;
    private final ClickHouseCluster clickHouseCluster;
    private final TableName distributedTable;
    private final TableName dataTable;
    private final TableName tempDistributedTable;
    private final TableName tempDataTable;

    private final List<String> shardingKeys;
    private final MergeTree mergeTree;
    private final List<Column> columns;
    private final Key key;
    private final String ytPartitionNameColumn;
    private final ClickHousePartitionExtractor partitionExtractor;
    private final String tmQueueName;
    private final int rotationPeriodDays;

    private DealerConfig(Builder builder) {
        globalConfig = builder.globalConfig;
        configName = builder.configName;
        ytPath = builder.ytPath;
        ytCluster = builder.ytCluster;
        clickHouseCluster = builder.clickHouseCluster;
        tmQueueName = (null == builder.tmQueueName) ? globalConfig.getTmQueueName() : builder.tmQueueName;
        rotationPeriodDays = (builder.rotationPeriodDays >= 0)
            ? builder.rotationPeriodDays
            : globalConfig.getDefaultRotationPeriodDays();

        shardingKeys = builder.shardingKeys;
        mergeTree = new MergeTree(
            builder.partitionBy,
            builder.orderBy,
            builder.sampleBy
        );
        columns = Collections.unmodifiableList(builder.columns);

        String database = builder.tableName.getDatabase();
        String tempDatabase = globalConfig.getTempDatabase().orElse(database);

        distributedTable = builder.tableName;
        dataTable = distributedTable.withTablePostfix(globalConfig.getDataTablePostfix());

        String tempTablePrefix = (globalConfig.getTempDatabase().isPresent()) ? database + "__" : "";
        tempDistributedTable = new TableName(
            tempDatabase,
            tempTablePrefix + builder.tableName.getTable() + globalConfig.getTempTablePostfix()
        );
        tempDataTable = new TableName(
            tempDatabase,
            tempTablePrefix + builder.tableName.getTable() + globalConfig.getTempTablePostfix()
                + globalConfig.getDataTablePostfix()
        );

        key = new Key(ytPath, clickHouseCluster.cluster, distributedTable.getFullName());
        ytPartitionNameColumn = builder.ytPartitionNameColumn;
        partitionExtractor = builder.partitionExtractor;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public String getConfigName() {
        return configName;
    }

    public int getCheckIntervalMinutes() {
        return globalConfig.getCheckIntervalMinutes();
    }

    public int getRetryOnErrorIntervalMinutes() {
        return globalConfig.getRetryOnErrorIntervalMinutes();
    }

    public int getMaxClickHousePartitionsPerIteration() {
        return globalConfig.getMaxClickHousePartitionsPerIteration();
    }

    public int getClickHouseSocketTimeoutSeconds() {
        return globalConfig.getClickHouseSocketTimeoutSeconds();
    }

    public int getClickHouseKeepAliveTimeoutSeconds() {
        return globalConfig.getClickHouseKeepAliveTimeoutSeconds();
    }

    public int getYtPartitionStateValidationPeriodSeconds() {
        return globalConfig.getYtPartitionStateValidationPeriodSeconds();
    }

    public String getYtPath() {
        return ytPath;
    }

    public String getYtCluster() {
        return ytCluster;
    }

    public List<String> getShardingKeys() {
        return shardingKeys;
    }

    public MergeTree getMergeTree() {
        return mergeTree;
    }

    public List<Column> getColumns() {
        return columns;
    }

    public String getZookeeperPrefix() {
        return globalConfig.getZookeeperPrefix();
    }

    public TableName getDistributedTable() {
        return distributedTable;
    }

    public TableName getDataTable() {
        return dataTable;
    }

    public TableName getTempDistributedTable() {
        return tempDistributedTable;
    }

    public TableName getTempDataTable() {
        return tempDataTable;
    }

    public ClickHouseCluster getClickHouseCluster() {
        return clickHouseCluster;
    }

    public Key getKey() {
        return key;
    }

    public String getYtPartitionNameColumn() {
        return ytPartitionNameColumn;
    }

    public ClickHousePartitionExtractor getPartitionExtractor() {
        return partitionExtractor;
    }

    public String getTmQueueName() {
        return tmQueueName;
    }

    public int getTimeForReplicationSeconds() {
        return globalConfig.getTimeForReplicationSeconds();
    }

    public String getMdbEndpoint() {
        return globalConfig.getMdbEndpoint();
    }

    public DealerGlobalConfig getGlobalConfig() {
        return globalConfig;
    }

    public int getRotationPeriodDays() {
        return rotationPeriodDays;
    }

    public static class Key {
        private final String ytPath;
        private final String clickHouseClusterId;
        private final String clickHouseFullTableName;

        public Key(String ytPath, String clickHouseClusterId, String clickHouseFullTableName) {
            this.ytPath = ytPath;
            this.clickHouseClusterId = clickHouseClusterId;
            this.clickHouseFullTableName = clickHouseFullTableName;
        }

        public String getYtPath() {
            return ytPath;
        }

        public String getClickHouseClusterId() {
            return clickHouseClusterId;
        }

        public String getClickHouseFullTableName() {
            return clickHouseFullTableName;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            Key key = (Key) o;
            return Objects.equals(ytPath, key.ytPath) &&
                Objects.equals(clickHouseClusterId, key.clickHouseClusterId) &&
                Objects.equals(clickHouseFullTableName, key.clickHouseFullTableName);
        }

        @Override
        public int hashCode() {
            return Objects.hash(ytPath, clickHouseClusterId, clickHouseFullTableName);
        }

        @Override
        public String toString() {
            return "DealerConfig.Key{" +
                "ytPath='" + ytPath + '\'' +
                ", clickHouseClusterId='" + clickHouseClusterId + '\'' +
                ", clickHouseFullTableName='" + clickHouseFullTableName + '\'' +
                '}';
        }
    }

    public static final class Builder {
        private DealerGlobalConfig globalConfig;
        private String configName;
        private String ytPath;
        private String ytCluster;
        private ClickHouseCluster clickHouseCluster;
        private String tmQueueName;
        private TableName tableName;
        private List<String> shardingKeys;
        private String partitionBy;
        private List<String> orderBy;
        private String sampleBy;
        private List<Column> columns;
        private String ytPartitionNameColumn;
        private ClickHousePartitionExtractor partitionExtractor;
        private int rotationPeriodDays;


        private Builder() {
        }

        public Builder withGlobalConfig(DealerGlobalConfig globalConfig) {
            this.globalConfig = globalConfig;
            return this;
        }

        public Builder withConfigName(String configName) {
            this.configName = configName;
            return this;
        }

        public Builder withYtPath(String ytPath) {
            this.ytPath = ytPath;
            return this;
        }

        public Builder withYtCluster(String ytCluster) {
            this.ytCluster = ytCluster;
            return this;
        }

        public Builder withClickHouseTmCluster(String tmCluster, String user, String password) {
            this.clickHouseCluster = new ClickHouseCluster(false, tmCluster, user, password, null);
            return this;
        }

        public Builder withClickHouseDbaasCluster(String clusterId, String user, String password, String dbaasToken) {
            this.clickHouseCluster = new ClickHouseCluster(true, clusterId, user, password, dbaasToken);
            return this;
        }

        public Builder withTmQueueName(String tmQueueName) {
            this.tmQueueName = Strings.emptyToNull(tmQueueName.trim());
            return this;
        }

        public Builder withTableName(TableName tableName) {
            this.tableName = tableName;
            return this;
        }

        public Builder withTableName(String fullTableName) {
            this.tableName = new TableName(fullTableName);
            return this;
        }

        public Builder withShardingKeys(List<String> shardingKeys) {
            this.shardingKeys = shardingKeys;
            return this;
        }

        public Builder withShardingKeys(String... shardingKeys) {
            return withShardingKeys(Arrays.asList(shardingKeys));
        }


        public Builder withPartitionBy(String partitionBy) {
            this.partitionBy = partitionBy;
            return this;
        }

        public Builder withOrderBy(List<String> orderBy) {
            this.orderBy = orderBy;
            return this;
        }

        public Builder withOrderBy(String... orderBy) {
            return withOrderBy(Arrays.asList(orderBy));
        }


        public Builder withSampleBy(String sampleBy) {
            this.sampleBy = sampleBy;
            return this;
        }

        public Builder withColumns(List<Column> columns) {
            this.columns = columns;
            return this;
        }

        public Builder withYtPartitionNameColumn(String ytPartitionNameColumn) {
            this.ytPartitionNameColumn = ytPartitionNameColumn;
            return this;
        }

        public Builder withPartitionExtractor(ClickHousePartitionExtractor partitionExtractor) {
            this.partitionExtractor = partitionExtractor;
            return this;
        }

        public Builder withRotationPeriodDays(int rotationPeriodDays) {
            this.rotationPeriodDays = rotationPeriodDays;
            return this;
        }

        public Builder withColumns(Column... columns) {
            return withColumns(Arrays.asList(columns));
        }

        public DealerConfig build() {
            return new DealerConfig(this);
        }
    }

    public static class ClickHouseCluster {
        private final boolean dbaas;
        private final String cluster;
        private final String user;
        private final String password;
        private final String dbaasToken;

        public ClickHouseCluster(boolean dbaas, String cluster, String user, String password, String dbaasToken) {
            Preconditions.checkArgument(
                !dbaas || !Strings.isNullOrEmpty(dbaasToken), "dbaasToken show be provided for DBaas database"
            );
            this.dbaas = dbaas;
            this.cluster = cluster;
            this.user = user;
            this.password = password;
            this.dbaasToken = dbaasToken;
        }

        public boolean isDbaas() {
            return dbaas;
        }

        public String getCluster() {
            return cluster;
        }

        public String getUser() {
            return user;
        }

        public String getPassword() {
            return password;
        }

        public String getDbaasToken() {
            return dbaasToken;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            ClickHouseCluster that = (ClickHouseCluster) o;
            return dbaas == that.dbaas &&
                Objects.equals(cluster, that.cluster) &&
                Objects.equals(user, that.user) &&
                Objects.equals(password, that.password) &&
                Objects.equals(dbaasToken, that.dbaasToken);
        }

        @Override
        public int hashCode() {

            return Objects.hash(dbaas, cluster, user, password, dbaasToken);
        }

        @Override
        public String toString() {
            return "ClickHouseCluster{" +
                "dbaas=" + dbaas +
                ", cluster='" + cluster + '\'' +
                ", user='" + user + '\'' +
                '}';
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        DealerConfig that = (DealerConfig) o;
        return rotationPeriodDays == that.rotationPeriodDays &&
            Objects.equals(globalConfig, that.globalConfig) &&
            Objects.equals(configName, that.configName) &&
            Objects.equals(ytPath, that.ytPath) &&
            Objects.equals(ytCluster, that.ytCluster) &&
            Objects.equals(clickHouseCluster, that.clickHouseCluster) &&
            Objects.equals(distributedTable, that.distributedTable) &&
            Objects.equals(dataTable, that.dataTable) &&
            Objects.equals(tempDistributedTable, that.tempDistributedTable) &&
            Objects.equals(tempDataTable, that.tempDataTable) &&
            Objects.equals(shardingKeys, that.shardingKeys) &&
            Objects.equals(mergeTree, that.mergeTree) &&
            Objects.equals(columns, that.columns) &&
            Objects.equals(key, that.key) &&
            Objects.equals(ytPartitionNameColumn, that.ytPartitionNameColumn) &&
            Objects.equals(partitionExtractor, that.partitionExtractor) &&
            Objects.equals(tmQueueName, that.tmQueueName);
    }

    @Override
    public int hashCode() {
        return Objects.hash(globalConfig, configName, ytPath, ytCluster, clickHouseCluster, distributedTable, dataTable,
            tempDistributedTable, tempDataTable, shardingKeys, mergeTree, columns, key, ytPartitionNameColumn,
            partitionExtractor, tmQueueName, rotationPeriodDays);
    }

    @Override
    public String toString() {
        return "DealerConfig{" +
            "globalConfig=" + globalConfig +
            ", configName='" + configName + '\'' +
            ", ytPath='" + ytPath + '\'' +
            ", ytCluster='" + ytCluster + '\'' +
            ", clickHouseCluster=" + clickHouseCluster +
            ", distributedTable=" + distributedTable +
            ", dataTable=" + dataTable +
            ", tempDistributedTable=" + tempDistributedTable +
            ", tempDataTable=" + tempDataTable +
            ", shardingKeys=" + shardingKeys +
            ", mergeTree=" + mergeTree +
            ", columns=" + columns +
            ", key=" + key +
            ", ytPartitionNameColumn='" + ytPartitionNameColumn + '\'' +
            ", partitionExtractor=" + partitionExtractor +
            ", tmQueueName='" + tmQueueName + '\'' +
            ", rotationPeriodDays=" + rotationPeriodDays +
            '}';
    }
}
