package ru.yandex.market.clickhouse.dealer.config;

import com.google.common.base.CharMatcher;
import com.google.gson.JsonObject;
import org.springframework.util.StringValueResolver;
import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.dealer.clickhouse.ClickHousePartitionExtractor;
import ru.yandex.market.clickhouse.dealer.clickhouse.ClickHousePartitionParser;

import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 18/01/2018
 */
public class DealerConfigParser extends DealerCommonConfigParser<DealerConfig> {

    private static final String EMPTY_MDB_TOKEN = "EMPTY_MDB_TOKEN";
    private final DealerGlobalConfig globalConfig;

    public DealerConfigParser(StringValueResolver resolver, DealerGlobalConfig globalConfig) {
        super(resolver);
        this.globalConfig = globalConfig;
    }

    public DealerGlobalConfig getGlobalConfig() {
        return globalConfig;
    }

    public DealerConfig parseConfig(JsonObject configObject, String name) {
        JsonObject columnsObject = configObject.getAsJsonObject("columns");
        List<Column> columns = columnsObject.entrySet()
            .stream()
            .map(this::parseColumn)
            .collect(Collectors.toList());

        Set<String> columnNames = columns.stream().map(Column::getName).collect(Collectors.toSet());

        List<String> orderBy = parseResolvedList(configObject, "orderBy");
        checkColumns(orderBy, columnNames, "orderBy", true);

        List<String> shardingKey = parseResolvedList(configObject, "shardingKey");
        checkColumns(shardingKey, columnNames, "shardingKey", false);


        String fullTableName = getResolvedString(configObject, "clickHouseTable");


        String ytPartitionNameColumn = getOptionalResolvedString(configObject, "ytPartitionNameColumn", null);
        if (ytPartitionNameColumn != null) {
            checkColumn(ytPartitionNameColumn, columnNames, "ytPartitionNameColumn", false);
        }

        String partitionBy = getResolvedString(configObject, "partitionBy");

        String sampleBy = getOptionalResolvedString(configObject, "sampleBy", null);
        if (sampleBy != null) {
            checkColumn(sampleBy, columnNames, "sampleBy", true);
        }

        ClickHousePartitionExtractor partitionExtractor = ClickHousePartitionParser.parsePartitionSystem(
            partitionBy, ytPartitionNameColumn
        );

        int rotationPeriodDays = getOptionalResolvedInt(configObject, "rotationPeriodDays", -1);

        DealerConfig.Builder configBuilder = DealerConfig.newBuilder()
            .withGlobalConfig(globalConfig)
            .withConfigName(name)
            .withYtCluster(getResolvedString(configObject, "ytCluster"))
            .withYtPath(CharMatcher.is('/').trimTrailingFrom(getResolvedString(configObject, "ytPath")))
            .withTableName(fullTableName)
            .withPartitionBy(partitionBy)
            .withOrderBy(orderBy)
            .withColumns(columns)
            .withShardingKeys(shardingKey)
            .withSampleBy(sampleBy)
            .withYtPartitionNameColumn(ytPartitionNameColumn)
            .withPartitionExtractor(partitionExtractor)
            .withRotationPeriodDays(rotationPeriodDays);

        String clickHouseCluster = getResolvedString(configObject, "clickHouseTmCluster");
        String clickHouseUser = getResolvedString(configObject, "clickHouseUser");
        String clickHousePassword = getResolvedString(configObject, "clickHousePassword");

        String dbaasToken = getOptionalResolvedString(configObject, "dbaasToken", EMPTY_MDB_TOKEN);

        if (EMPTY_MDB_TOKEN.equals(dbaasToken)) {
            configBuilder.withClickHouseTmCluster(clickHouseCluster, clickHouseUser, clickHousePassword);
        } else {
            configBuilder.withClickHouseDbaasCluster(clickHouseCluster, clickHouseUser, clickHousePassword, dbaasToken);
        }
        return configBuilder.build();
    }
}
