package ru.yandex.market.clickhouse.dealer.config;

import com.google.common.base.Preconditions;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import ru.yandex.market.application.monitoring.ComplicatedMonitoring;
import ru.yandex.market.application.monitoring.MonitoringUnit;

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 26/04/2018
 */
public class DealerConfigurationService {

    private static final Logger log = LogManager.getLogger();

    private final List<DealerConfig> configs;
    private final int threadCount;
    private final DealerGlobalConfig globalConfig;

    public DealerConfigurationService(DealerConfigParser configParser, String configDir,
                                      Set<String> whitelistConfigs, ComplicatedMonitoring monitoring, int threadCount) {
        this.configs = Collections.unmodifiableList(
            parseConfigs(
                configDir, configParser, whitelistConfigs, monitoring.createUnit("config")
            )
        );
        this.threadCount = threadCount;
        this.globalConfig = configParser.getGlobalConfig();
    }

    public DealerGlobalConfig getGlobalConfig() {
        return globalConfig;
    }

    public List<DealerConfig> getConfigs() {
        return configs;
    }

    public static <T> List<T> parseConfigs(String configDirName, DealerCommonConfigParser<T> configParser,
                                           Set<String> whitelistConfigs, MonitoringUnit configMonitoringUnit) {
        List<File> configFiles;
        try {
            configFiles = getConfigFiles(configDirName);
        } catch (Exception e) {
            log.error("Failed to read configs.", e);
            configMonitoringUnit.critical("Failed to read configs. Application does not work.", e);
            return Collections.emptyList();
        }

        List<T> configs = new ArrayList<>();
        List<File> failedConfigs = new ArrayList<>();

        if (!whitelistConfigs.isEmpty()) {
            log.info("Whitelist configs (others will be skipped): {}", whitelistConfigs);
        }


        for (File configFile : configFiles) {
            if (!whitelistConfigs.isEmpty() && !whitelistConfigs.contains(configFile.getName())) {
                log.info("Config {} not in whitelist. Skipping...", configFile.getName());
                continue;
            }
            try {
                configs.add(configParser.parseConfig(configFile));
                log.info("Config {} parsed", configFile.getName());
            } catch (Exception e) {
                log.error("Exception while parsing config {}.", configFile.getName(), e);
                failedConfigs.add(configFile);
            }
        }

        if (!failedConfigs.isEmpty()) {
            configMonitoringUnit.critical(
                "Failed to parse some configs (see logs for details): " +
                    failedConfigs.stream().map(File::getName).collect(Collectors.joining(", "))
            );
        }
        Preconditions.checkState(!configs.isEmpty(), "No configs found for dealer");
        return configs;
    }

    private static List<File> getConfigFiles(String configDirName) {
        log.info("Parsing configs in dir {}", configDirName);
        File configDir = new File(configDirName);
        Preconditions.checkState(configDir.exists(), "Config dir doesn't exist %s", configDirName);
        Preconditions.checkState(configDir.isDirectory(), "Config dir is not a directory %s", configDirName);
        File[] configFiles = configDir.listFiles((dir, name) -> name.toLowerCase().endsWith(".json"));
        Preconditions.checkState(configFiles != null, "Failed to read config dir %s", configDirName);
        Preconditions.checkState(configFiles.length > 0, "No configs found in dir %s", configDirName);
        log.info(
            "Found {} configs: {}",
            configFiles.length,
            Arrays.stream(configFiles).map(File::getName).collect(Collectors.joining(", "))
        );
        return Arrays.asList(configFiles);
    }

    public int getThreadCount() {
        return threadCount;
    }
}
