package ru.yandex.market.clickhouse.dealer.config;

import com.google.common.base.Strings;

import java.util.Collections;
import java.util.Objects;
import java.util.Optional;
import java.util.SortedMap;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 09/06/2018
 */
public class DealerGlobalConfig {
    private final String tempDatabase;
    private final String tmQueueName;
    private final SortedMap<String, DealerClusterConfig> clusterConfigs;
    private final int defaultRotationPeriodDays;

    private final String dataTablePostfix = "_lr";
    private final String tempTablePostfix = "_tmp";
    private final String zookeeperPrefix = "/clickhouse/tables/{shard}";
    private final String mdbEndpoint = "gw.db.yandex-team.ru";
    private final int checkIntervalMinutes = 15; //TODO move to config
    private final int retryOnErrorIntervalMinutes = 5; //TODO move to config
    private final int maxClickHousePartitionsPerIteration = 3;
    private final int clickHouseSocketTimeoutSeconds = 10 * 60;
    private final int clickHouseKeepAliveTimeoutSeconds = 60;
    private final int ytPartitionStateValidationPeriodSeconds = 2 * 60 * 60;
    private final int dataMismatchMonitoringPeriodHours = 24;
    private final int dataMismatchesForCritical = 2;
    private final int dataMismatchesForWarning = 0;
    private final int timeForReplicationSeconds = 5;
    private final int rowCountValidationSleepBeforeRetriesSeconds = 30;
    private final int rowCountValidationAttempts = 10;
    private final int theNumberOfConfigErrorsForWarning = 5;

    private DealerGlobalConfig(Builder builder) {
        this.tempDatabase = Strings.emptyToNull(builder.tempDatabase);
        this.tmQueueName = builder.tmQueueName;
        this.clusterConfigs = builder.clusterConfigs;
        this.defaultRotationPeriodDays = builder.defaultRotationPeriodDays;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public int getCheckIntervalMinutes() {
        return checkIntervalMinutes;
    }

    public int getRetryOnErrorIntervalMinutes() {
        return retryOnErrorIntervalMinutes;
    }

    public Optional<String> getTempDatabase() {
        return Optional.ofNullable(tempDatabase);
    }

    public String getTmQueueName() {
        return tmQueueName;
    }

    public String getDataTablePostfix() {
        return dataTablePostfix;
    }

    public String getTempTablePostfix() {
        return tempTablePostfix;
    }

    public String getZookeeperPrefix() {
        return zookeeperPrefix;
    }

    public int getMaxClickHousePartitionsPerIteration() {
        return maxClickHousePartitionsPerIteration;
    }

    public int getClickHouseSocketTimeoutSeconds() {
        return clickHouseSocketTimeoutSeconds;
    }

    public int getClickHouseKeepAliveTimeoutSeconds() {
        return clickHouseKeepAliveTimeoutSeconds;
    }

    public int getYtPartitionStateValidationPeriodSeconds() {
        return ytPartitionStateValidationPeriodSeconds;
    }

    public int getDataMismatchMonitoringPeriodHours() {
        return dataMismatchMonitoringPeriodHours;
    }

    public int getDataMismatchesForCritical() {
        return dataMismatchesForCritical;
    }

    public int getDataMismatchesForWarning() {
        return dataMismatchesForWarning;
    }

    public int getTimeForReplicationSeconds() {
        return timeForReplicationSeconds;
    }

    public String getMdbEndpoint() {
        return mdbEndpoint;
    }

    public int getRowCountValidationSleepBeforeRetriesSeconds() {
        return rowCountValidationSleepBeforeRetriesSeconds;
    }

    public int getRowCountValidationAttempts() {
        return rowCountValidationAttempts;
    }

    public SortedMap<String, DealerClusterConfig> getClusterConfigs() {
        return clusterConfigs;
    }

    public int getDefaultRotationPeriodDays() {
        return defaultRotationPeriodDays;
    }

    public static final class Builder {
        private String tempDatabase;
        private String tmQueueName;
        private SortedMap<String, DealerClusterConfig> clusterConfigs = Collections.emptySortedMap();
        private int defaultRotationPeriodDays;


        private Builder() {
        }

        public Builder withTempDatabase(String tempDatabase) {
            this.tempDatabase = tempDatabase;
            return this;
        }

        public Builder withTmQueueName(String tmQueueName) {
            this.tmQueueName = tmQueueName;
            return this;
        }

        public Builder withClusterConfigs(SortedMap<String, DealerClusterConfig> clusterConfigs) {
            this.clusterConfigs = clusterConfigs;
            return this;
        }

        public Builder withDefaultRotationPeriodDays(int defaultRotationPeriodDays) {
            this.defaultRotationPeriodDays = defaultRotationPeriodDays;
            return this;
        }

        public DealerGlobalConfig build() {
            return new DealerGlobalConfig(this);
        }
    }

    public int getTheNumberOfConfigErrorsForWarning() {
        return theNumberOfConfigErrorsForWarning;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        DealerGlobalConfig that = (DealerGlobalConfig) o;
        return defaultRotationPeriodDays == that.defaultRotationPeriodDays &&
            checkIntervalMinutes == that.checkIntervalMinutes &&
            retryOnErrorIntervalMinutes == that.retryOnErrorIntervalMinutes &&
            maxClickHousePartitionsPerIteration == that.maxClickHousePartitionsPerIteration &&
            clickHouseSocketTimeoutSeconds == that.clickHouseSocketTimeoutSeconds &&
            clickHouseKeepAliveTimeoutSeconds == that.clickHouseKeepAliveTimeoutSeconds &&
            ytPartitionStateValidationPeriodSeconds == that.ytPartitionStateValidationPeriodSeconds &&
            dataMismatchMonitoringPeriodHours == that.dataMismatchMonitoringPeriodHours &&
            dataMismatchesForCritical == that.dataMismatchesForCritical &&
            dataMismatchesForWarning == that.dataMismatchesForWarning &&
            timeForReplicationSeconds == that.timeForReplicationSeconds &&
            rowCountValidationSleepBeforeRetriesSeconds == that.rowCountValidationSleepBeforeRetriesSeconds &&
            rowCountValidationAttempts == that.rowCountValidationAttempts &&
            theNumberOfConfigErrorsForWarning == that.theNumberOfConfigErrorsForWarning &&
            Objects.equals(tempDatabase, that.tempDatabase) &&
            Objects.equals(tmQueueName, that.tmQueueName) &&
            Objects.equals(clusterConfigs, that.clusterConfigs) &&
            Objects.equals(dataTablePostfix, that.dataTablePostfix) &&
            Objects.equals(tempTablePostfix, that.tempTablePostfix) &&
            Objects.equals(zookeeperPrefix, that.zookeeperPrefix) &&
            Objects.equals(mdbEndpoint, that.mdbEndpoint);
    }

    @Override
    public int hashCode() {
        return Objects.hash(tempDatabase, tmQueueName, clusterConfigs, defaultRotationPeriodDays, dataTablePostfix,
            tempTablePostfix, zookeeperPrefix, mdbEndpoint, checkIntervalMinutes, retryOnErrorIntervalMinutes,
            maxClickHousePartitionsPerIteration, clickHouseSocketTimeoutSeconds, clickHouseKeepAliveTimeoutSeconds,
            ytPartitionStateValidationPeriodSeconds, dataMismatchMonitoringPeriodHours, dataMismatchesForCritical,
            dataMismatchesForWarning, timeForReplicationSeconds, rowCountValidationSleepBeforeRetriesSeconds,
            rowCountValidationAttempts, theNumberOfConfigErrorsForWarning);
    }

    @Override
    public String toString() {
        return "DealerGlobalConfig{" +
            "tempDatabase='" + tempDatabase + '\'' +
            ", tmQueueName='" + tmQueueName + '\'' +
            ", clusterConfigs=" + clusterConfigs +
            ", defaultRotationPeriodDays=" + defaultRotationPeriodDays +
            ", dataTablePostfix='" + dataTablePostfix + '\'' +
            ", tempTablePostfix='" + tempTablePostfix + '\'' +
            ", zookeeperPrefix='" + zookeeperPrefix + '\'' +
            ", mdbEndpoint='" + mdbEndpoint + '\'' +
            ", checkIntervalMinutes=" + checkIntervalMinutes +
            ", retryOnErrorIntervalMinutes=" + retryOnErrorIntervalMinutes +
            ", maxClickHousePartitionsPerIteration=" + maxClickHousePartitionsPerIteration +
            ", clickHouseSocketTimeoutSeconds=" + clickHouseSocketTimeoutSeconds +
            ", clickHouseKeepAliveTimeoutSeconds=" + clickHouseKeepAliveTimeoutSeconds +
            ", ytPartitionStateValidationPeriodSeconds=" + ytPartitionStateValidationPeriodSeconds +
            ", dataMismatchMonitoringPeriodHours=" + dataMismatchMonitoringPeriodHours +
            ", dataMismatchesForCritical=" + dataMismatchesForCritical +
            ", dataMismatchesForWarning=" + dataMismatchesForWarning +
            ", timeForReplicationSeconds=" + timeForReplicationSeconds +
            ", rowCountValidationSleepBeforeRetriesSeconds=" + rowCountValidationSleepBeforeRetriesSeconds +
            ", rowCountValidationAttempts=" + rowCountValidationAttempts +
            ", theNumberOfConfigErrorsForWarning=" + theNumberOfConfigErrorsForWarning +
            '}';
    }
}
