package ru.yandex.market.clickhouse.dealer.operation;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.data.annotation.Transient;
import ru.yandex.market.clickhouse.dealer.tm.TmTaskState;

import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

/**
 * @author Aleksei Malygin <a href="mailto:Malygin-Me@yandex-team.ru"></a>
 * Date: 05.07.18
 */
public abstract class AbstractTransferOperation implements DealerOperation {

    protected static final Logger log = LogManager.getLogger();
    protected static final int TM_RETRIES = 3;
    protected static final int TM_SLEEP_BEFORE_RETRY_SECONDS = 120;

    protected Step step = Step.PREPARE;

    protected String tmTaskId;
    private int tmAttemptNumber = 1;
    protected TmTaskState tmTaskState;
    protected String clickHousePartition;
    @Transient
    protected OperationContext context;

    @Override
    public boolean canBeCanceled() {
        return step == Step.PREPARE || step == Step.READY_TO_COPY || step == Step.TM_RUNNING || step == Step.VALIDATION;
    }

    protected abstract void startTmCopy(OperationContext context);

    protected void prepare(OperationContext context) throws InterruptedException {
        context.applyDdl();
        context.clearTempTable();
        updateStep(context, Step.READY_TO_COPY);
    }

    protected void pollCopyOperation(OperationContext context) throws InterruptedException {
        Preconditions.checkNotNull(tmTaskId);
        tmTaskState = context.pollTmTask(tmTaskId);

        if (!tmTaskState.isSuccess()) {
            retryCopy(context, tmTaskState);
            return;
        }
        tmAttemptNumber = 1;
        updateStep(context, Step.VALIDATION);
    }

    protected void retryCopy(OperationContext context, TmTaskState taskState) throws InterruptedException {
        tmAttemptNumber++;
        Preconditions.checkState(
            tmAttemptNumber <= TM_RETRIES,
            "Copy task %s failed. Too many attempts: %s (max %s). Error: {}",
            tmTaskId, tmAttemptNumber, TM_RETRIES, taskState.getError()
        );
        log.info("Waiting {} seconds before retry.", TM_SLEEP_BEFORE_RETRY_SECONDS);
        TimeUnit.SECONDS.sleep(TM_SLEEP_BEFORE_RETRY_SECONDS);
        log.warn("Retrying tm task {}. Attempt number {}.", tmTaskId, tmAttemptNumber);
        context.retryTmCopyOperation(tmTaskId);
        pollCopyOperation(context);
    }

    @VisibleForTesting
    Step getStep() {
        return step;
    }

    protected void updateStep(OperationContext context, Step newStep) {
        log.info("Step changed from {} to {} for operation: {}", step, newStep, this);
        step = newStep;
        context.saveState();
    }

    protected void validateReplacePartition(Supplier<Long> function, long requiredClickHousePartitionRows)
        throws InterruptedException {

        validateRowCount(
            function,
            requiredClickHousePartitionRows,
            "Replaced clickHousePartition = " + clickHousePartition + ", " +
                "Expected clickHousePartition rows = " + requiredClickHousePartitionRows + ", " +
                "Actual clickHousePartition rows %s."
        );
    }

    /**
     * @param function      - function to get an actual value
     * @param expectedCount - an expected value to compare to the actual value
     * @param errorMessage  - error message is using in IllegalStateException, '%s' is required to set the actual value
     */
    protected void validateRowCount(Supplier<Long> function, long expectedCount, String errorMessage)
        throws InterruptedException {

        int attempts = context.getGlobalConfig().getRowCountValidationAttempts();

        while (!doCountsEqual(function, expectedCount, errorMessage, attempts)) {
            --attempts;
        }
    }

    private boolean doCountsEqual(Supplier<Long> function, long expectedCount, String errorMessage, int attemptNumber)
        throws InterruptedException {

        log.info("validateRowCount: attempt = " + attemptNumber);
        long actualCount = function.get();

        if (actualCount == expectedCount) {
            return true;
        }

        if (attemptNumber == 0) {
            throw new IllegalStateException(String.format(errorMessage, actualCount));
        }

        TimeUnit.SECONDS.sleep(context.getGlobalConfig().getRowCountValidationSleepBeforeRetriesSeconds());
        return false;
    }

    @VisibleForTesting
    enum Step {
        PREPARE,
        REPLACE_TEMP_PARTITION,
        READY_TO_COPY,
        TM_RUNNING,
        VALIDATION,
        REPLACE_TARGET_PARTITION,
        DONE
    }
}
