package ru.yandex.market.clickhouse.dealer.operation;

import ru.yandex.market.clickhouse.dealer.state.PartitionClickHouseState;
import ru.yandex.market.clickhouse.dealer.state.PartitionState;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 04/06/2018
 */
public class CopyOperation extends AbstractTransferOperation {

    private final PartitionState partition;
    private long transferredClickHouseRowCount;

    public CopyOperation(PartitionState partition, long transferredClickHouseRowCount) {
        this.partition = partition;
        this.transferredClickHouseRowCount = transferredClickHouseRowCount;
        this.clickHousePartition = partition.getClickHousePartition();
    }

    @Override
    public void runOperation(OperationContext context) throws InterruptedException {
        super.context = context;
        //No breaks by design
        switch (step) {
            case PREPARE:
                prepare(context);
            case REPLACE_TEMP_PARTITION:
                replaceTempPartition(context);
            case READY_TO_COPY:
                startTmCopy(context);
            case TM_RUNNING:
                pollCopyOperation(context);
            case VALIDATION:
                validateData(context);
            case REPLACE_TARGET_PARTITION:
                replaceTargetPartition(context);
            case DONE:
                context.cleanError();
                return;
            default:
                throw new IllegalStateException();
        }
    }

    @Override
    protected void prepare(OperationContext context) throws InterruptedException {
        context.applyDdl();
        context.clearTempTable();
        updateStep(context, Step.REPLACE_TEMP_PARTITION);
    }

    private void replaceTempPartition(OperationContext context) throws InterruptedException {
        context.replacePartitionFromTargetTable(partition.getClickHousePartition());
        validateReplacePartition(context::countTempTableRows, transferredClickHouseRowCount);
        updateStep(context, Step.READY_TO_COPY);
    }

    @Override
    protected void startTmCopy(OperationContext context) {
        tmTaskId = context.startTmCopyOperation(partition.getYtPartition());
        updateStep(context, Step.TM_RUNNING);
    }

    private void validateData(OperationContext context) throws InterruptedException {
        long ytRowCount = partition.getYtState().getRowCount();

        validateRowCount(
            context::countTempTableRows,
            ytRowCount + transferredClickHouseRowCount,
            "The actual number of rows in ClickHouse temp table %s " +
                "and the expected number of YP partition's rows "
                + (ytRowCount + transferredClickHouseRowCount) + " is different. " +
                "Replaced yt partition '" + partition.getYtPartition() + "', Affected clickHouse partition '"
                + partition.getClickHousePartition() + "'"
        );

        updateStep(context, Step.REPLACE_TARGET_PARTITION);
    }

    private void replaceTargetPartition(OperationContext context) throws InterruptedException {
        context.replacePartitionFromTempTable(partition.getClickHousePartition());
        validateReplacePartition(() -> context.countTargetTableRows(clickHousePartition),
            transferredClickHouseRowCount + partition.getYtState().getRowCount());
        finishOperation(context);
    }

    private void finishOperation(OperationContext context) {
        partition.setStatus(PartitionState.Status.TRANSFERRED);
        partition.increaseTransferCount();
        partition.setClickHouseState(
            new PartitionClickHouseState(
                partition.getYtState(), tmTaskState.getFinishTime(), tmTaskState.getFinishTime()
            )
        );
        context.updatePartitionState(partition);
        updateStep(context, Step.DONE);
    }

    @Override
    public String toString() {
        return "CopyOperation{" +
            "partition=" + partition +
            ", transferredClickHouseRowCount=" + transferredClickHouseRowCount +
            ", step=" + step +
            ", tmTaskId='" + tmTaskId + '\'' +
            ", tmTaskState=" + tmTaskState +
            ", clickHousePartition='" + clickHousePartition + '\'' +
            '}';
    }
}
