package ru.yandex.market.clickhouse.dealer.operation;

import com.google.common.base.Preconditions;
import ru.yandex.market.clickhouse.dealer.state.PartitionClickHouseState;
import ru.yandex.market.clickhouse.dealer.state.PartitionState;
import ru.yandex.market.clickhouse.dealer.tm.TmTaskState;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;


/**
 * @author Aleksei Malygin <a href="mailto:Malygin-Me@yandex-team.ru"></a>
 * Date: 05.07.18
 */
public class UpdateOperation extends AbstractTransferOperation implements DealerOperation {
    private final List<PartitionState> partitionStates;
    private final List<TmTaskState> tmTaskStates;

    private int partitionIndex = 0;
    private long processedPartitionRows = 0;
    private PartitionState partition;

    public UpdateOperation(String clickHousePartition, Collection<PartitionState> partitionStates) {
        this.clickHousePartition = clickHousePartition;
        this.partitionStates = new ArrayList<>(partitionStates);
        tmTaskStates = new ArrayList<>();

        if (this.partitionStates.isEmpty()) {
            step = Step.DONE;
        }
    }

    @Override
    public void runOperation(OperationContext context) throws InterruptedException {
        super.context = context;
        //No breaks by design
        while (true) {
            switch (step) {
                case PREPARE:
                    prepare(context);
                case READY_TO_COPY:
                    startTmCopy(context);
                case TM_RUNNING:
                    pollCopyOperation(context);
                case VALIDATION:
                    validateData(context);

                    /* Copy until all (yt) partitions have been copied */
                    if (step == Step.READY_TO_COPY) {
                        continue;
                    }
                case REPLACE_TARGET_PARTITION:
                    replacePartition(context);
                case DONE:
                    context.cleanError();
                    return;
                default:
                    throw new IllegalStateException();
            }
        }
    }

    private PartitionState getNextPartition() {
        return hasNextPartition() ? partitionStates.get(partitionIndex++) : null;
    }

    private boolean hasNextPartition() {
        return partitionIndex < partitionStates.size();
    }

    @Override
    protected void startTmCopy(OperationContext context) {
        partition = getNextPartition();
        Preconditions.checkNotNull(partition);
        tmTaskId = context.startTmCopyOperation(partition.getYtPartition());
        updateStep(context, Step.TM_RUNNING);
    }

    private void validateData(OperationContext context) throws InterruptedException {
        tmTaskStates.add(tmTaskState);
        processedPartitionRows += partition.getYtState().getRowCount();

        validateRowCount(
            context::countTempTableRows,
            processedPartitionRows,
            "The actual number of rows in ClickHouse temp table %s " +
                "and the expected number of YP partition's rows " + processedPartitionRows + " is different." +
                System.lineSeparator() + "Last copied partitions: " + partitionStates.subList(0, partitionIndex)
        );

        updateStep(context, hasNextPartition() ? Step.READY_TO_COPY : Step.REPLACE_TARGET_PARTITION);
    }

    private void replacePartition(OperationContext context) throws InterruptedException {
        context.replacePartitionFromTempTable(clickHousePartition);
        validateReplacePartition(() -> context.countTargetTableRows(clickHousePartition), processedPartitionRows);
        finishOperation(context);
    }

    private void finishOperation(OperationContext context) {
        PartitionState partitionState;
        TmTaskState tmState;

        for (int i = 0; i < partitionStates.size(); i++) {
            partitionState = partitionStates.get(i);
            tmState = tmTaskStates.get(i);

            partitionState.setStatus(PartitionState.Status.TRANSFERRED);
            partitionState.increaseTransferCount();
            partitionState.setClickHouseState(
                new PartitionClickHouseState(
                    partitionState.getYtState(), tmState.getFinishTime(), tmState.getFinishTime()
                )
            );
        }

        context.updatePartitionStates(partitionStates);
        updateStep(context, Step.DONE);
    }

    @Override
    public String toString() {
        return "UpdateOperation{" +
            "partitionStates=" + partitionStates +
            ", tmTaskStates=" + tmTaskStates +
            ", partitionIndex=" + partitionIndex +
            ", processedPartitionRows=" + processedPartitionRows +
            ", partition=" + partition +
            ", step=" + step +
            ", tmTaskId='" + tmTaskId + '\'' +
            ", tmTaskState=" + tmTaskState +
            ", clickHousePartition='" + clickHousePartition + '\'' +
            '}';
    }
}
