package ru.yandex.market.clickhouse.dealer.spring;

import com.google.common.base.Splitter;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.beans.factory.config.ConfigurableBeanFactory;
import org.springframework.beans.factory.config.EmbeddedValueResolver;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import ru.yandex.market.application.monitoring.ComplicatedMonitoring;
import ru.yandex.market.clickhouse.dealer.config.DealerClusterConfig;
import ru.yandex.market.clickhouse.dealer.config.DealerClusterConfigParser;
import ru.yandex.market.clickhouse.dealer.config.DealerConfigParser;
import ru.yandex.market.clickhouse.dealer.config.DealerConfigurationService;
import ru.yandex.market.clickhouse.dealer.config.DealerGlobalConfig;

import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author Alexander Kedrik <a href="mailto:alkedr@yandex-team.ru"></a>
 * @date 09.07.2019
 */
@Configuration
public class DealerConfigurationSpringConfig {

    @Bean
    public DealerGlobalConfig globalConfig(
        DealerClusterConfigParser clusterConfigParser,
        ComplicatedMonitoring monitoring,
        @Value("${dealer.clusters.config.dir}") String clusterConfigDir,
        @Value("${dealer.clickhouse.temp-database:}") String tempDatabase,
        @Value("${dealer.tm.queue-name:}") String tmQueueName,
        @Value("${dealer.config.defaultRotationPeriodDays:-1}") int defaultRotationPeriodDays
    ) {

        SortedMap<String, DealerClusterConfig> clusterConfigs = DealerConfigurationService.parseConfigs(
            clusterConfigDir, clusterConfigParser, Collections.emptySet(), monitoring.createUnit("clusterConfig")
        ).stream()
            .flatMap(Collection::stream)
            .collect(
                Collectors.toMap(
                    DealerClusterConfig::getClusterId,
                    Function.identity(),
                    (v1, v2) -> {
                        throw new RuntimeException(String.format("Duplicate key for values %s and %s", v1, v2));
                    },
                    TreeMap::new
                )
            );

        return DealerGlobalConfig.newBuilder()
            .withTempDatabase(tempDatabase)
            .withTmQueueName(tmQueueName)
            .withClusterConfigs(clusterConfigs)
            .withDefaultRotationPeriodDays(defaultRotationPeriodDays)
            .build();
    }


    @Bean
    public DealerConfigParser configParser(ConfigurableBeanFactory beanFactory, DealerGlobalConfig globalConfig) {
        return new DealerConfigParser(new EmbeddedValueResolver(beanFactory), globalConfig);
    }

    @Bean
    public DealerClusterConfigParser clusterConfigParser(ConfigurableBeanFactory beanFactory) {
        return new DealerClusterConfigParser(new EmbeddedValueResolver(beanFactory));
    }

    @Bean
    public DealerConfigurationService configurationService(
        DealerConfigParser configParser,
        ComplicatedMonitoring monitoring,
        @Value("${dealer.config.dir}") String configDir,
        @Value("${dealer.config.whitelist:}") String whitelist,
        @Value("${dealer.thread-count:100}") int threadCount
    ) {
        Set<String> whitelistConfigs = new HashSet<>(
            Splitter.on(',').trimResults().omitEmptyStrings().splitToList(whitelist)
        );
        return new DealerConfigurationService(configParser, configDir, whitelistConfigs, monitoring, threadCount);
    }
}
