package ru.yandex.market.clickhouse.dealer.spring;

import com.google.common.base.CharMatcher;
import com.google.common.base.Strings;
import com.mongodb.MongoClient;
import com.mongodb.MongoClientOptions;
import com.mongodb.MongoClientURI;
import com.mongodb.ReadPreference;
import com.mongodb.WriteConcern;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.CuratorFrameworkFactory;
import org.apache.curator.retry.RetryForever;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.data.mongodb.MongoDbFactory;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.SimpleMongoDbFactory;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.impl.YtUtils;
import ru.yandex.market.application.MarketApplicationCommonConfig;
import ru.yandex.market.application.monitoring.ComplicatedMonitoring;
import ru.yandex.market.clickhouse.dealer.DealerService;
import ru.yandex.market.clickhouse.dealer.config.DealerConfigurationService;
import ru.yandex.market.clickhouse.dealer.state.DealerDao;
import ru.yandex.market.clickhouse.dealer.tm.TransferManagerClient;
import ru.yandex.market.request.netty.HttpClientConfig;
import ru.yandex.market.request.netty.NettyHttpClientContext;
import ru.yandex.market.request.netty.retry.RetryIdempotentWithSleepPolicy;
import ru.yandex.market.request.trace.Module;

import java.util.concurrent.TimeUnit;


/**
 * @author Alexander Kedrik <a href="mailto:alkedr@yandex-team.ru"></a>
 * @date 09.07.2019
 */
@Configuration
@Import(DealerConfigurationSpringConfig.class)
public class DealerSpringConfig extends MarketApplicationCommonConfig {

    public DealerSpringConfig() {
        super(Module.CLICKHOUSE_DEALER);
    }

    @Bean
    public MongoTemplate mongoTemplate(@Value("${dealer.mongo.url}") String mongoUrl,
                                       @Value("${dealer.mongo.database}") String mongoDataBase,
                                       @Value("${dealer.mongo.ssl:false}") boolean ssl,
                                       @Value("${dealer.mongo.replica-set:}") String replicaSet) {
        MongoClientOptions.Builder options = MongoClientOptions.builder()
            .writeConcern(WriteConcern.MAJORITY)
            .readPreference(ReadPreference.primary())
            .sslEnabled(ssl);
        if (!Strings.isNullOrEmpty(replicaSet)) {
            options.requiredReplicaSetName(replicaSet);
        }
        MongoClient mongoClient = new MongoClient(new MongoClientURI(mongoUrl, options));
        MongoDbFactory mongoDbFactory = new SimpleMongoDbFactory(mongoClient, mongoDataBase);
        return new MongoTemplate(mongoDbFactory);
    }

    @Bean
    public DealerDao clickHouseDealerDao(MongoTemplate mongoTemplate) {
        return new DealerDao(mongoTemplate);
    }

    @Bean
    public Yt yt(@Value("${dealer.yt.proxy}") String ytProxy,
                 @Value("${dealer.yt.token}") String ytToken) {
        return YtUtils.http(ytProxy, ytToken);
    }

    @Bean
    public TransferManagerClient transferManagerClient(@Value("${dealer.tm.api-url}") String apiUrl,
                                                       @Value("${dealer.tm.ui-url}") String uiUrl,
                                                       @Value("${dealer.yt.token}") String token) {
        return new TransferManagerClient(apiUrl, uiUrl, token, new HttpClientConfig());
    }

    @Bean
    public CuratorFramework curatorFramework(@Value("${dealer.zookeeper.quorum}") String zkQuorum,
                                             @Value("${dealer.zookeeper.prefix}") String zkPrefix,
                                             @Value("${dealer.zookeeper.timeout-seconds}") int zkTimeoutSeconds) {
        if (Strings.isNullOrEmpty(zkQuorum)) {
            return null;
        }
        zkPrefix = CharMatcher.is('/').trimFrom(zkPrefix);
        int timeoutMillis = (int) TimeUnit.SECONDS.toMillis(zkTimeoutSeconds);
        CuratorFramework curatorFramework = CuratorFrameworkFactory.builder()
            .connectString(zkQuorum)
            .retryPolicy(new RetryForever(timeoutMillis))
            .connectionTimeoutMs(timeoutMillis)
            .sessionTimeoutMs(timeoutMillis)
            .namespace(zkPrefix)
            .build();

        curatorFramework.start();
        return curatorFramework;
    }


    @Bean
    public NettyHttpClientContext httpClientContext(
        @Value("${dealer.http.retry-count:10}") int retryCount,
        @Value("${dealer.http.retry-sleep-millis:5000}") int retrySleepMillis,
        @Value("${dealer.http.connect-timeout-millis:10000}") int connectTimeoutMillis,
        @Value("${dealer.http.request-timeout-millis:180000}") int requestTimeoutMillis,
        @Value("${dealer.http.read-timeout-millis:180000}") int readTimeoutMillis) {

        HttpClientConfig config = new HttpClientConfig();
        config.setConnectTimeoutMillis(connectTimeoutMillis);
        config.setRequestTimeoutMillis(requestTimeoutMillis);
        config.setReadTimeoutMillis(readTimeoutMillis);
        config.setRetryPolicy(new RetryIdempotentWithSleepPolicy(retryCount, retrySleepMillis));
        return new NettyHttpClientContext(config);
    }

    @Bean(initMethod = "start")
    public DealerService dealerService(DealerDao dealerDao, Yt yt, TransferManagerClient tmClient,
                                       DealerConfigurationService configurationService,
                                       CuratorFramework curator, ComplicatedMonitoring monitoring) {
        return new DealerService(dealerDao, yt, tmClient, configurationService, curator, monitoring);
    }
}
