package ru.yandex.market.clickhouse.dealer.state;

import org.springframework.data.annotation.Id;
import org.springframework.data.annotation.Version;
import org.springframework.data.mongodb.core.mapping.Document;
import ru.yandex.market.clickhouse.dealer.config.DealerConfig;
import ru.yandex.market.clickhouse.dealer.operation.DealerOperation;

import java.time.Instant;
import java.util.Collection;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.stream.Collectors;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 26/01/2018
 */
@Document(collection = "state")
public class DealerState {
    @Id
    private final DealerConfig.Key configKey;

    @Version
    private Integer version;

    private final SortedMap<String, PartitionState> ytPartitionToState = new TreeMap<>();

    private DealerOperation currentOperation;

    private Error error;

    public DealerState(DealerConfig.Key configKey) {
        this.configKey = configKey;
    }

    public DealerConfig.Key getConfigKey() {
        return configKey;
    }

    public PartitionState getPartitionState(String ytPartition) {
        return ytPartitionToState.get(ytPartition);
    }

    public void putPartitionState(PartitionState state) {
        ytPartitionToState.put(state.getYtPartition(), state);
    }

    public DealerOperation getCurrentOperation() {
        return currentOperation;
    }

    public DealerState setCurrentOperation(DealerOperation currentOperation) {
        this.currentOperation = currentOperation;
        return this;
    }

    public Error getError() {
        return error;
    }

    public void setError(Error error) {
        this.error = error;
    }

    public void cleanError() {
        this.error = null;
    }

    public Collection<PartitionState> getPartitionStates(String clickHousePartition) {
        return getPartitionStates().stream()
            .filter(p -> p.getClickHousePartition().equals(clickHousePartition))
            .collect(Collectors.toList());
    }

    public Collection<PartitionState> getPartitionStates() {
        return ytPartitionToState.values();
    }

    public void removeYtDeletedPartitionStates(String clickHousePartition) {
        getPartitionStates(clickHousePartition)
            .stream()
            .filter(p -> p.getStatus() == PartitionState.Status.YT_DELETED)
            .forEach(p -> ytPartitionToState.remove(p.getYtPartition()));
    }

    public static class Error {
        private final String message;
        private final int number;
        private final Instant lastErrorTime;


        public Error(String message, int number) {
            this.message = message;
            this.number = number;
            this.lastErrorTime = Instant.now();
        }

        public String getMessage() {
            return message;
        }

        public int getNumber() {
            return number;
        }

        public Instant getLastErrorTime() {
            return lastErrorTime;
        }
    }
}
