package ru.yandex.market.clickhouse.dealer.state;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableSet;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeStringNode;
import ru.yandex.market.clickhouse.dealer.YtAttributes;
import ru.yandex.market.clickhouse.dealer.clickhouse.ClickHousePartitionExtractor;

import java.time.Instant;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 26/01/2018
 */
public class DealerStateProcessor {

    private static final Logger log = LogManager.getLogger();

    private DealerStateProcessor() {
    }

    public static void processYtNodes(DealerState state, ClickHousePartitionExtractor partitionExtractor,
                                      List<YTreeStringNode> nodes, Instant checkDate) {

        Set<PartitionYtState> ytPartitionStates = new HashSet<>();

        for (YTreeStringNode node : nodes) {
            String type = node.getAttributeOrThrow(YtAttributes.TYPE).stringValue();

            if (!type.equals("table")) {
                log.info("Node {} type is {} (not table). Skipping...", node.getValue(), type);
                continue;
            }
            PartitionYtState ytState = toYtState(node, checkDate);
            ytPartitionStates.add(ytState);
            updateState(state, partitionExtractor, node.getValue(), ytState);
        }
        checkForYtDeletedPartitions(ytPartitionStates, state);
    }

    private static void updateState(DealerState dealerState, ClickHousePartitionExtractor partitionExtractor,
                                    String ytPartition, PartitionYtState ytState) {

        PartitionState partitionState = dealerState.getPartitionState(ytPartition);
        partitionState = checkForNewPartition(dealerState, partitionState, partitionExtractor, ytPartition, ytState);
        checkForUpdatedPartition(partitionState, ytPartition, ytState);
        checkForSkippedPartition(partitionState, ytState);
    }

    private static PartitionState checkForNewPartition(DealerState dealerState, PartitionState partitionState,
                                                       ClickHousePartitionExtractor partitionExtractor,
                                                       String ytPartition, PartitionYtState ytState) {
        if (partitionState == null) {
            partitionState = new PartitionState(ytPartition, partitionExtractor.extract(ytPartition));
            dealerState.putPartitionState(partitionState);
            log.info("New partition in YT {}", ytPartition);
        }
        partitionState.setYtState(ytState);
        return partitionState;
    }

    private static void checkForUpdatedPartition(PartitionState partitionState, String ytPartition,
                                                 PartitionYtState ytState) {
        PartitionClickHouseState clickHouseState = partitionState.getClickHouseState();

        if (clickHouseState != null) {
            PartitionYtState transferredYtState = clickHouseState.getTransferredYtState();

            if (transferredYtState.getRevision() != ytState.getRevision()) {
                log.info(
                    "Partition {} updated in YT. Revision was {}, now {}",
                    ytPartition, transferredYtState.getRevision(), ytState.getRevision()
                );
                partitionState.setStatus(PartitionState.Status.TRANSFERRED_NEED_UPDATE);
            }
        } else if (PartitionState.Status.YT_DELETED == partitionState.getStatus()) {
            log.info("YT_DELETED partition {} is re-created in YT.", ytPartition);
            partitionState.setStatus(PartitionState.Status.TRANSFERRED_NEED_UPDATE);
        } else if (PartitionState.Status.SKIPPED == partitionState.getStatus() && ytState.getRowCount() != 0) {
            log.info("Changing SKIPPED partition {} to NEW.", ytPartition);
            partitionState.setStatus(PartitionState.Status.NEW);
        }
    }

    private static void checkForSkippedPartition(PartitionState partitionState, PartitionYtState ytState) {
        /* ytPartitions without any rows are marking as SKIPPED, because we shouldn't provide them to TM,
         ROTATED ytPartitions are skipped by default */
        if (ytState.getRowCount() == 0 && partitionState.getStatus() != PartitionState.Status.ROTATED) {
            partitionState.setStatus(PartitionState.Status.SKIPPED);
        }
    }

    private static void checkForYtDeletedPartitions(Set<PartitionYtState> ytPartitionStates, DealerState dealerState) {
        if (dealerState.getPartitionStates().size() == ytPartitionStates.size()) {
            return;
        }

        dealerState.getPartitionStates().stream()
            .filter(st -> !ytPartitionStates.contains(st.getYtState()))
            .forEach(st -> st.setStatus(PartitionState.Status.YT_DELETED));
    }

    /* if rotationPeriodDays is increased. We should transfer ytPartitions again */
    public static void checkForRotatedPartitionsRequiredBeTransferred(DealerState state, String chPartition) {
        state.getPartitionStates().stream()
            .filter(ps -> (
                    ps.getStatus() == PartitionState.Status.ROTATED || ps.getStatus() == PartitionState.Status.ROTATING
                ) && !shouldBeRotated(chPartition, ps)
            )
            .forEach(ps -> ps.setStatus(PartitionState.Status.TRANSFERRED_NEED_UPDATE));
    }

    /* when we got new yt's partitions, we should to check them for rotation status, deletion isn't required */
    public static void checkForNewPartitionsRequiredRotation(DealerState state, String chPartition) {
        state.getPartitionStates().stream()
            .filter(ps -> ps.getStatus() == PartitionState.Status.NEW && shouldBeRotated(chPartition, ps))
            .forEach(ps -> ps.setStatus(PartitionState.Status.ROTATED));
    }

    /* check dealer state for partitions required rotating statuses */
    public static void checkForPartitionsRequiredBeRotated(DealerState state, String chPartition) {
        Set<PartitionState.Status> statuses = ImmutableSet.of(PartitionState.Status.TRANSFERRED,
            PartitionState.Status.TRANSFERRED_NEED_UPDATE, PartitionState.Status.TRANSFERRED_DATA_MISMATCH);

        state.getPartitionStates().stream()
            .filter(ps -> statuses.contains(ps.getStatus()) && shouldBeRotated(chPartition, ps))
            .forEach(ps -> ps.setStatus(PartitionState.Status.ROTATING));
    }

    @VisibleForTesting
    public static boolean shouldBeRotated(String chPartition, PartitionState ps) {
        return ps.getClickHousePartition().replaceAll("'", "").compareTo(chPartition) < 0;
    }

    private static PartitionYtState toYtState(YTreeNode node, Instant checkDate) {
        long revision = node.getAttributeOrThrow(YtAttributes.REVISION).longValue();
        long rowCount = node.getAttributeOrThrow(YtAttributes.ROW_COUNT).longValue();
        String stringDate = node.getAttributeOrThrow(YtAttributes.MODIFICATION_TIME).stringValue();
        Instant modificationDate = Instant.parse(stringDate);
        return new PartitionYtState(modificationDate, rowCount, checkDate, revision);
    }
}
