package ru.yandex.market.clickhouse.dealer.state;

import org.springframework.data.annotation.PersistenceConstructor;

import java.util.Objects;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 01/02/2018
 */
public class PartitionState {
    public static final String SINGLE_PARTITION_ID = "<single>";

    private final String ytPartition;
    private final String clickHousePartition;
    private PartitionClickHouseState clickHouseState;
    private PartitionYtState ytState;
    private Status status;
    private int transferCount;

    public PartitionState(String ytPartition, String clickHousePartition) {
        this.ytPartition = ytPartition;
        this.clickHousePartition = clickHousePartition;
        this.status = Status.NEW;
        this.transferCount = 0;
    }

    @PersistenceConstructor
    public PartitionState(String ytPartition, String clickHousePartition, PartitionClickHouseState clickHouseState,
                          PartitionYtState ytState, Status status) {
        this(ytPartition, clickHousePartition, clickHouseState, ytState, status, 0);
    }

    public PartitionState(String ytPartition, String clickHousePartition, PartitionClickHouseState clickHouseState,
                          PartitionYtState ytState, Status status, int transferCount) {
        this.ytPartition = ytPartition;
        this.clickHousePartition = clickHousePartition;
        this.clickHouseState = clickHouseState;
        this.ytState = ytState;
        this.status = status;
        this.transferCount = transferCount;
    }


    public String getClickHousePartition() {
        return clickHousePartition;
    }

    public PartitionClickHouseState getClickHouseState() {
        return clickHouseState;
    }

    public PartitionYtState getYtState() {
        return ytState;
    }

    public Status getStatus() {
        return status;
    }

    public PartitionState setYtState(PartitionYtState ytState) {
        this.ytState = ytState;
        return this;
    }

    public PartitionState setClickHouseState(PartitionClickHouseState clickHouseState) {
        this.clickHouseState = clickHouseState;
        return this;
    }

    public PartitionState setStatus(Status status) {
        this.status = status;
        return this;
    }

    public String getYtPartition() {
        return ytPartition;
    }

    public int getTransferCount() {
        return transferCount;
    }

    public void increaseTransferCount() {
        transferCount++;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        PartitionState that = (PartitionState) o;
        return transferCount == that.transferCount &&
            Objects.equals(ytPartition, that.ytPartition) &&
            Objects.equals(clickHousePartition, that.clickHousePartition) &&
            Objects.equals(clickHouseState, that.clickHouseState) &&
            Objects.equals(ytState, that.ytState) &&
            status == that.status;
    }

    @Override
    public int hashCode() {
        return Objects.hash(ytPartition, clickHousePartition, clickHouseState, ytState, status, transferCount);
    }

    @Override
    public String toString() {
        return "PartitionState{" +
            "ytPartition='" + ytPartition + '\'' +
            ", clickHousePartition='" + clickHousePartition + '\'' +
            ", clickHouseState=" + clickHouseState +
            ", ytState=" + ytState +
            ", status=" + status +
            ", transferCount=" + transferCount +
            '}';
    }

    public enum Status {
        NEW,                        // ytPartition is a new one, will be transferred after processing
        TRANSFERRED,                // ytPartition is successful transferred to chPartition in Clickhouse
        TRANSFERRED_NEED_UPDATE,    // ytPartition was updated, its yt-table's revision is changed
        TRANSFERRED_DATA_MISMATCH,  // ytPartitions in chPartition count is different between CH and meta-information
        SKIPPED,                    // ytPartition exists but don't have any rows
        ROTATED,                    // chPartition doesn't exist anymore, it was deleted according to rotationPeriodDays
        ROTATING,                   // chPartition is processing in a rotation process
        YT_DELETED                  /* ytPartition is deleted in YT, but still in CH,
                                       will be deleted after TNU status for any ytPartition in chPartition */
    }
}
