package ru.yandex.market.clickhouse.dealer.tm;

import com.google.common.annotations.VisibleForTesting;
import com.google.gson.annotations.SerializedName;

import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 21/03/2018
 */
public class TmYt2ClickHouseCopyTask {
    @SerializedName("source_cluster")
    private final String ytCluster;
    @SerializedName("source_table")
    private final String ytPath;
    @SerializedName("destination_cluster")
    private final String clickHouseCluster;

    @SerializedName("destination_table")
    private final String clickHouseTable;

    @SerializedName("queue_name")
    private final String tmQueueName;

    @SerializedName("clickhouse_credentials")
    private final ClickHouseCredentials credentials;

    @SerializedName("clickhouse_copy_options")
    private final ClickHouseCopyOptions copyOptions;

    @SerializedName("clickhouse_copy_tool_settings_patch")
    private final Settings settings;

    @SerializedName("mdb_auth")
    private final MdbAuth mdbAuth;

    @SerializedName("mdb_cluster_address")
    private final MdbClusterAddress mdbClusterAddress;

    private TmYt2ClickHouseCopyTask(Builder builder) {
        ytCluster = builder.ytCluster;
        ytPath = builder.ytPath;
        clickHouseCluster = builder.clickHouseCluster;
        clickHouseTable = builder.clickHouseTable;
        tmQueueName = builder.tmQueueName;
        credentials = builder.credentials;
        mdbAuth = builder.mdbAuth;
        mdbClusterAddress = builder.mdbClusterAddress;

        String label = (builder.label == null) ? clickHouseCluster : builder.label;
        boolean addDate = (builder.dateColumn != null);
        copyOptions = new ClickHouseCopyOptions(
            builder.shardingKeys, addDate, builder.dateColumn, builder.resetState, label
        );

        settings = new Settings(new ClickHouseClient(builder.quorum));
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        TmYt2ClickHouseCopyTask that = (TmYt2ClickHouseCopyTask) o;
        return Objects.equals(ytCluster, that.ytCluster) &&
            Objects.equals(ytPath, that.ytPath) &&
            Objects.equals(clickHouseCluster, that.clickHouseCluster) &&
            Objects.equals(clickHouseTable, that.clickHouseTable) &&
            Objects.equals(tmQueueName, that.tmQueueName) &&
            Objects.equals(credentials, that.credentials) &&
            Objects.equals(copyOptions, that.copyOptions) &&
            Objects.equals(settings, that.settings) &&
            Objects.equals(mdbAuth, that.mdbAuth) &&
            Objects.equals(mdbClusterAddress, that.mdbClusterAddress);
    }

    @Override
    public int hashCode() {
        return Objects.hash(ytCluster, ytPath, clickHouseCluster, clickHouseTable, tmQueueName, credentials,
            copyOptions, settings, mdbAuth, mdbClusterAddress);
    }

    @Override
    public String toString() {
        return "TmYt2ClickHouseCopyTask{" +
            "ytCluster='" + ytCluster + '\'' +
            ", ytPath='" + ytPath + '\'' +
            ", clickHouseCluster='" + clickHouseCluster + '\'' +
            ", clickHouseTable='" + clickHouseTable + '\'' +
            ", tmQueueName='" + tmQueueName + '\'' +
            ", credentials=" + credentials +
            ", copyOptions=" + copyOptions +
            ", settings=" + settings +
            ", mdbAuth=" + mdbAuth +
            ", mdbClusterAddress=" + mdbClusterAddress +
            '}';
    }

    @VisibleForTesting
    ClickHouseCredentials getCredentials() {
        return credentials;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static class ClickHouseCredentials {
        private final String user;
        private final String password;

        public ClickHouseCredentials(String user, String password) {
            this.user = user;
            this.password = password;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            ClickHouseCredentials that = (ClickHouseCredentials) o;
            return Objects.equals(user, that.user) &&
                Objects.equals(password, that.password);
        }

        @Override
        public int hashCode() {

            return Objects.hash(user, password);
        }

        @Override
        public String toString() {
            return "ClickHouseCredentials{" +
                "user='" + user + '\'' +
                ", password='********\''" +
                '}';
        }
    }

    private static class Settings {
        @SerializedName("clickhouse_client")
        private final ClickHouseClient clickHouseClient;

        private Settings(ClickHouseClient clickHouseClient) {
            this.clickHouseClient = clickHouseClient;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            Settings settings = (Settings) o;
            return Objects.equals(clickHouseClient, settings.clickHouseClient);
        }

        @Override
        public int hashCode() {
            return Objects.hash(clickHouseClient);
        }

        @Override
        public String toString() {
            return "Settings{" +
                "clickHouseClient=" + clickHouseClient +
                '}';
        }
    }


    private static class ClickHouseClient {
        @SerializedName("per_shard_quorum")
        private final Quorum quorum;

        private ClickHouseClient(Quorum quorum) {
            this.quorum = quorum;
        }

        @Override
        public String toString() {
            return "ClickHouseClient{" +
                "quorum=" + quorum +
                '}';
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            ClickHouseClient that = (ClickHouseClient) o;
            return quorum == that.quorum;
        }

        @Override
        public int hashCode() {

            return Objects.hash(quorum);
        }
    }

    private static class MdbAuth {
        @SerializedName("organization_id")
        private final String organizationId;
        @SerializedName("oauth_token")
        private final String oauthToken;

        private MdbAuth(String organizationId, String oauthToken) {
            this.organizationId = organizationId;
            this.oauthToken = oauthToken;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            MdbAuth mdbAuth = (MdbAuth) o;
            return Objects.equals(organizationId, mdbAuth.organizationId) &&
                Objects.equals(oauthToken, mdbAuth.oauthToken);
        }

        @Override
        public int hashCode() {
            return Objects.hash(organizationId, oauthToken);
        }

        @Override
        public String toString() {
            return "MdbAuth{" +
                "organizationId='" + organizationId + '\'' +
                ", oauthToken='******************'" +
                '}';
        }
    }

    private static class MdbClusterAddress {
        @SerializedName("cluster_id")
        private final String clusterId;

        private MdbClusterAddress(String clusterId) {
            this.clusterId = clusterId;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            MdbClusterAddress that = (MdbClusterAddress) o;
            return Objects.equals(clusterId, that.clusterId);
        }

        @Override
        public int hashCode() {
            return Objects.hash(clusterId);
        }

        @Override
        public String toString() {
            return "MdbClusterAddress{" +
                "clusterId='" + clusterId + '\'' +
                '}';
        }
    }

    private static class ClickHouseCopyOptions {
        private final String command = "append";
        @SerializedName("sharding_key")
        private final String shardingKey;

        @SerializedName("add_date")
        private final boolean addDate;

        @SerializedName("date_column")
        private final String dateColumn;

        @SerializedName("reset_state")
        private final boolean resetState;

        @SerializedName("labels")
        private final List<String> labels;

        /**
         * @param shardingKeys Ключи шардирования
         * @param addDate      Возможность добавить дату в КХ таблицу
         * @param dateColumn   Название колонки для даты добавленной (addDate)
         * @param resetState   Возможность удалить стейт предыдущего копирования
         * @param label        Лейбл добавляется к ключу директории где ТМ хранит стейт копирования
         */
        private ClickHouseCopyOptions(List<String> shardingKeys, boolean addDate,
                                      String dateColumn, boolean resetState, String label) {
            if (shardingKeys.isEmpty()) {
                this.shardingKey = null;
            } else {
                this.shardingKey = shardingKeys.stream().collect(Collectors.joining(",", "(", ")"));
            }
            this.addDate = addDate;
            this.dateColumn = dateColumn;
            this.resetState = resetState;
            this.labels = Collections.singletonList(label);
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            ClickHouseCopyOptions that = (ClickHouseCopyOptions) o;
            return addDate == that.addDate &&
                Objects.equals(command, that.command) &&
                Objects.equals(shardingKey, that.shardingKey) &&
                Objects.equals(dateColumn, that.dateColumn) &&
                Objects.equals(resetState, that.resetState) &&
                Objects.equals(labels, that.labels);
        }


        @Override
        public int hashCode() {
            return Objects.hash(command, shardingKey, addDate, dateColumn, resetState, labels);
        }

        @Override
        public String toString() {
            return "ClickHouseCopyOptions{" +
                "command='" + command + '\'' +
                ", shardingKey='" + shardingKey + '\'' +
                ", addDate=" + addDate +
                ", dateColumn='" + dateColumn + '\'' +
                ", resetState=" + resetState +
                ", labels=" + labels +
                '}';
        }
    }

    public enum Quorum {
        @SerializedName("majority")
        MAJORITY,
        @SerializedName("at_least_one")
        AT_LEAST_ONE,
        @SerializedName("all")
        ALL
    }

    public static final class Builder {
        private String ytCluster;
        private String ytPath;
        private String clickHouseCluster;
        private String clickHouseTable;
        private String tmQueueName;
        private ClickHouseCredentials credentials;
        private List<String> shardingKeys = Collections.emptyList();
        private Quorum quorum = Quorum.MAJORITY;
        private String dateColumn;
        private MdbAuth mdbAuth;
        private MdbClusterAddress mdbClusterAddress;
        private boolean resetState;
        private String label;

        private Builder() {
        }

        public Builder withYtCluster(String ytCluster) {
            this.ytCluster = ytCluster;
            return this;
        }

        public Builder withYtPath(String ytPath) {
            this.ytPath = ytPath;
            return this;
        }

        public Builder withClickHouseCluster(String clickHouseCluster) {
            this.clickHouseCluster = clickHouseCluster;
            return this;
        }

        public Builder withTmQueueName(String tmQueueName) {
            this.tmQueueName = tmQueueName;
            return this;
        }

        public Builder withDbaasToken(String dbaasToken) {
            return withDbaasToken(null, dbaasToken);
        }

        public Builder withDbaasToken(String organizationId, String dbaasToken) {
            this.mdbAuth = new MdbAuth(organizationId, dbaasToken);
            return this;
        }

        public Builder withDbaasClusterAddress(String clusterId) {
            this.mdbClusterAddress = new MdbClusterAddress(clusterId);
            this.clickHouseCluster = "mdb-clickhouse";
            return this;
        }

        public Builder withClickHouseTable(String clickHouseTable) {
            this.clickHouseTable = clickHouseTable;
            return this;
        }

        public Builder withCredentials(String user, String password) {
            this.credentials = new ClickHouseCredentials(user, password);
            return this;
        }

        public Builder withShardingKey(String shardingKey) {
            this.shardingKeys = Collections.singletonList(shardingKey);
            return this;
        }

        public Builder withShardingKey(List<String> shardingKeys) {
            this.shardingKeys = shardingKeys;
            return this;
        }

        public Builder withQuorum(Quorum quorum) {
            this.quorum = quorum;
            return this;
        }

        public Builder withDateColumn(String dateColumn) {
            this.dateColumn = dateColumn;
            return this;
        }

        public Builder withResetState(boolean resetState) {
            this.resetState = resetState;
            return this;
        }

        public Builder withLabel(String label) {
            this.label = label;
            return this;
        }

        public TmYt2ClickHouseCopyTask build() {
            return new TmYt2ClickHouseCopyTask(this);
        }
    }
}
