package ru.yandex.market.clickhouse.dealer.tm;

import com.google.common.base.Preconditions;
import com.google.gson.FieldNamingPolicy;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonNull;
import com.google.gson.JsonObject;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import ru.yandex.market.request.netty.HttpClientConfig;
import ru.yandex.market.request.netty.JsonNettyHttpClient;
import ru.yandex.market.request.netty.auth.OAuthProvider;
import ru.yandex.market.request.trace.Module;

import java.time.Instant;
import java.time.ZonedDateTime;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 21/03/2018
 * https://wiki.yandex-team.ru/yt/userdoc/transfermanager/#api
 * https://wiki.yandex-team.ru/users/rlipovsky/yttoclickhouse/transfer_manager/
 */
public class TransferManagerClient {

    public static final Gson GSON = new Gson();

    private static final String CLUSTER_TYPE = "clickhouse";

    private static final Logger log = LogManager.getLogger();

    private final JsonNettyHttpClient client;

    private final String uiUrl;

    public TransferManagerClient(String apiUrl, String uiUrl, String ytOAuthToken, HttpClientConfig config) {
        Gson gson = new GsonBuilder()
            .setFieldNamingPolicy(FieldNamingPolicy.LOWER_CASE_WITH_UNDERSCORES)
            .registerTypeAdapter(Instant.class,
                (JsonDeserializer<Instant>) (json, type, ctx) -> ZonedDateTime.parse(json.getAsString()).toInstant())
            .create();

        client = new JsonNettyHttpClient(
            apiUrl, new OAuthProvider(ytOAuthToken), Module.TRANSFER_MANAGER, config, gson
        );
        this.uiUrl = uiUrl;
    }

    public String createTask(TmYt2ClickHouseCopyTask task) {
        log.info("Creating tm task with params: {}", task.toString());
        String taskId = client.executePostRequest("/tasks/", task, String.class);
        log.info("Task id is {}, url: {}/task?id={}", taskId, uiUrl, taskId);
        return taskId;
    }

    public void restartTask(String taskId) {
        log.info("Restarting tm task {}", taskId);
        client.executePostRequest("/tasks/" + taskId + "/restart/", null, JsonNull.class);
    }

    public TmTaskState getTaskState(String taskId) {
        return client.executeGetRequest("/tasks/" + taskId + "/", TmTaskState.class);
    }

    public TmClickHouseClusterOptions getClusterOptions(String tmClusterId) {
        JsonObject tmConfig = client.executeGetRequest("/config/", JsonObject.class);
        JsonObject clusters = tmConfig.getAsJsonObject("clusters");
        Preconditions.checkState(
            clusters.has(tmClusterId),
            "No cluster with name %s in transfer manager config. Available clusters: {}",
            tmClusterId,
            clusters.entrySet()
                .stream()
                .filter(e -> e.getValue().getAsJsonObject().get("type").getAsString().equals(CLUSTER_TYPE))
                .map(Map.Entry::getKey)
                .collect(Collectors.joining(","))
        );
        JsonObject cluster = clusters.getAsJsonObject(tmClusterId);
        String type = cluster.get("type").getAsString();
        Preconditions.checkState(
            CLUSTER_TYPE.equals(type),
            "TmCluster {} type is {}, must be {}", tmClusterId, type, CLUSTER_TYPE
        );

        JsonObject options = cluster.getAsJsonObject("options");
        log.info("Cluster options for cluster {}: {}", tmClusterId, options);
        return GSON.fromJson(options, TmClickHouseClusterOptions.class);
    }

}
