package ru.yandex.market.clickhouse.dealer.util;

import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

/**
 * @author Aleksei Malygin <a href="mailto:Malygin-Me@yandex-team.ru"></a>
 * Date: 2019-05-29
 */
public enum Retrier {
    INSTANCE;

    private static final int DEFAULT_RETRY_TIMEOUT_SECONDS = 5;
    int retryTimeoutSeconds;

    Retrier() {
        retryTimeoutSeconds = DEFAULT_RETRY_TIMEOUT_SECONDS;
    }

    public Retrier withTimeout(int retryTimeoutSeconds) {
        this.retryTimeoutSeconds = retryTimeoutSeconds;
        return this;
    }

    public void retryOnException(Runnable r, int attempts, List<String> ignoreIfMessageContains) {
        retryOnException(
            () -> {
                r.run();
                return null;
            },
            attempts,
            ignoreIfMessageContains
        );
    }

    public <T> T retryOnException(Supplier<T> s, int attempts, List<String> ignoreIfMessageContains) {
        for (int i = 0; i <= attempts; i++) {
            try {
                return s.get();
            } catch (Exception e) {
                if ((ignoreIfMessageContains.isEmpty() || (ignoreIfMessageContains.stream()
                    .anyMatch(m -> e.getMessage() != null && e.getMessage().contains(m)))
                ) && i < attempts) {
                    try {
                        TimeUnit.SECONDS.sleep(retryTimeoutSeconds);
                    } catch (InterruptedException ex) {
                        throw new RuntimeException("Retrier's sleep interrupted", ex);
                    }
                    continue;
                }
                throw e;
            }
        }
        return s.get();
    }
}
