package ru.yandex.market.calendar;

import java.time.LocalDate;

/**
 * Карточка календарного дня, выгруженного из Яндекс.Календаря.
 *
 * @author Denis Chernyshov <zoom@yandex-team.ru>
 */
public class YandexCalendarDay {

    /**
     * Тип дня.
     */
    public enum Type {
        /**
         * Праздничный день.
         */
        HOLIDAY,
        /**
         * Рабочий день.
         */
        WORKDAY,
        /**
         * Выходной день.
         */
        WEEKEND
    }

    /**
     * Дата, к которой относится день.
     */
    private final LocalDate date;
    /**
     * Тип дня.
     */
    private final Type type;
    /**
     * Признак, что день является нерабочим.
     */
    private final boolean holiday;
    /**
     * С какого дня был перенесен праздничный (нерабочий) /рабочий день.
     */
    private final LocalDate originalDate;

    public YandexCalendarDay(LocalDate date, Type type, boolean holiday, LocalDate originalDate) {
        this.date = date;
        this.type = type;
        this.holiday = holiday;
        this.originalDate = originalDate;
    }

    public LocalDate getDate() {
        return date;
    }

    public Type getType() {
        return type;
    }

    public boolean isWorkday() {
        return type == Type.WORKDAY;
    }

    public boolean isHoliday() {
        return holiday;
    }

    public LocalDate getOriginalDate() {
        return originalDate;
    }

    @Override
    public String toString() {
        return "YandexCalendarDay{" +
            "date=" + date +
            ", type=" + type +
            ", holiday=" + holiday +
            ", originalDate=" + originalDate +
            '}';
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        YandexCalendarDay that = (YandexCalendarDay) o;

        if (holiday != that.holiday) {
            return false;
        }
        if (!date.equals(that.date)) {
            return false;
        }
        if (type != that.type) {
            return false;
        }
        return !(originalDate != null ? !originalDate.equals(that.originalDate) : that.originalDate != null);

    }

    @Override
    public int hashCode() {
        int result = date.hashCode();
        result = 31 * result + type.hashCode();
        result = 31 * result + (holiday ? 1 : 0);
        result = 31 * result + (originalDate != null ? originalDate.hashCode() : 0);
        return result;
    }
}
