package ru.yandex.market.calendar;

import com.google.common.base.CharMatcher;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.io.IOUtils;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Required;

import java.io.IOException;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * @author Denis Chernyshov <zoom@yandex-team.ru>
 */
public class YandexCalendarService implements InitializingBean {

    private static final Logger log = LogManager.getLogger();

    private static final int RU_REGION = 225;


    private String apiUrl = "http://calendar.yandex-team.ru";
    private String clientName;

    private HttpClient httpClient;

    private int httpTimeoutSeconds = 60;
    private int maxConnections = 5;

    public YandexCalendarService() {
    }

    public YandexCalendarService(String clientName) throws Exception {
        this.clientName = clientName;
        afterPropertiesSet();
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        int timeoutMillis = (int) TimeUnit.SECONDS.toMillis(httpTimeoutSeconds);

        RequestConfig requestConfig = RequestConfig.custom()
            .setConnectionRequestTimeout(timeoutMillis)
            .setConnectTimeout(timeoutMillis)
            .setSocketTimeout(timeoutMillis)
            .build();

        httpClient = HttpClientBuilder.create()
            .setMaxConnPerRoute(maxConnections)
            .setMaxConnTotal(maxConnections)
            .setDefaultRequestConfig(requestConfig)
            .build();
    }

    public List<YandexCalendarDay> getDays(LocalDate start, LocalDate end) throws IOException {
        return getDays(RU_REGION, start, end);
    }

    public List<YandexCalendarDay> getDays(long regionId, LocalDate start, LocalDate end) throws IOException {

        HttpGet request = new HttpGet(
            apiUrl + "/export/holidays.xml?"
                + "start_date=" + DateTimeFormatter.ISO_LOCAL_DATE.format(start)
                + "&out_mode=all"
                + "&country_id=" + regionId
                + "&end_date=" + DateTimeFormatter.ISO_LOCAL_DATE.format(end)
                + "&who_am_i=" + clientName
        );

        HttpResponse response = executeAndCheckCode(request);
        YandexCalendarXmlParser parser = new YandexCalendarXmlParser();
        parser.parse(response.getEntity().getContent());
        return parser.getDays();
    }

    private HttpResponse executeAndCheckCode(HttpUriRequest request) throws IOException {
        HttpResponse response = httpClient.execute(request);
        if (response.getStatusLine().getStatusCode() == HttpStatus.SC_OK) {
            return response;
        }
        String error = IOUtils.toString(response.getEntity().getContent());

        throw new IOException(
            "Error. Code:" + response.getStatusLine().getStatusCode() +
                ", error: " + error + ", request: '" + request.getURI().toString() + "'"
        );

    }

    @Required
    public void setClientName(String clientName) {
        this.clientName = clientName;
    }

    public void setHttpTimeoutSeconds(int httpTimeoutSeconds) {
        this.httpTimeoutSeconds = httpTimeoutSeconds;
    }

    public void setMaxConnections(int maxConnections) {
        this.maxConnections = maxConnections;
    }

    public void setApiUrl(String apiUrl) {
        this.apiUrl = CharMatcher.is('/').trimFrom(apiUrl);
    }


}
