package ru.yandex.market.calendar;

import org.xml.sax.Attributes;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.XMLReader;
import ru.yandex.common.util.xml.parser.ElementListenerAdapter;
import ru.yandex.common.util.xml.parser.ElementOrientedSAXHandler;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;

/**
 * @author Denis Chernyshov <zoom@yandex-team.ru>
 */
public class YandexCalendarXmlParser extends ElementOrientedSAXHandler {

    private static final DateTimeFormatter DATE_FORMATTER = DateTimeFormatter.ISO_LOCAL_DATE;

    private final List<YandexCalendarDay> days = new ArrayList<>();

    public YandexCalendarXmlParser() {
        addElementListener("/holidays/get-holidays/days/day", new ElementListenerAdapter() {
            @Override
            public void onOpen(ElementOrientedSAXHandler handler, Attributes attributes) throws SAXParseException {
                LocalDate originalDate = null;
                String value = attributes.getValue("transfer-date");
                if (value != null) {
                    originalDate = LocalDate.from(DATE_FORMATTER.parse(value));
                }
                days.add(new YandexCalendarDay(
                    LocalDate.from(DATE_FORMATTER.parse(attributes.getValue("date"))),
                    toType(attributes.getValue("day-type")),
                    "1".equals(attributes.getValue("is-holiday")),
                    originalDate));
            }
        });
    }

    public void parse(InputStream in) throws IOException {
        InputStream bufferedIn = null;
        try {
            bufferedIn = new BufferedInputStream(in);

            try {
                XMLReader xmlReader = new org.apache.xerces.parsers.SAXParser();
                xmlReader.setContentHandler(this);
                xmlReader.parse(new InputSource(bufferedIn));
            } catch (SAXException e) {
                throw new IOException("Error parsing XML content", e);
            }
        } finally {
            if (bufferedIn != null) {
                bufferedIn.close();
            }
        }
    }


    private YandexCalendarDay.Type toType(String value) {
        switch (value) {
            case "holiday":
                return YandexCalendarDay.Type.HOLIDAY;
            case "weekday":
                return YandexCalendarDay.Type.WORKDAY;
            case "weekend":
                return YandexCalendarDay.Type.WEEKEND;
            default:
                throw new IllegalArgumentException("Unknown day type: " + value);
        }
    }

    public List<YandexCalendarDay> getDays() {
        return days;
    }
}
