package ru.yandex.market.clickphite;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Splitter;
import com.google.common.base.Stopwatch;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.ListMultimap;
import com.google.common.collect.RangeSet;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.CuratorFrameworkFactory;
import org.apache.curator.framework.recipes.leader.LeaderLatch;
import org.apache.curator.framework.recipes.leader.LeaderLatchListener;
import org.apache.curator.framework.state.ConnectionState;
import org.apache.curator.framework.state.ConnectionStateListener;
import org.apache.curator.retry.RetryNTimes;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.common.util.date.DateUtil;
import ru.yandex.market.clickphite.config.ClickphiteConfiguration;
import ru.yandex.market.clickphite.config.ConfigurationService;
import ru.yandex.market.clickphite.config.metric.MetricPeriod;
import ru.yandex.market.clickphite.graphite.GraphiteClient;
import ru.yandex.market.clickphite.http.Clickphite;
import ru.yandex.market.clickphite.meta.ClickphiteMetaDao;
import ru.yandex.market.clickphite.metric.AsyncMetricGroupMonitoring;
import ru.yandex.market.clickphite.metric.MetricContext;
import ru.yandex.market.clickphite.metric.MetricContextGroup;
import ru.yandex.market.clickphite.metric.MetricService;
import ru.yandex.market.clickphite.metric.QueryWeight;
import ru.yandex.market.clickphite.worker.MetricGroupWorker;
import ru.yandex.market.clickphite.worker.Worker;
import ru.yandex.market.health.HealthMetaDao;
import ru.yandex.market.health.OutputInfo;
import ru.yandex.market.http.MonitoringResult;
import ru.yandex.market.monitoring.ComplicatedMonitoring;
import ru.yandex.market.monitoring.MonitoringStatus;
import ru.yandex.market.monitoring.MonitoringUnit;

import java.io.IOException;
import java.net.InetAddress;
import java.text.ParseException;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 23/12/14
 */
public class ClickphiteService implements LeaderLatchListener, ConnectionStateListener,
    Runnable, ru.yandex.market.clickphite.http.ClickphiteService {

    private static final Logger log = LogManager.getLogger();

    private int lightThreadCount = 5;
    private int mediumThreadCount = 3;
    private int heavyThreadCount = 2;
    private int updateIntervalSeconds = 10;
    private int metricDelaySeconds = 20;
    private int maxQueriesPerUpdate = 10;
    private int maxMetricsAgeOnLagMinutes = 30;
    private String zookeeperQuorum;
    private int zookeeperTimeoutSeconds = 30;
    private String zookeeperPrefix = "clickphite";
    private ComplicatedMonitoring monitoring;
    private ConfigurationService configurationService;
    private MetricService metricService;
    private GraphiteClient graphiteClient;
    private ClickphiteMetaDao clickphiteMetaDao;
    private HealthMetaDao healthMetaDao;
    private AsyncMetricGroupMonitoring asyncMetricGroupMonitoring;

    private final MonitoringUnit zkConnectionMonitoringUnit = new MonitoringUnit("ZooKeeper");
    private AtomicReference<ClickphiteConfiguration> conf = new AtomicReference<>();
    private AtomicLong masterBecomeTimeMillis = new AtomicLong();

    private ExecutorService lightExecutorService;
    private ExecutorService mediumExecutorService;
    private ExecutorService heavyExecutorService;

    private CuratorFramework curatorFramework;
    private LeaderLatch leaderLatch;
    private int fullRebuildMaxDays = 2;

    private volatile boolean lagMode = false;

    private MonitoringUnit executionTimeMonitoring;
    private MonitoringUnit mongoDataMonitoring;
    private int executionLimitMinutesWarn = 5;
    private int executionLimitMinutesCritical = 15;
    private int mongoDataExecutionLimitMinutesCritical = 5;
    private Set<String> heavyTables = new HashSet<>();
    private Set<String> mediumTables = new HashSet<>();

    private static final DateUtil.ThreadLocalDateFormat DATE_FORMAT = new DateUtil.ThreadLocalDateFormat(
        "yyyyMMddHHmm"
    );

    public void afterPropertiesSet() throws Exception {
        lightExecutorService = Executors.newFixedThreadPool(lightThreadCount);
        mediumExecutorService = Executors.newFixedThreadPool(mediumThreadCount);
        heavyExecutorService = Executors.newFixedThreadPool(heavyThreadCount);

        int timeoutMillis = (int) TimeUnit.SECONDS.toMillis(zookeeperTimeoutSeconds);

        curatorFramework = CuratorFrameworkFactory.builder()
            .connectString(zookeeperQuorum)
            .retryPolicy(new RetryNTimes(Integer.MAX_VALUE, timeoutMillis))
            .connectionTimeoutMs(timeoutMillis)
            .sessionTimeoutMs(timeoutMillis)
            .namespace(zookeeperPrefix)
            .build();
        curatorFramework.getConnectionStateListenable().addListener(this);
        curatorFramework.start();
        monitoring.addUnit(zkConnectionMonitoringUnit);
        if (!curatorFramework.getZookeeperClient().blockUntilConnectedOrTimedOut()) {
            zkConnectionMonitoringUnit.critical("No connection to zk: " + zookeeperQuorum);
        }
        String host = InetAddress.getLocalHost().getCanonicalHostName();
        log.info("I am " + host);
        leaderLatch = new LeaderLatch(curatorFramework, "/master", host);
        leaderLatch.addListener(this);
        leaderLatch.start();
        leaderLatch.await(1, TimeUnit.SECONDS);
        logCurrentMaster();
        Thread thread = new Thread(this, "ClickphiteService");
        thread.start();

        executionTimeMonitoring = monitoring.createPersistUnit("Clickphite service");
        executionTimeMonitoring.setCriticalTimeout(executionLimitMinutesCritical, TimeUnit.MINUTES);
        executionTimeMonitoring.setWarningTimeout(executionLimitMinutesWarn, TimeUnit.MINUTES);

        mongoDataMonitoring = monitoring.createPersistUnit("Mongo data");
        mongoDataMonitoring.setCriticalTimeout(mongoDataExecutionLimitMinutesCritical, TimeUnit.MINUTES);

        Runtime.getRuntime().addShutdownHook(
            new Thread(() -> {
                try {
                    leaderLatch.close();
                } catch (IOException e) {
                    log.error("Cannon release leadership", e);
                }
            })
        );
    }

    @Override
    public void run() {
        while (!Thread.interrupted()) {
            try {
                workIfMaster();
            } catch (Exception e) {
                log.error("Error in ClickphiteService.", e);
            }

            try {
                TimeUnit.SECONDS.sleep(updateIntervalSeconds);
            } catch (InterruptedException e) {
                log.info("Interrupted");
                System.exit(0);
            }
        }
    }

    private void workIfMaster() {
        if (isMaster()) {
            Profiler profiler = new Profiler();
            profiler.totalStopwatch.start();

            profiler.configurationLoadStopwatch.start();
            updateConfiguration();
            profiler.configurationLoadStopwatch.stop();

            profiler.actualMetricDataLoadStopwatch.start();
            loadActualMetricData();
            profiler.actualMetricDataLoadStopwatch.stop();

            profiler.taskCreateStopwatch.start();
            createTasks(profiler);
            profiler.taskCreateStopwatch.stop();

            profiler.taskSchedulerStopwatch.start();
            scheduleTasks();
            profiler.taskSchedulerStopwatch.stop();

            profiler.totalStopwatch.stop();
            profiler.logInfo();

            log.info("Graphite client status: " + graphiteClient.getStatus());
            executionTimeMonitoring.ok();
        } else {
            asyncMetricGroupMonitoring.setOk();
            configurationService.resetConfigMonitoring();
        }
    }

    @Override
    public void notLeader() {
        log.info("Mastership lost :(");
        logCurrentMaster();
    }

    @Override
    public void isLeader() {
        log.info("New master is on the road!");
        logCurrentMaster();
        masterBecomeTimeMillis.set(System.currentTimeMillis());
    }

    public boolean isMaster() {
        return leaderLatch != null && leaderLatch.hasLeadership();
    }

    @Override
    public void stateChanged(CuratorFramework client, ConnectionState newState) {
        switch (newState) {
            case CONNECTED:
            case RECONNECTED:
                log.info("Got connection to ZK. Status " + newState);
                logCurrentMaster();
                zkConnectionMonitoringUnit.ok();
                break;
            case LOST:
            case SUSPENDED:
                zkConnectionMonitoringUnit.critical("Lost connection to ZK");
                log.error("Lost connection to ZK. Status " + newState);
                break;
            default:
                throw new IllegalStateException("Unknown connection state:" + newState);
        }
    }

    private void logCurrentMaster() {
        if (leaderLatch == null) {
            return;
        }
        try {
            final String currentMasterInfo = "Current master is: " + getMasterId();
            zkConnectionMonitoringUnit.ok(currentMasterInfo);
            log.info(currentMasterInfo);
        } catch (Exception e) {
            log.error("Failed to get current master", e);
        }
    }

    public String getMasterId() throws Exception {
        return leaderLatch.getLeader().getId();
    }

    private void loadActualMetricData() {
        List<MetricContext> metricContexts = conf.get().getMetricContexts()
            .stream()
            .filter(metricContext -> metricContext.getLastLoadTimeSeconds() <= masterBecomeTimeMillis.get())
            .collect(Collectors.toList());

        if (metricContexts.isEmpty()) {
            return;
        }
        long currentTimeMillis = System.currentTimeMillis();

        int updatedCount = clickphiteMetaDao.load(metricContexts);
        log.info("Loaded info for {} metric context. Updated {} metrics, consider {} metric  as new",
            metricContexts.size(), updatedCount, metricContexts.size() - updatedCount
        );
        for (MetricContext metricContext : metricContexts) {
            metricContext.setLastLoadTimeSeconds(currentTimeMillis);
        }
    }

    private void scheduleTasks() {
        List<MetricContextGroup> metricGroups = conf.get().getMetricContextGroups();
        asyncMetricGroupMonitoring.setTotalMetricCount(metricGroups.size());
        int created = 0;
        int processing = 0;
        int actual = 0;

        for (MetricContextGroup metricGroup : metricGroups) {
            if (metricGroup.getProcessStatus().isOnAir()) {
                processing++;
                continue;
            }

            long maxDateTimeMillis = getMaxDateTimeMillis();
            metricGroup.getMetricContexts().forEach(metric -> metric.getMetricQueue().compact(maxDateTimeMillis));

            int metricEndTimeSeconds = metricGroup.getPeriod().getEndTimeSeconds(metricDelaySeconds);
            boolean hasActualTasks = metricGroup.getMetricContexts().stream()
                .anyMatch(metric -> metric.getMetricQueue().hasActualTasks(metricEndTimeSeconds));

            if (hasActualTasks) {
                submit(new MetricGroupWorker(
                    this, metricService, asyncMetricGroupMonitoring, metricGroup,
                    getQueryWeight(metricGroup)
                ));
                created++;
            } else {
                actual++;
            }
        }

        log.info("Processed " + metricGroups.size() + " metric groups. " +
            "Scheduled: " + created + ", already on air: " + processing + ", actual: " + actual
        );
    }

    private QueryWeight getQueryWeight(MetricContextGroup metricGroup) {
        return getQueryWeight(metricGroup, mediumTables, heavyTables);
    }

    @VisibleForTesting
    protected static QueryWeight getQueryWeight(MetricContextGroup metricGroup, Set<String> mediumTables,
                                                Set<String> heavyTables) {
        QueryWeight weightByPeriod;
        switch (metricGroup.getPeriod()) {
            case QUARTER:
            case MONTH:
            case WEEK:
            case DAY:
                weightByPeriod = QueryWeight.HEAVY;
                break;
            case HOUR:
                weightByPeriod = QueryWeight.MEDIUM;
                break;
            default:
                weightByPeriod = QueryWeight.LIGHT;
        }

        if (heavyTables.contains(metricGroup.getTable().getFullName())) {
            return maxWeight(weightByPeriod, QueryWeight.HEAVY);
        }
        if (mediumTables.contains(metricGroup.getTable().getFullName())) {
            return maxWeight(weightByPeriod, QueryWeight.MEDIUM);
        }

        return weightByPeriod;
    }

    private static QueryWeight maxWeight(QueryWeight leftWeight, QueryWeight rightWeight) {
        return leftWeight.compareTo(rightWeight) > 0 ? leftWeight : rightWeight;
    }

    public long getMaxDateTimeMillis() {
        return System.currentTimeMillis() - TimeUnit.SECONDS.toMillis(metricDelaySeconds);
    }

    private void createTasks(Profiler profiler) {

        profiler.outputInfoLoadStopwatch.start();
        List<OutputInfo> outputInfos = healthMetaDao.get();
        profiler.outputInfoLoadStopwatch.stop();

        if (outputInfos.isEmpty()) {
            mongoDataMonitoring.critical("Received empty meta data list");
        } else {
            mongoDataMonitoring.ok();
        }

        profiler.taskGenerationStopwatch.start();
        createTasks(outputInfos, System.currentTimeMillis());
        profiler.taskGenerationStopwatch.stop();

        profiler.metricSaveStopwatch.start();
        clickphiteMetaDao.save(conf.get().getMetricContexts());
        profiler.metricSaveStopwatch.stop();

        profiler.outputInfoDeleteStopwatch.start();
        healthMetaDao.delete(outputInfos);
        profiler.outputInfoDeleteStopwatch.stop();

        log.info(
            "Processed " + outputInfos.size() + " outputs. " +
                "Updated " + conf.get().getMetricContexts().size() + " metrics."
        );
    }

    private void createTasks(List<OutputInfo> outputInfos, Long outputInfoTimeMillis) {
        ListMultimap<String, OutputInfo> tableToOutputs = ArrayListMultimap.create();
        for (OutputInfo outputInfo : outputInfos) {
            tableToOutputs.put(outputInfo.getTable(), outputInfo);
        }
        for (String table : tableToOutputs.keySet()) {
            createTasks(table, tableToOutputs.get(table), outputInfoTimeMillis);
        }
    }

    private void createTasks(String tableName, List<OutputInfo> outputInfos, Long outputInfoTimeMillis) {
        ClickHouseTable table = conf.get().getClickHouseTable(tableName);
        if (table == null) {
            return;
        }
        ListMultimap<MetricPeriod, MetricContext> metricByPeriod = conf.get().getMetricsForTableByPeriod(table);
        RangeSet<Integer> rangeSet = DateTimeUtils.toRangeSet(outputInfos);


        for (MetricPeriod period : metricByPeriod.keySet()) {
            createTasks(period, rangeSet, metricByPeriod.get(period), outputInfoTimeMillis);
        }
    }

    private void createTasks(MetricPeriod period, RangeSet<Integer> rangeSet,
                             List<MetricContext> metricContexts, Long outputInfoTimeMillis) {
        RangeSet<Integer> periodRangeSet = DateTimeUtils.toPeriodRangeSet(rangeSet, period);
        for (MetricContext metricContext : metricContexts) {
            metricContext.getMetricQueue().add(outputInfoTimeMillis, periodRangeSet);
        }
    }

    private void submit(Worker worker) {
        switch (worker.getQueryWeight()) {
            case LIGHT:
                lightExecutorService.submit(worker);
                break;
            case MEDIUM:
                mediumExecutorService.submit(worker);
                break;
            case HEAVY:
                heavyExecutorService.submit(worker);
                break;
            default:
                throw new UnsupportedOperationException("Unknown query weight: " + worker.getQueryWeight());
        }
    }

    private void updateConfiguration() {
        configurationService.reload();
        conf.set(configurationService.getConfiguration());
        asyncMetricGroupMonitoring.actualizeMetrics(conf.get().getMetricContextGroups());
    }

    public int getMetricDelaySeconds() {
        return metricDelaySeconds;
    }

    public int getMaxQueriesPerUpdate() {
        return maxQueriesPerUpdate;
    }

    public boolean isLagMode() {
        return lagMode;
    }

    public int getMaxMetricsAgeOnLagMinutes() {
        return maxMetricsAgeOnLagMinutes;
    }

    public static class Profiler {
        private Stopwatch totalStopwatch = Stopwatch.createUnstarted();

        private Stopwatch configurationLoadStopwatch = Stopwatch.createUnstarted();
        private Stopwatch actualMetricDataLoadStopwatch = Stopwatch.createUnstarted();

        private Stopwatch taskCreateStopwatch = Stopwatch.createUnstarted();
        //Task create detalization
        private Stopwatch outputInfoLoadStopwatch = Stopwatch.createUnstarted();
        private Stopwatch taskGenerationStopwatch = Stopwatch.createUnstarted();
        private Stopwatch metricSaveStopwatch = Stopwatch.createUnstarted();
        private Stopwatch outputInfoDeleteStopwatch = Stopwatch.createUnstarted();
        //
        private Stopwatch taskSchedulerStopwatch = Stopwatch.createUnstarted();

        public void logInfo() {
            log.info(
                "Profile. Tolat: {}ms. Config load: {}ms. Actual metric data load: {}ms. " +
                    "Task create: {}ms. " +
                    "(outputInfo load: {}ms; generation: {}ms; metricSave {}ms; outputInfo delete {}ms). " +
                    "Task schedule: {}ms",
                totalStopwatch.elapsed(TimeUnit.MILLISECONDS),
                configurationLoadStopwatch.elapsed(TimeUnit.MILLISECONDS),
                actualMetricDataLoadStopwatch.elapsed(TimeUnit.MILLISECONDS),
                taskCreateStopwatch.elapsed(TimeUnit.MILLISECONDS),
                outputInfoLoadStopwatch.elapsed(TimeUnit.MILLISECONDS),
                taskGenerationStopwatch.elapsed(TimeUnit.MILLISECONDS),
                metricSaveStopwatch.elapsed(TimeUnit.MILLISECONDS),
                outputInfoDeleteStopwatch.elapsed(TimeUnit.MILLISECONDS),
                taskSchedulerStopwatch.elapsed(TimeUnit.MILLISECONDS)
            );
        }
    }

    @Override
    public MonitoringResult ping() {
        return monitoring();
    }

    @Override
    public MonitoringResult monitoring() {
        ComplicatedMonitoring.Result result = monitoring.getResult();
        return new MonitoringResult(convertStatus(result.getStatus()), result.getMessage());
    }


    private void throwIfNowMaster() {
        if (!isMaster()) {
            throw new RuntimeException("Not a master");
        }
    }

    @Override
    public Clickphite.Empty rebuildMetric(Clickphite.MetricRebuildRequest request) {

        throwIfNowMaster();

        MetricContext metric = conf.get().getMetricContext(request.getMetric());
        if (metric == null) {
            throw new RuntimeException("Metric: " + request.getMetric() + " not found. Use full metric id.");
        }
        TimeRange timeRange = getTimeRange(
            request.getStartTimestampSeconds(), request.getEndTimestampSeconds(),
            request.getStart(), request.getEnd(),
            metric.getClickHouseTable()
        );
        metric.getMetricQueue().add(
            System.currentTimeMillis(),
            DateTimeUtils.toPeriodRangeSet(timeRange, metric.getPeriod())
        );
        log.info("Rebuilding metric: '" + metric.getId() + " for period: " + timeRange);
        return Clickphite.Empty.newBuilder().build();
    }

    @Override
    public Clickphite.Empty rebuildForTable(Clickphite.TableRebuildRequest request) {


        ClickHouseTable table = conf.get().getClickHouseTable(request.getTable());
        if (table == null) {
            throw new RuntimeException(
                "Table: " + request.getTable() + " not found. Use full table name (ex: 'database.table')."
            );
        }

        TimeRange timeRange = getTimeRange(
            request.getStartTimestampSeconds(), request.getEndTimestampSeconds(),
            request.getStart(), request.getEnd(),
            table
        );
        OutputInfo outputInfo = new OutputInfo(
            table.getFullName(),
            timeRange.getStartTimestampSeconds(),
            timeRange.getEndTimestampSeconds()
        );

        log.info("Saving outputInfo to rebuild table '" + table.getFullName() + "':" + outputInfo);
        healthMetaDao.save(outputInfo);
        return Clickphite.Empty.newBuilder().build();
    }

    @Override
    public Clickphite.Empty rebuildAllMetrics(Clickphite.FullRebuildRequest request) {
        throwIfNowMaster();

        final boolean hasStart = (request.getStart() != null && request.getStart().isEmpty()) ||
            (request.getStartTimestampSeconds() > 0);

        if (!hasStart) {
            throw new RuntimeException("Start of period must be specified");
        }

        final TimeRange timeRange = getTimeRange(
            request.getStartTimestampSeconds(), request.getEndTimestampSeconds(),
            request.getStart(), request.getEnd(),
            null
        );

        if (TimeUnit.SECONDS.toDays(timeRange.getDurationSeconds()) > fullRebuildMaxDays) {
            throw new RuntimeException("Full rebuild interval must be less than " + fullRebuildMaxDays + " days");
        }

        int metricCount = 0;
        for (MetricContext metric : conf.get().getMetricContexts()) {
            metric.getMetricQueue().add(
                System.currentTimeMillis(),
                DateTimeUtils.toPeriodRangeSet(timeRange, metric.getPeriod())
            );
            metricCount++;
        }

        log.info("Rebuilding " + metricCount + " + metrics for period: " + timeRange);
        return Clickphite.Empty.newBuilder().build();
    }

    public TimeRange getTimeRange(int startTimestampSeconds, int endTimestampSeconds,
                                  String start, String end, ClickHouseTable table) {

        if (startTimestampSeconds <= 0) {
            if (start == null || end.isEmpty()) {
                startTimestampSeconds = metricService.getTableFirstDataTimestampSeconds(table);
            } else {
                try {
                    startTimestampSeconds = DateTimeUtils.dateToSeconds(DATE_FORMAT.parse(start));
                } catch (ParseException e) {
                    throw new RuntimeException("Wrong date format for start date, must be 'yyyyMMddHHmm'", e);
                }
            }
        }
        if (endTimestampSeconds <= 0) {
            if (end == null || end.isEmpty()) {
                endTimestampSeconds = DateTimeUtils.currentTimeSeconds();
            } else {
                try {
                    endTimestampSeconds = DateTimeUtils.dateToSeconds(DATE_FORMAT.parse(end));
                } catch (ParseException e) {
                    throw new RuntimeException("Wrong date format for end date, must be 'yyyyMMddHHmm'", e);
                }
            }
        }
        return new TimeRange(startTimestampSeconds, endTimestampSeconds);
    }


    private MonitoringResult.Status convertStatus(MonitoringStatus status) {
        switch (status) {
            case OK:
                return MonitoringResult.Status.OK;
            case WARNING:
                return MonitoringResult.Status.WARNING;
            case CRITICAL:
                return MonitoringResult.Status.ERROR;
            default:
                throw new IllegalStateException();
        }
    }

    @Required
    public void setConfigurationService(ConfigurationService configurationService) {
        this.configurationService = configurationService;
    }


    @Required
    public void setMetricService(MetricService metricService) {
        this.metricService = metricService;
    }

    @Required
    public void setGraphiteClient(GraphiteClient graphiteClient) {
        this.graphiteClient = graphiteClient;
    }

    @Required
    public void setClickphiteMetaDao(ClickphiteMetaDao clickphiteMetaDao) {
        this.clickphiteMetaDao = clickphiteMetaDao;
    }

    @Required
    public void setZookeeperQuorum(String zookeeperQuorum) {
        this.zookeeperQuorum = zookeeperQuorum;
    }

    @Required
    public void setMonitoring(ComplicatedMonitoring monitoring) {
        this.monitoring = monitoring;
    }

    @Required
    public void setHealthMetaDao(HealthMetaDao healthMetaDao) {
        this.healthMetaDao = healthMetaDao;
    }

    @Required
    public void setAsyncMetricGroupMonitoring(AsyncMetricGroupMonitoring asyncMetricGroupMonitoring) {
        this.asyncMetricGroupMonitoring = asyncMetricGroupMonitoring;
    }

    public void setZookeeperTimeoutSeconds(int zookeeperTimeoutSeconds) {
        this.zookeeperTimeoutSeconds = zookeeperTimeoutSeconds;
    }

    public void setZookeeperPrefix(String zookeeperPrefix) {
        this.zookeeperPrefix = zookeeperPrefix;
    }

    public void setMetricDelaySeconds(int metricDelaySeconds) {
        this.metricDelaySeconds = metricDelaySeconds;
    }


    public void setLightThreadCount(int lightThreadCount) {
        this.lightThreadCount = lightThreadCount;
    }

    public void setMediumThreadCount(int mediumThreadCount) {
        this.mediumThreadCount = mediumThreadCount;
    }

    public void setHeavyThreadCount(int heavyThreadCount) {
        this.heavyThreadCount = heavyThreadCount;
    }

    public void setUpdateIntervalSeconds(int updateIntervalSeconds) {
        this.updateIntervalSeconds = updateIntervalSeconds;
    }

    public void setMaxQueriesPerUpdate(int maxQueriesPerUpdate) {
        this.maxQueriesPerUpdate = maxQueriesPerUpdate;
    }

    public void setFullRebuildMaxDays(int fullRebuildMaxDays) {
        this.fullRebuildMaxDays = fullRebuildMaxDays;
    }

    public void setMaxMetricsAgeOnLagMinutes(int maxMetricsAgeOnLagMinutes) {
        this.maxMetricsAgeOnLagMinutes = maxMetricsAgeOnLagMinutes;
    }

    protected void setLagMode(boolean lagMode) {
        this.lagMode = lagMode;
    }

    public ConfigurationService getConfigurationService() {
        return configurationService;
    }

    public void setExecutionLimitMinutesWarn(int executionLimitMinutesWarn) {
        this.executionLimitMinutesWarn = executionLimitMinutesWarn;
    }

    public void setExecutionLimitMinutesCritical(int executionLimitMinutesCritical) {
        this.executionLimitMinutesCritical = executionLimitMinutesCritical;
    }

    public void setMetaDaoExecutionLimitMinutesCritical(int metaDaoExecutionLimitMinutesCritical) {
        this.mongoDataExecutionLimitMinutesCritical = metaDaoExecutionLimitMinutesCritical;
    }

    public void setHeavyTables(String heavyTables) {
        this.heavyTables = splitStringToSet(heavyTables);
    }

    public void setMediumTables(String mediumTables) {
        this.mediumTables = splitStringToSet(mediumTables);
    }

    private static Set<String> splitStringToSet(String value) {
        return new HashSet<>(Splitter.on(",").trimResults().omitEmptyStrings().splitToList(value));
    }
}
