package ru.yandex.market.clickphite;

import com.google.common.collect.Range;
import com.google.common.collect.RangeSet;
import com.google.common.collect.TreeRangeSet;
import ru.yandex.market.CalendarUtils;
import ru.yandex.market.clickphite.config.metric.MetricPeriod;
import ru.yandex.market.health.OutputInfo;

import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.TimeZone;
import java.util.concurrent.TimeUnit;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 04/03/15
 */
public class DateTimeUtils {

    private static final int MONTHS_IN_QUARTER = 3;
    private static final int FIVE = 5;

    private DateTimeUtils() {
    }

    public static int currentTimeSeconds() {
        return millisToSeconds(System.currentTimeMillis());
    }


    public static int graphiteOffset(int timeSeconds) {
        int offsetSeconds = millisToSeconds(TimeZone.getDefault().getOffset(secondsToMillis(timeSeconds)));
        return timeSeconds + offsetSeconds;
    }

    public static RangeSet<Integer> toRangeSet(List<OutputInfo> outputInfos) {
        RangeSet<Integer> rangeSet = TreeRangeSet.create();
        for (OutputInfo outputInfo : outputInfos) {
            rangeSet.add(outputInfo.getRange());
        }
        return rangeSet;
    }

    public static RangeSet<Integer> toPeriodRangeSet(TimeRange timeRange, MetricPeriod period) {
        RangeSet<Integer> rangeSet = TreeRangeSet.create();
        rangeSet.add(timeRange.getRange());
        return toPeriodRangeSet(rangeSet, period);
    }


    public static RangeSet<Integer> toPeriodRangeSet(RangeSet<Integer> rangeSet, MetricPeriod period) {
        RangeSet<Integer> periodRangeSet = TreeRangeSet.create();
        for (Range<Integer> range : rangeSet.asRanges()) {
            periodRangeSet.add(correctBordersForPeriod(range, period));
        }
        return periodRangeSet;
    }

    public static Range<Integer> correctBordersForPeriod(Range<Integer> range, MetricPeriod period) {
        Date start = toPeriodStart(period, range.lowerEndpoint());
        Date end = toPeriodEnd(period, range.upperEndpoint() - 1); //TODO Cause open
        return Range.closedOpen(dateToSeconds(start), dateToSeconds(end));
    }

    public static TimeRange correctBordersForPeriod(TimeRange timeRange, MetricPeriod period) {
        Range<Integer> range = correctBordersForPeriod(timeRange.getRange(), period);
        return new TimeRange(range.lowerEndpoint(), range.upperEndpoint());
    }

    public static List<TimeRange> sliceToTimeRanges(Range<Integer> range, MetricPeriod period,
                                                    int limit, boolean movingPeriods) {
        if (movingPeriods) {
            return sliceToTimeRangesByPeriod(range, period, limit);
        }
        switch (period) {
            case ONE_SEC:
            case FIVE_SEC:
            case ONE_MIN:
            case FIVE_MIN:
            case HOUR:
                return sliceToTimeRangesByTs(range, MetricPeriod.HOUR.getDurationSeconds(), limit);
            default:
                return sliceToTimeRangesByPeriod(range, period, limit);
        }
    }

    public static List<TimeRange> sliceToTimeRangesByTs(Range<Integer> range, int maxPeriodSeconds, int limit) {
        List<TimeRange> timeRanges = new ArrayList<>();
        int tsSeconds = range.upperEndpoint();
        while (tsSeconds > range.lowerEndpoint() && timeRanges.size() < limit) {
            int periodSize = Math.min(maxPeriodSeconds, tsSeconds - range.lowerEndpoint());
            tsSeconds -= periodSize;
            timeRanges.add(new TimeRange(tsSeconds, tsSeconds + periodSize));
        }
        return timeRanges;
    }

    public static List<TimeRange> sliceToTimeRangesByPeriod(Range<Integer> range, MetricPeriod period, int limit) {
        List<TimeRange> timeRanges = new ArrayList<>();
        int tsSeconds = range.upperEndpoint();
        while (tsSeconds > range.lowerEndpoint() && timeRanges.size() < limit) {
            Date periodStart = toPeriodStart(period, tsSeconds - 1);
            timeRanges.add(new TimeRange(periodStart, dateFromTimeStampSeconds(tsSeconds)));
            tsSeconds = millisToSeconds(periodStart.getTime());
        }
        return timeRanges;

    }

    public static int dateToSeconds(Date date) {
        return millisToSeconds(date.getTime());
    }

    public static int millisToSeconds(long millis) {
        return (int) TimeUnit.MILLISECONDS.toSeconds(millis);
    }

    public static long secondsToMillis(long seconds) {
        return TimeUnit.SECONDS.toMillis(seconds);
    }

    public static TimeRange getTimeRange(MetricPeriod period, int timeStampSeconds) {
        return new TimeRange(toPeriodStart(period, timeStampSeconds), toPeriodEnd(period, timeStampSeconds));
    }

    public static TimeRange getTimeRange(MetricPeriod period, Date date) {
        return new TimeRange(toPeriodStart(period, date), toPeriodEnd(period, date));
    }

    public static Date toPeriodStart(MetricPeriod period, int timeStampSeconds) {
        return toPeriodBorder(period, secondsToMillis(timeStampSeconds), false).getTime();
    }

    public static Date toPeriodStart(MetricPeriod period, Date date) {
        return toPeriodBorder(period, date.getTime(), false).getTime();
    }

    public static Date toPeriodEnd(MetricPeriod period, int timeStampSeconds) {
        return toPeriodBorder(period, secondsToMillis(timeStampSeconds), true).getTime();
    }

    public static Date toPeriodEnd(MetricPeriod period, Date date) {
        return toPeriodBorder(period, date.getTime(), true).getTime();
    }


    public static Date dateFromTimeStampSeconds(int timestampSeconds) {
        return new Date(secondsToMillis(timestampSeconds));
    }

    public static LocalDate toLocalDate(int timestampSeconds) {
        return Instant.ofEpochSecond(timestampSeconds).atZone(ZoneOffset.UTC).toLocalDate();
    }

    public static TimeRange toTimeRange(LocalDate oneDay) {
        LocalDateTime dateTime = oneDay.atTime(0, 0);
        int startTimestampSeconds = (int) dateTime.toEpochSecond(ZoneOffset.UTC);
        int endTimestampSeconds = (int) dateTime.plusDays(1).toEpochSecond(ZoneOffset.UTC);
        return new TimeRange(startTimestampSeconds, endTimestampSeconds);
    }

    private static Calendar toPeriodBorder(MetricPeriod period, long timeStampMillis, boolean up) {
        Calendar calendar = new GregorianCalendar();
        calendar.setTimeInMillis(timeStampMillis);
        switch (period) {
            case ONE_SEC:
                calendar = CalendarUtils.round(calendar, Calendar.SECOND, up);
                break;
            case FIVE_SEC:
                calendar = CalendarUtils.roundToModulo(calendar, Calendar.SECOND, FIVE, up);
                break;
            case ONE_MIN:
                calendar = CalendarUtils.round(calendar, Calendar.MINUTE, up);
                break;
            case FIVE_MIN:
                calendar = CalendarUtils.roundToModulo(calendar, Calendar.MINUTE, FIVE, up);
                break;
            case HOUR:
                calendar = CalendarUtils.round(calendar, Calendar.HOUR, up);
                break;
            case DAY:
                calendar = CalendarUtils.round(calendar, Calendar.DATE, up);
                break;
            case WEEK:
                calendar = CalendarUtils.round(calendar, Calendar.DATE, up);
                while (calendar.get(Calendar.DAY_OF_WEEK) != Calendar.MONDAY) {
                    calendar.add(Calendar.DATE, up ? 1 : -1);
                }
                break;
            case MONTH:
                calendar = CalendarUtils.round(calendar, Calendar.MONTH, up);
                break;
            case QUARTER:
                calendar = CalendarUtils.roundToModulo(calendar, Calendar.MONTH, MONTHS_IN_QUARTER, up);
                break;
            default:
                throw new UnsupportedOperationException("Unknown period: " + period.toString());
        }
        return calendar;
    }
}
