package ru.yandex.market.clickphite;

import com.google.common.collect.Range;
import com.google.common.math.Quantiles;
import com.google.common.util.concurrent.AbstractScheduledService;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.market.clickphite.metric.MetricContext;
import ru.yandex.market.monitoring.ComplicatedMonitoring;
import ru.yandex.market.monitoring.MonitoringUnit;
import ru.yandex.market.request.trace.TskvRecordBuilder;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * @author Ilya Sapachev <a href="mailto:sid-hugo@yandex-team.ru"></a>
 * @date 10.04.17
 */
public class MetricLagService extends AbstractScheduledService implements InitializingBean {

    private ClickphiteService clickphiteService;
    private ComplicatedMonitoring monitoring;

    private int intervalSeconds;
    private int lagMonitoringUnitDelayMinutes = 5;
    private int lagMonitoringQuantile;
    private int lagMonitoringMaxMetricsDelaySeconds;
    private int metricDelaySeconds;

    private MonitoringUnit lagMonitoringUnit;

    private static final Logger log = LogManager.getLogger();
    private static final Logger lagLog = LogManager.getLogger("metric_lag");

    private final DateFormat logDateFormat = new SimpleDateFormat("[dd/MMM/yyyy:HH:mm:ss Z]");

    @Override
    public void afterPropertiesSet() throws Exception {
        lagMonitoringUnit = new MonitoringUnit("lag", lagMonitoringUnitDelayMinutes, TimeUnit.MINUTES);
        monitoring.addUnit(lagMonitoringUnit);
    }

    @Override
    protected void runOneIteration() throws Exception {
        try {
            if (clickphiteService.isMaster()) {
                processMetrics();
                log.info("Successfully logged all lags");
            } else {
                lagMonitoringUnit.ok();
            }
        } catch (Exception e) {
            log.error("Error in MetricLagLoggerService iteration", e);
        }
    }

    private void processMetrics() {
        List<MetricContext> metricContexts = clickphiteService
            .getConfigurationService()
            .getConfiguration()
            .getMetricContexts();

        List<Integer> realTimeLagsSeconds = new ArrayList<>();

        for (MetricContext metricContext : metricContexts) {
            String date = logDateFormat.format(new Date());
            String table = metricContext.getClickHouseTable().getFullName();
            String metricId = metricContext.getId();
            String storage = metricContext.getStorage().name();
            String period = metricContext.getPeriod().name();

            int currentTimeSeconds = (int) TimeUnit.MILLISECONDS.toSeconds(System.currentTimeMillis());
            QueueMetrics fullMetrics = getQueueMetrics(
                metricContext.getMetricQueue().get(currentTimeSeconds),
                metricContext.getPeriod().getDurationSeconds(),
                currentTimeSeconds
            );

            QueueMetrics realTimeMetrics = getQueueMetrics(
                metricContext.getMetricQueue().getAfterMaxProcessed(currentTimeSeconds),
                metricContext.getPeriod().getDurationSeconds(),
                currentTimeSeconds
            );

            realTimeLagsSeconds.add(Math.max(realTimeMetrics.getLagSeconds() - metricDelaySeconds, 0));

            TskvRecordBuilder tskvRecordBuilder = new TskvRecordBuilder();
            tskvRecordBuilder.add("date", date);
            tskvRecordBuilder.add("table", table);
            tskvRecordBuilder.add("metricId", metricId);
            tskvRecordBuilder.add("storage", storage);
            tskvRecordBuilder.add("period", period);
            tskvRecordBuilder.add("fullLagSeconds", fullMetrics.getLagSeconds());
            tskvRecordBuilder.add("fullQueuePeriods", fullMetrics.getQueuePeriods());
            tskvRecordBuilder.add("fullQueueSeconds", fullMetrics.getQueueSeconds());
            tskvRecordBuilder.add("realTimeLagSeconds", realTimeMetrics.getLagSeconds());
            tskvRecordBuilder.add("realTimeQueuePeriods", realTimeMetrics.getQueuePeriods());
            tskvRecordBuilder.add("realTimeQueueSeconds", realTimeMetrics.getQueueSeconds());

            lagLog.info(
                tskvRecordBuilder.build()
            );
        }
        checkDelayMode(realTimeLagsSeconds);

    }

    private void checkDelayMode(List<Integer> realTimeLagsSeconds) {
        int delaySeconds = (int) Quantiles.percentiles().index(lagMonitoringQuantile).compute(realTimeLagsSeconds);
        if (lagMonitoringMaxMetricsDelaySeconds > 0 && delaySeconds > lagMonitoringMaxMetricsDelaySeconds) {
            String message = "Metrics lag (q" + lagMonitoringQuantile + ", since metric-delay) is " + delaySeconds +
                " seconds (more than " + lagMonitoringMaxMetricsDelaySeconds + " seconds)";
            log.warn(message);
            lagMonitoringUnit.warning(message);
            clickphiteService.setLagMode(true);
        } else {
            log.info(
                "Metrics lag (q" + lagMonitoringQuantile + ", since metric-delay) is " + delaySeconds + " seconds"
            );
            lagMonitoringUnit.ok();
            clickphiteService.setLagMode(false);
        }
    }

    @Override
    protected Scheduler scheduler() {
        return Scheduler.newFixedDelaySchedule(intervalSeconds, intervalSeconds, TimeUnit.SECONDS);
    }

    public static QueueMetrics getQueueMetrics(List<Range<Integer>> queue, int metricPeriodSeconds,
                                               int currentTimeSeconds) {
        int lagSeconds = 0;
        int queuePeriods = 0;
        int queueSeconds = 0;
        for (Range<Integer> range : queue) {
            int rangeDurationSeconds = range.upperEndpoint() - range.lowerEndpoint();
            queuePeriods += rangeDurationSeconds / metricPeriodSeconds;
        }
        if (queuePeriods > 0) {
            lagSeconds = currentTimeSeconds - (queue.get(queue.size() - 1).lowerEndpoint() + metricPeriodSeconds);
            queueSeconds = queuePeriods * metricPeriodSeconds;
        }
        return new QueueMetrics(lagSeconds, queuePeriods, queueSeconds);
    }

    static class QueueMetrics {
        private final int lagSeconds;
        private final int queuePeriods;
        private final int queueSeconds;

        private QueueMetrics(int lagSeconds, int queuePeriods, int queueSeconds) {
            this.lagSeconds = lagSeconds;
            this.queuePeriods = queuePeriods;
            this.queueSeconds = queueSeconds;
        }

        int getLagSeconds() {
            return lagSeconds;
        }

        int getQueuePeriods() {
            return queuePeriods;
        }

        int getQueueSeconds() {
            return queueSeconds;
        }
    }


    @Required
    public void setIntervalSeconds(int intervalSeconds) {
        this.intervalSeconds = intervalSeconds;
    }

    @Required
    public void setClickphiteService(ClickphiteService clickphiteService) {
        this.clickphiteService = clickphiteService;
    }

    @Required
    public void setMonitoring(ComplicatedMonitoring monitoring) {
        this.monitoring = monitoring;
    }

    public void setLagMonitoringUnitDelayMinutes(int lagMonitoringUnitDelayMinutes) {
        this.lagMonitoringUnitDelayMinutes = lagMonitoringUnitDelayMinutes;
    }

    @Required
    public void setLagMonitoringQuantile(int lagMonitoringQuantile) {
        this.lagMonitoringQuantile = lagMonitoringQuantile;
    }

    @Required
    public void setLagMonitoringMaxMetricsDelaySeconds(int lagMonitoringMaxMetricsDelaySeconds) {
        this.lagMonitoringMaxMetricsDelaySeconds = lagMonitoringMaxMetricsDelaySeconds;
    }

    @Required
    public void setMetricDelaySeconds(int metricDelaySeconds) {
        this.metricDelaySeconds = metricDelaySeconds;
    }
}
