package ru.yandex.market.clickphite;

import com.google.common.base.Strings;
import ru.yandex.market.clickphite.config.dashboard.DashboardConfig;
import ru.yandex.market.clickphite.config.metric.AbstractMetricConfig;
import ru.yandex.market.clickphite.config.metric.GraphiteMetricConfig;
import ru.yandex.market.clickphite.config.metric.MetricField;
import ru.yandex.market.clickphite.config.metric.MetricSplit;
import ru.yandex.market.clickphite.config.metric.SubAggregateConfig;
import ru.yandex.market.clickphite.config.metric.SubAggregateExpression;
import ru.yandex.market.clickphite.dashboard.DashboardQueries;
import ru.yandex.market.clickphite.metric.MetricQueries;

import java.util.Collections;
import java.util.List;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 12/12/14
 */
public class QueryBuilder {

    //TODO change all String tableName to Clickhouse table

    public static final String TIMESTAMP_ALIAS = "metric_ts";

    public static final String UNREAL_PERIOD = "timestamp = 0 and date = toDate(42000)";
    public static final String PERIOD_VARIABLE = "${CP_PERIOD}";
    public static final String TIMESTAMP_VARIABLE = "${CP_TIMESTAMP}";

    private static final int DASHBOARD_UPDATE_INTERVAL_DAYS = 1;
    private static final List<String> DASHBOARD_DEFAULT_SORT = Collections.singletonList("count() desc");

    public static final int TIMESTAMP_INDEX = 1;
    public static final int VALUE_INDEX = 2;

    private QueryBuilder() {
    }

    public static MetricQueries buildQueries(Queriable queriable) {
        return new MetricQueries(
            buildMetricQueryTemplate(queriable, PERIOD_VARIABLE)
        );
    }

    public static String buildMetricQueryTemplate(Queriable queriable) {
        return buildMetricQueryTemplate(queriable, PERIOD_VARIABLE);
    }

    public static String buildMetricQueryTemplate(Queriable queriable, String period) {
        return buildMetricQueryTemplate(queriable, period, "", TIMESTAMP_VARIABLE);
    }

    public static String buildMetricQueryTemplate(Queriable queriable, String period,
                                                  String tablePostfix, String timestampVariable) {
        boolean containsSubQuery = queriable.getSubAggregate() != null;

        StringBuilder stringBuilder = new StringBuilder();

        //SELECT
        stringBuilder.append("SELECT ");

        String metricTsField;
        if (queriable.getMovingWindowPeriods() > 1) {
            metricTsField = timestampVariable + " AS " + TIMESTAMP_ALIAS;
        } else {
            metricTsField = queriable.getPeriod().getClickHouseFunction() + " AS " + TIMESTAMP_ALIAS;
        }

        if (containsSubQuery) {
            stringBuilder.append(TIMESTAMP_ALIAS);
        } else {
            stringBuilder.append(metricTsField);
        }

        List<AbstractMetricConfig<?>> metricConfigs = queriable.getMetricConfigs();
        for (int i = 0; i < metricConfigs.size(); i++) {
            final int metricIndex = i;
            AbstractMetricConfig<?> metricConfig = metricConfigs.get(metricIndex);

            List<? extends MetricField> fields = metricConfig.getFields();

            List<FieldAliasReplacer> aliasReplacers = fields.stream()
                .map(f -> FieldAliasReplacer.fromMetricField(f, metricIndex))
                .collect(Collectors.toList());

            for (MetricField field : fields) {
                String metricFieldExpression = getMetricField(field, metricConfig.getFilter());

                for (FieldAliasReplacer aliasReplacer : aliasReplacers) {
                    if (!aliasReplacer.field.equals(field)) {
                        metricFieldExpression = aliasReplacer.replaceFieldAlias(metricFieldExpression);
                    }
                }

                stringBuilder.append(", ").append(metricFieldExpression)
                    .append(" AS ").append(field.getClickHouseName())
                    .append("_").append(i);
            }
        }

        for (MetricField split : queriable.getSplits()) {
            stringBuilder.append(", ").append(split.getClickHouseName());
        }

        stringBuilder.append(" ");

        //FROM
        CharSequence tableQuery;

        String tableFullName = queriable.getTable().getFullName() + tablePostfix;

        if (containsSubQuery) {
            StringBuilder subSelect = new StringBuilder("( SELECT ")
                .append(metricTsField);

            SubAggregateConfig subAggregate = queriable.getSubAggregate();

            // SELECT
            for (String key : subAggregate.getKeys()) {
                subSelect.append(", ").append(key);
            }

            for (SubAggregateExpression aggregateExpression : subAggregate.getAggregateExpressions()) {
                subSelect.append(", ")
                    .append(aggregateExpression.getExpression())
                    .append(" as ")
                    .append(aggregateExpression.getName());
            }

            //FROM
            subSelect.append(" FROM ").append(tableFullName);

            //WHERE
            subSelect.append(" WHERE ").append(period);

            subSelect.append(" GROUP BY ").append(TIMESTAMP_ALIAS);

            //GROUP BY
            for (String key : subAggregate.getKeys()) {
                subSelect.append(", ").append(key);
            }

            subSelect.append(" )");

            tableQuery = subSelect;
        } else {
            tableQuery = tableFullName;
        }
        stringBuilder.append("FROM ").append(tableQuery);

        //WHERE
        stringBuilder.append(" WHERE 1");
        if (!containsSubQuery) {
            stringBuilder.append(" AND ").append(period);
        }
        if (queriable.getFilter() != null) {
            stringBuilder.append(" AND (").append(queriable.getFilter()).append(")");
        }

        //GROUP BY
        stringBuilder.append(" GROUP BY ").append(TIMESTAMP_ALIAS).append(",");

        List<? extends MetricField> splits = queriable.getSplits();
        for (MetricField split : splits) {
            stringBuilder.append(" ").append(split.getField());
            stringBuilder.append(" AS ").append(split.getClickHouseName()).append(",");
        }

        stringBuilder.deleteCharAt(stringBuilder.length() - 1);
        stringBuilder.append(" ");

        //ORDER BY
        stringBuilder.append("ORDER BY ").append(TIMESTAMP_ALIAS);

        return stringBuilder.toString();
    }

    private static class FieldAliasReplacer {
        private final Pattern pattern;
        private final MetricField field;
        private final int metricIndex;

        private FieldAliasReplacer(Pattern pattern, MetricField field, int metricIndex) {
            this.pattern = pattern;
            this.field = field;
            this.metricIndex = metricIndex;
        }

        static FieldAliasReplacer fromMetricField(MetricField field, int metricIndex) {
            return new FieldAliasReplacer(
                // lookahead за скобкой, чтобы случайно не зареплейсить функцию.
                // например, когда кто-то сделал алиас sum
                // поле не должно быть в кавычках, так как в этом случае - это буквальное значение
                Pattern.compile("\\b(?!')" + field.getClickHouseName() + "(?!')\\b(?!\\s*\\()"), field, metricIndex
            );
        }

        String replaceFieldAlias(String expression) {
            String actualFieldName = field.getClickHouseName() + "_" + metricIndex;
            return pattern.matcher(expression).replaceAll(actualFieldName);
        }
    }

    public static DashboardQueries buildDashboardQueries(DashboardConfig dashboardConfig,
                                                         GraphiteMetricConfig metricConfig) {
        StringBuilder query = new StringBuilder();

        //SELECT
        query.append("SELECT");
        List<MetricSplit> splits = metricConfig.getSplits();
        if (splits.isEmpty()) {
            throw new IllegalStateException("No splits for dashboard: " + dashboardConfig.getId());
        }
        for (MetricSplit split : splits) {
            query.append(" ").append(split.getClickHouseName()).append(",");
        }
        query.deleteCharAt(query.length() - 1);
        query.append(" ");

        //FROM
        query.append("FROM ").append(metricConfig.getTable().getFullName()).append(" ");

        //WHERE
        query.append("WHERE ");
        query.append("date >= today() - ").append(DASHBOARD_UPDATE_INTERVAL_DAYS).append(" and ");
        query.append(
            "timestamp >= toUInt32(toDateTime(today() - ").append(DASHBOARD_UPDATE_INTERVAL_DAYS).append(")) "
        );

        if (metricConfig.getFilter() != null) {
            query.append("AND (").append(metricConfig.getFilter()).append(") ");
        }

        //GROUP BY
        query.append("GROUP BY");
        for (MetricSplit split : splits) {
            query.append(" ").append(split.getField()).append(" AS ").append(split.getClickHouseName()).append(",");
        }
        query.deleteCharAt(query.length() - 1);
        query.append(" ");

        //HAVING
        query.append("HAVING ");

        for (int i = 0; i < splits.size(); i++) {
            MetricSplit split = splits.get(i);
            if (i > 0) {
                query.append("AND ");
            }
            query.append("notEmpty(toString(").append(split.getClickHouseName()).append(")) ");
        }

        //ORDER
        query.append("ORDER BY");
        List<String> sorts = dashboardConfig.getSort();
        if (sorts == null || sorts.isEmpty()) {
            sorts = DASHBOARD_DEFAULT_SORT;
        }
        for (String sort : sorts) {
            query.append(" ").append(sort).append(",");
        }
        query.deleteCharAt(query.length() - 1);

        return new DashboardQueries(query.toString());
    }

    public static String placeTimeConditionToQuery(String queryTemplate, TimeRange timeRange, int movingWindowPeriods) {
        int startSeconds = timeRange.getStartTimestampSeconds();
        if (movingWindowPeriods > 1) {
            startSeconds -= timeRange.getDurationSeconds() * (movingWindowPeriods - 1);
            queryTemplate = queryTemplate.replace(
                TIMESTAMP_VARIABLE,
                String.valueOf(timeRange.getStartTimestampSeconds())
            );

        }
        return queryTemplate.replace(
            PERIOD_VARIABLE,
            buildTimeCondition(startSeconds, timeRange.getEndTimestampSeconds())
        );
    }

    private static String buildTimeCondition(int startSeconds, int endSeconds) {
        String condition = "timestamp >= " + startSeconds +
            " and timestamp <" + endSeconds +
            " and date >= toDate(toDateTime(" + startSeconds + "))" +
            " and date <= toDate(toDateTime(" + endSeconds + "))";
        return condition;
    }

    private static String getQuantilesMetricField(MetricField field, String filter) {
        StringBuilder builder = new StringBuilder();
        switch (field.getType()) {
            case QUANTILE:
                if (!Strings.isNullOrEmpty(filter)) {
                    builder.append("quantilesIf(");
                } else {
                    builder.append("quantiles(");
                }
                break;
            case QUANTILE_TIMING:
                if (!Strings.isNullOrEmpty(filter)) {
                    builder.append("quantilesTimingIf(");
                } else {
                    builder.append("quantilesTiming(");
                }
                break;
            case QUANTILE_TIMING_WEIGHTED:
                if (!Strings.isNullOrEmpty(filter)) {
                    builder.append("quantilesTimingWeightedIf(");
                } else {
                    builder.append("quantilesTimingWeighted(");
                }
                break;
            default:
                throw new IllegalStateException();
        }
        for (String quantile : field.getQuantiles()) {
            builder.append(quantile).append(",");
        }
        builder.deleteCharAt(builder.length() - 1).append(")");
        builder.append("(").append(field.getField());

        if (!Strings.isNullOrEmpty(filter)) {
            builder.append(", ").append(filter);
        }

        builder.append(")");
        return builder.toString();
    }


    private static String getMetricField(MetricField field, String filter) {
        switch (field.getType()) {
            case SIMPLE:
                return field.getField();
            case QUANTILE:
            case QUANTILE_TIMING:
            case QUANTILE_TIMING_WEIGHTED:
                return getQuantilesMetricField(field, filter);
            default:
                throw new UnsupportedOperationException();
        }
    }
}
