package ru.yandex.market.clickphite;

import com.google.common.collect.ComparisonChain;
import com.google.common.collect.Range;

import java.util.Date;
import java.util.Objects;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 04/03/15
 */
public class TimeRange implements Comparable<TimeRange> {

    private final Range<Integer> range;
    private volatile Date startDate;
    private volatile Date endDate;

    public TimeRange(int startTimestampSeconds, int endTimestampSeconds) {
        range = Range.closedOpen(startTimestampSeconds, endTimestampSeconds);
    }

    public TimeRange(Date start, Date end) {
        this(DateTimeUtils.millisToSeconds(start.getTime()), DateTimeUtils.millisToSeconds(end.getTime()));
        startDate = start;
        endDate = end;
    }

    private void ensureDates() {
        if (startDate == null || endDate == null) {
            startDate = DateTimeUtils.dateFromTimeStampSeconds(range.lowerEndpoint());
            endDate = DateTimeUtils.dateFromTimeStampSeconds(range.upperEndpoint());
        }
    }

    public Range<Integer> getRange() {
        return range;
    }

    public int getStartTimestampSeconds() {
        return range.lowerEndpoint();
    }

    public int getEndTimestampSeconds() {
        return range.upperEndpoint();
    }

    public int getDurationSeconds() {
        return range.upperEndpoint() - range.lowerEndpoint();
    }

    public Date getStartDate() {
        ensureDates();
        return startDate;
    }

    public Date getEndDate() {
        ensureDates();
        return endDate;
    }

    public boolean contains(int timestampSeconds) {
        return range.contains(timestampSeconds);
    }

    public boolean contains(TimeRange o) {
        return range.encloses(o.range);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        TimeRange timeRange = (TimeRange) o;
        return Objects.equals(range, timeRange.range);
    }

    @Override
    public int hashCode() {
        return Objects.hash(range);
    }

    @Override
    public String toString() {
        ensureDates();
        return "TimeRange{" +
            "start=" + startDate +
            ", end=" + endDate +
            '}';
    }

    /*
    Сначала по твремени начала, потом по длинне наоборот
     */
    @Override
    public int compareTo(TimeRange o) {
        return ComparisonChain.start()
            .compare(range.lowerEndpoint(), o.range.lowerEndpoint())
            .compare(o.range.upperEndpoint(), range.upperEndpoint())
            .result();
    }
}
