package ru.yandex.market.clickphite.config;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.ListMultimap;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import ru.yandex.market.clickphite.ClickHouseTable;
import ru.yandex.market.clickphite.config.metric.AbstractMetricConfig;
import ru.yandex.market.clickphite.config.metric.MetricField;
import ru.yandex.market.clickphite.config.metric.MetricPeriod;
import ru.yandex.market.clickphite.config.metric.MetricType;
import ru.yandex.market.clickphite.config.metric.SolomonSensorConfig;
import ru.yandex.market.clickphite.config.metric.SubAggregateConfig;
import ru.yandex.market.clickphite.dashboard.DashboardContext;
import ru.yandex.market.clickphite.metric.MetricContext;
import ru.yandex.market.clickphite.metric.MetricContextGroup;
import ru.yandex.market.clickphite.metric.MetricContextGroupImpl;
import ru.yandex.market.clickphite.metric.MetricStorage;
import ru.yandex.market.clickphite.monitoring.MonitoringContext;
import ru.yandex.market.clickphite.monitoring.MonitoringType;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 19/01/15
 */
public class ClickphiteConfiguration {
    private static final Logger log = LogManager.getLogger();

    private static final Pattern ARRAY_JOIN_PATTERN = Pattern.compile("(^|[^a-zA-Z])arrayJoin\\s*\\(");

    private final Map<String, ConfigFile> configFiles;

    private final List<MetricContext> metricContexts;
    private final List<DashboardContext> dashboardContexts;
    private final List<MonitoringContext> monitoringContexts;

    private final ListMultimap<ClickHouseTable, MetricContext> tableToMetricContexts;
    private final Map<ClickHouseTable, ListMultimap<MetricPeriod, MetricContext>> tableToMetricsByPeriod;
    private final Map<String, ClickHouseTable> clickHouseTables;
    private final Map<String, MetricContext> metricContextsById;

    private final ListMultimap<MonitoringType, MonitoringContext> monitoringsByType;
    private final List<MetricContextGroup> metricContextGroups;

    ClickphiteConfiguration(Map<String, ConfigFile> configFiles, List<MetricContext> metricContexts) {
        this.configFiles = configFiles;
        this.metricContexts = metricContexts;
        this.dashboardContexts = new ArrayList<>();
        this.monitoringContexts = new ArrayList<>();
        this.metricContextsById = new HashMap<>();
        for (MetricContext metricContext : metricContexts) {
            dashboardContexts.addAll(metricContext.getDashboardContexts());
            monitoringContexts.addAll(metricContext.getMonitoringContexts());
            metricContextsById.put(metricContext.getId(), metricContext);
        }
        tableToMetricContexts = ArrayListMultimap.create();
        tableToMetricsByPeriod = new HashMap<>();
        clickHouseTables = new HashMap<>();
        for (MetricContext metricContext : metricContexts) {
            ClickHouseTable table = metricContext.getClickHouseTable();
            clickHouseTables.put(table.getFullName(), table);
            tableToMetricContexts.put(table, metricContext);
            ListMultimap<MetricPeriod, MetricContext> byPeriod = tableToMetricsByPeriod.get(table);
            if (byPeriod == null) {
                byPeriod = ArrayListMultimap.create();
                tableToMetricsByPeriod.put(table, byPeriod);
            }
            byPeriod.put(metricContext.getPeriod(), metricContext);
        }

        monitoringsByType = ArrayListMultimap.create();
        for (MonitoringContext monitoringContext : monitoringContexts) {
            monitoringsByType.put(monitoringContext.getType(), monitoringContext);
        }

        metricContextGroups = groupMetricContexts(metricContexts);
    }

    public ClickHouseTable getClickHouseTable(String name) {
        return clickHouseTables.get(name);
    }

    public List<MetricContext> getMetricsForTable(ClickHouseTable table) {
        return tableToMetricContexts.get(table);
    }

    public ListMultimap<MetricPeriod, MetricContext> getMetricsForTableByPeriod(ClickHouseTable table) {
        return tableToMetricsByPeriod.get(table);
    }

    public ListMultimap<ClickHouseTable, MetricContext> getTableToMetricContexts() {
        return tableToMetricContexts;
    }

    public ListMultimap<MonitoringType, MonitoringContext> getMonitoringsByType() {
        return monitoringsByType;
    }

    public List<MonitoringContext> getMonitoringContexts() {
        return monitoringContexts;
    }

    public List<MetricContext> getMetricContexts() {
        return metricContexts;
    }

    public MetricContext getMetricContext(String id) {
        return metricContextsById.get(id);
    }

    public List<DashboardContext> getDashboardContexts() {
        return dashboardContexts;
    }

    Map<String, ConfigFile> getConfigFiles() {
        return configFiles;
    }

    public List<MetricContextGroup> getMetricContextGroups() {
        return metricContextGroups;
    }

    private static List<MetricContextGroup> groupMetricContexts(List<MetricContext> metricContexts) {
        return metricContexts.stream()
            .collect(Collectors.groupingBy(MetricContextGroupKey::fromMetricContext))
            .values()
            .stream()
            .map(MetricContextGroupImpl::create)
            .collect(Collectors.toList());
    }

    static class MetricContextGroupKey {
        private final String table;
        private final Set<String> splitFields;
        private final MetricPeriod period;
        private final String filter;
        private final SubAggregateConfig subAggregateConfig;
        private final MetricType metricType;
        private final MetricStorage storage;
        private final Set<String> expressionsUnderArrayJoin;
        private final String solomonProject;
        private final int movingWindowPeriods;

        @SuppressWarnings("checkstyle:parameternumber")
        private MetricContextGroupKey(String table, List<? extends MetricField> splits, MetricPeriod period,
                                      String filter, SubAggregateConfig subAggregateConfig,
                                      MetricType metricType, MetricStorage storage,
                                      Set<String> expressionsUnderArrayJoin,
                                      String solomonProject, int movingWindowPeriods) {
            this.table = table;
            this.splitFields = splits.stream().map(MetricField::getField).collect(Collectors.toSet());
            this.period = period;
            this.filter = filter;
            this.subAggregateConfig = subAggregateConfig;
            this.metricType = metricType;
            this.storage = storage;
            this.expressionsUnderArrayJoin = expressionsUnderArrayJoin;
            this.solomonProject = solomonProject;
            this.movingWindowPeriods = movingWindowPeriods;
        }

        static MetricContextGroupKey fromMetricContext(MetricContext metricContext) {
            AbstractMetricConfig<?> config = metricContext.getMetricConfig();

            Set<String> expressionsUnderArrayJoin = config.getFields().stream()
                .map(MetricField::getField)
                .map(ClickphiteConfiguration::extractExpressionsUnderArrayJoin)
                .flatMap(Collection::stream)
                .collect(Collectors.toSet());

            // st/MARKETINFRA-3640.
            // Не объединяем соломоновые сенсоры разных проектов чтобы ошибки в одном проекте не ломали другие проекты.
            String solomonProject = config instanceof SolomonSensorConfig
                ? ((SolomonSensorConfig) config).getLabels().get("project")
                : null;

            return new MetricContextGroupKey(
                config.getTableName(), config.getSplits(), metricContext.getPeriod(), config.getFilter(),
                config.getSubAggregate(), config.getType(), config.getStorage(), expressionsUnderArrayJoin,
                solomonProject, config.getMovingWindowPeriods()
            );
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }

            MetricContextGroupKey that = (MetricContextGroupKey) o;

            return Objects.equals(table, that.table) &&
                Objects.equals(splitFields, that.splitFields) &&
                period == that.period &&
                Objects.equals(filter, that.filter) &&
                Objects.equals(subAggregateConfig, that.subAggregateConfig) &&
                storage == that.storage &&
                Objects.equals(expressionsUnderArrayJoin, that.expressionsUnderArrayJoin) &&
                Objects.equals(solomonProject, that.solomonProject) &&
                movingWindowPeriods == that.movingWindowPeriods;
        }

        @Override
        public int hashCode() {
            return Objects.hash(
                table, splitFields, period, filter, subAggregateConfig,
                expressionsUnderArrayJoin, solomonProject
            );
        }
    }

    @VisibleForTesting
    protected static List<String> extractExpressionsUnderArrayJoin(String expression) {
        List<String> expressionsUnderArrayJoin = new ArrayList<>();

        Matcher matcher = ARRAY_JOIN_PATTERN.matcher(expression);

        while (matcher.find()) {
            int openParenthesisIndex = matcher.end() - 1;
            String expressionUnderParenthesis = extractExpressionUnderParenthesis(expression, openParenthesisIndex);
            if (expressionUnderParenthesis == null) {
                break;
            }

            expressionsUnderArrayJoin.add(expressionUnderParenthesis);
        }

        return expressionsUnderArrayJoin;
    }

    private static String extractExpressionUnderParenthesis(String expression, int openParenthesisIndex) {
        int closeParenthesisIndex = -1;
        int parenthesisCounter = 0;

        for (int i = openParenthesisIndex; i < expression.length(); ++i) {
            if (expression.charAt(i) == '(') {
                ++parenthesisCounter;
            }
            if (expression.charAt(i) == ')') {
                --parenthesisCounter;
            }

            if (parenthesisCounter == 0) {
                closeParenthesisIndex = i;
                break;
            }
        }

        if (closeParenthesisIndex == -1) {
            log.warn(
                "No matching closing parenthesis found for open one on position {} in expression {}",
                openParenthesisIndex, expression
            );
            return null;
        }

        return expression.substring(openParenthesisIndex + 1, closeParenthesisIndex).trim();
    }
}
