package ru.yandex.market.clickphite.config;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.github.fge.jackson.JacksonUtils;
import com.github.fge.jsonschema.core.report.ProcessingReport;
import com.github.fge.jsonschema.core.util.AsJson;
import com.github.fge.jsonschema.main.JsonSchemaFactory;
import com.github.fge.jsonschema.main.JsonValidator;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.MoreObjects;
import com.google.common.base.Stopwatch;
import com.google.common.base.Strings;
import com.google.common.collect.ListMultimap;
import com.google.common.collect.Multimaps;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.reflect.TypeToken;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.common.util.collections.MultiMap;
import ru.yandex.market.clickphite.ClickHouseTable;
import ru.yandex.market.clickphite.QueryBuilder;
import ru.yandex.market.clickphite.config.dashboard.DashboardConfig;
import ru.yandex.market.clickphite.config.global.GlobalConfig;
import ru.yandex.market.clickphite.config.metric.AbstractMetricConfig;
import ru.yandex.market.clickphite.config.metric.GraphiteMetricConfig;
import ru.yandex.market.clickphite.config.metric.MetricPeriod;
import ru.yandex.market.clickphite.config.metric.MetricSplit;
import ru.yandex.market.clickphite.config.metric.MetricType;
import ru.yandex.market.clickphite.config.metric.SolomonSensorConfig;
import ru.yandex.market.clickphite.config.metric.StatfaceReportConfig;
import ru.yandex.market.clickphite.config.monitoring.MonitoringConfig;
import ru.yandex.market.clickphite.config.validation.config.MetricContextGroupValidationError;
import ru.yandex.market.clickphite.config.validation.config.MetricContextGroupValidator;
import ru.yandex.market.clickphite.config.validation.context.ConfigValidationException;
import ru.yandex.market.clickphite.config.validation.context.ConfigValidator;
import ru.yandex.market.clickphite.config.validation.context.SolomonSensorConfigValidator;
import ru.yandex.market.clickphite.dashboard.DashboardContext;
import ru.yandex.market.clickphite.dashboard.DashboardQueries;
import ru.yandex.market.clickphite.metric.GraphiteMetricContext;
import ru.yandex.market.clickphite.metric.MetricContext;
import ru.yandex.market.clickphite.metric.MetricContextGroup;
import ru.yandex.market.clickphite.metric.StatfaceReportContext;
import ru.yandex.market.clickphite.metric.StringTemplate;
import ru.yandex.market.clickphite.metric.solomon.SolomonSensorContext;
import ru.yandex.market.clickphite.monitoring.MonitoringContext;
import ru.yandex.market.clickphite.monitoring.MonitoringService;
import ru.yandex.market.clickphite.monitoring.kronos.KronosMonitoringContext;
import ru.yandex.market.clickphite.monitoring.range.RangeMonitoringContext;
import ru.yandex.market.monitoring.ComplicatedMonitoring;
import ru.yandex.market.monitoring.MonitoringUnit;
import ru.yandex.market.statface.StatfaceField;

import java.io.File;
import java.io.IOException;
import java.io.StringReader;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.WeakHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 17/01/15
 */
public class ConfigurationService implements InitializingBean {

    private static final Logger log = LogManager.getLogger();

    private static final String VALIDATION_SCHEMA_PATH = "ClickphiteConfigSchema.json";
    private static final JsonNode VALIDATION_SCHEMA;
    private static final int DEFAULT_STATFACE_PRECISION = 3;

    private final Map<String, ClickHouseTable> clickHouseTables = new WeakHashMap<>();
    private ClickphiteConfiguration conf;

    private MonitoringService monitoringService;
    private File configDir;
    private ConfigValidator configValidator;
    private MetricContextGroupValidator metricContextGroupValidator;
    private String graphiteMetricPrefix = "";
    private String graphiteGlobalMetricPrefix = "";
    private String dashboardGraphiteDataSource;
    private String dashboardPrefix = "";
    private String defaultDatabase;
    private Set<String> dashboardDefaultTags = Collections.emptySet();
    private String solomonProjectOverride;
    private ComplicatedMonitoring monitoring;
    private double failedConfigsPercentToCrit = 10;
    private double failedMetricGroupsPercentToCrit = 10;
    private int fileReadersThreadCount = 10;
    private int failedConfigsRetryMinutes = 5;
    private long lastFailedConfigCheckMillis = 0;
    private final Set<ConfigFile> invalidFiles = new HashSet<>();
    private final Stopwatch metricContextGroupsValidationStopwatch = Stopwatch.createStarted();
    private List<MetricContextGroup> metricContextGroupsToValidate;

    private final MonitoringUnit configUnit = new MonitoringUnit("Config");
    private final MonitoringUnit metricGroupValidationUnit = new MonitoringUnit("MetricGroupValidation");

    private final Gson gson = new GsonBuilder().create();

    @Override
    public void afterPropertiesSet() throws Exception {
        monitoring.addUnit(configUnit);
        monitoring.addUnit(metricGroupValidationUnit);
    }

    static {
        try {
            String schema = IOUtils.toString(ClassLoader.getSystemResourceAsStream(VALIDATION_SCHEMA_PATH));
            VALIDATION_SCHEMA = getNodeData(schema);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private static JsonNode getNodeData(String data) throws IOException {
        final JsonFactory jsonFactory = new JsonFactory();
        jsonFactory.enable(JsonParser.Feature.ALLOW_COMMENTS);

        return new ObjectMapper(jsonFactory).readTree(new StringReader(data));
    }

    public synchronized ClickphiteConfiguration getConfiguration() {
        if (conf == null) {
            reload();
        }

        if (failedConfigsUpdated()) {
            updateConf(conf.getConfigFiles());
        }

        return conf;
    }

    public synchronized void reload() {
        processConfigDirectory();

        if (needToRevalidateMetricContextGroups()) {
            revalidateMetricContextGroups();
        }
    }

    public void resetConfigMonitoring() {
        configUnit.ok();
        metricGroupValidationUnit.ok();
    }

    private void processConfigDirectory() {
        final List<Future<ConfigFile>> configReaders = new ArrayList<>();
        final ExecutorService readersService = Executors.newFixedThreadPool(fileReadersThreadCount);

        boolean configFilesUpdated = false;
        Map<String, ConfigFile> knownFiles;
        if (conf == null) {
            configFilesUpdated = true;
            knownFiles = Collections.emptyMap();
        } else {
            knownFiles = conf.getConfigFiles();
        }
        File[] files = getCurrentConfigFiles();
        Map<String, ConfigFile> actualFiles = new LinkedHashMap<>();
        for (File file : files) {
            final ConfigFile configFile = knownFiles.getOrDefault(file.getPath(), new ConfigFile(file));
            if (file.lastModified() != configFile.getKnownModificationTime()) {
                if (configFile.getKnownModificationTime() != -1) {
                    log.info(
                        "Config file configFilesUpdated: " + configFile.getName() +
                            ". Known time: " + configFile.getKnownModificationTime() +
                            ", actual: " + file.lastModified()
                    );
                } else {
                    log.info("Found new config file: " + configFile.getName());
                }
                configReaders.add(readersService.submit(() -> processConfigFile(configFile)));
                configFilesUpdated = true;
            }
            actualFiles.put(configFile.getName(), configFile);
        }

        for (Future<ConfigFile> future : configReaders) {
            try {
                future.get();
            } catch (InterruptedException | ExecutionException e) {
                log.error("Config file reading failed", e);
            }
        }
        readersService.shutdown();

        for (String knownConfigName : knownFiles.keySet()) {
            if (!actualFiles.containsKey(knownConfigName)) {
                configFilesUpdated = true;
                log.info("Config file removed: " + knownConfigName);
            }
        }

        if (configFilesUpdated) {
            checkSimilarMetricIds(actualFiles.values());
            checkSimilarDashboardIds(actualFiles.values());
        }

        if (failedConfigsUpdated() || configFilesUpdated) {
            updateConf(actualFiles);
        } else {
            log.info("Configuration is up to date. Processed " + files.length + " config files.");
        }
    }

    private boolean failedConfigsUpdated() {
        final long millisFromLastReload = System.currentTimeMillis() - lastFailedConfigCheckMillis;

        if (millisFromLastReload < TimeUnit.MINUTES.toMillis(failedConfigsRetryMinutes)) {
            return false;
        }

        boolean updated = false;
        final Iterator<ConfigFile> iterator = invalidFiles.iterator();
        while (iterator.hasNext()) {
            final ConfigFile configFile = processConfigFile(iterator.next());
            if (configFile.isValid()) {
                updated = true;
                iterator.remove();
            }
        }

        lastFailedConfigCheckMillis = System.currentTimeMillis();

        return updated;
    }

    private void updateConf(Map<String, ConfigFile> configFiles) {
        List<MetricContext> metricContexts = new ArrayList<>();
        for (ConfigFile configFile : configFiles.values()) {
            if (configFile.isValid()) {
                metricContexts.addAll(configFile.getMetricContexts());
            } else {
                invalidFiles.add(configFile);
            }
        }
        //TODO
//        metaDao.loadMetricsInfo(metricContexts);
        log.info(
            "Configuration updated. Processed " + configFiles.size() + " config files. " +
                "Found " + metricContexts.size() + "  metric contexts " +
                "in " + clickHouseTables.keySet().size() + " clickhouse tables."
        );
        if (invalidFiles.isEmpty()) {
            configUnit.ok();
        } else {
            double failedConfigPercent = invalidFiles.size() * 100.0 / configFiles.size();
            log.error("Invalid files ignored: " + invalidFiles.toString());
            if (failedConfigPercent >= failedConfigsPercentToCrit) {
                configUnit.critical(
                    "Too many failed configs (" + invalidFiles.size() + " of " + configFiles.size() + ")"
                );
            } else {
                configUnit.warning(
                    "Some configs failed (" + invalidFiles.size() + " of " + configFiles.size() + ")" +
                        ":" + invalidFiles.toString()
                );
            }
        }
        conf = new ClickphiteConfiguration(configFiles, metricContexts);
        metricContextGroupsToValidate = conf.getMetricContextGroups();
        revalidateMetricContextGroups();
    }

    private boolean needToRevalidateMetricContextGroups() {
        if (metricContextGroupsToValidate.isEmpty()) {
            return false;
        }

        Duration elapsed = metricContextGroupsValidationStopwatch.elapsed();
        if (elapsed.compareTo(Duration.ofMinutes(failedConfigsRetryMinutes)) < 0) {
            log.info(
                "{} metric groups are invalid, will revalidate in {}",
                metricContextGroupsToValidate.size(),
                Duration.ofMinutes(failedConfigsRetryMinutes).minus(elapsed)
            );
            return false;
        }

        return true;
    }

    private void revalidateMetricContextGroups() {
        // Валидируем запросы из metricContextGroups
        List<MetricContextGroupValidationError> validationErrors =
            metricContextGroupValidator.checkQueries(metricContextGroupsToValidate);

        // Обновляем мониторинг про невалидные конфиги.
        if (!validationErrors.isEmpty()) {
            double failedPercent = validationErrors.size() * 100.0 / conf.getMetricContextGroups().size();
            String grepInstruction = "Run \"grep -A 5 'metric group considered invalid' " +
                "/var/log/clickphite/clickphite.log\"";
            if (failedPercent >= failedMetricGroupsPercentToCrit) {
                metricGroupValidationUnit.critical(
                    String.format(
                        "Too many metric groups have invalid queries (%d of %d). " + grepInstruction,
                        validationErrors.size(), conf.getMetricContextGroups().size()
                    )
                );
            } else {
                metricGroupValidationUnit.warning(
                    String.format(
                        "Some metric groups have invalid queries (%d of %d). " + grepInstruction,
                        validationErrors.size(), conf.getMetricContextGroups().size()
                    )
                );
            }
        } else {
            metricGroupValidationUnit.ok();
        }

        // Оставляем в metricContextGroups только невалидные конфиги
        metricContextGroupsToValidate = validationErrors.stream()
            .map(MetricContextGroupValidationError::getGroup)
            .collect(Collectors.toList());

        // Рестартим отсчёт времени до следующей валидации
        metricContextGroupsValidationStopwatch.reset();
        metricContextGroupsValidationStopwatch.start();
    }

    /**
     * Проверяем, что id метрики не пересекаются в разных файлах.
     * То, что они уникальны в рамках файла проверили ранее
     */
    private void checkSimilarMetricIds(Collection<ConfigFile> configFiles) {
        MultiMap<String, ConfigFile> idToConfigFile = new MultiMap<>();
        for (ConfigFile configFile : configFiles) {
            for (MetricContext metricContext : configFile.getMetricContexts()) {
                idToConfigFile.append(metricContext.getId(), configFile);
            }
        }

        for (Map.Entry<String, List<ConfigFile>> idToConfigFiles : idToConfigFile.entrySet()) {
            Collection<ConfigFile> idConfigFiles = idToConfigFiles.getValue();
            if (idConfigFiles.size() > 1) {
                for (ConfigFile configFile : idConfigFiles) {
                    configFile.setValid(false);
                }
                log.error(
                    "Same metric id (" + idToConfigFiles.getKey() + ") found in multiple files. " +
                        "All files marked as invalid and ignored: " + idConfigFiles
                );
            }
        }
    }

    /**
     * Проверяем, что id дашбордов не пересекаются в разных файлах.
     * То, что они уникальны в рамках файла проверили ранее
     */
    private void checkSimilarDashboardIds(Collection<ConfigFile> configFiles) {
        MultiMap<String, ConfigFile> idToConfigFile = new MultiMap<>();
        for (ConfigFile configFile : configFiles) {
            for (DashboardContext dashboardContext : configFile.getDashboardContexts()) {
                idToConfigFile.append(dashboardContext.getId(), configFile);
            }
        }

        for (Map.Entry<String, List<ConfigFile>> idToConfigFiles : idToConfigFile.entrySet()) {
            Collection<ConfigFile> idConfigFiles = idToConfigFiles.getValue();
            if (idConfigFiles.size() > 1) {
                for (ConfigFile configFile : idConfigFiles) {
                    configFile.setValid(false);
                }
                log.error(
                    "Same dashboard id (" + idToConfigFiles.getKey() + ") found in multiple files. " +
                        "All files marked as invalid and ignored: " + idConfigFiles
                );
            }
        }
    }

    private File[] getCurrentConfigFiles() {
        File[] files = configDir.listFiles((dir, name) -> name.toLowerCase().endsWith(".json"));
        Arrays.sort(files, Comparator.comparing(File::getName));
        return files;
    }

    private MetricContext createMetricContext(StatfaceReportConfig reportConfig) throws ConfigValidationException {
        ClickHouseTable table = getClickHouseTable(reportConfig.getTableName());
        return new StatfaceReportContext(reportConfig, table);
    }

    public MetricContext createMetricContext(GraphiteMetricConfig metricConfig) throws ConfigValidationException {
        ClickHouseTable table = getClickHouseTable(metricConfig.getTableName());
        Collection<DashboardContext> dashboardContexts = createDashboardContexts(metricConfig);
        Collection<MonitoringContext> monitoringContexts = createMonitoringContexts(metricConfig);
        switch (metricConfig.getStorage()) {
            case GRAPHITE:
                ConfigValidator.checkPatterns(metricConfig);
                return new GraphiteMetricContext(metricConfig, table, dashboardContexts, monitoringContexts);
            case STATFACE:
                log.warn("Deprecated config for metric:" + metricConfig);
                return new StatfaceReportContext(convertToStatfaceConfig(metricConfig), table);
            default:
                throw new UnsupportedOperationException();
        }
    }

    private MetricContext createMetricContext(SolomonSensorConfig solomonSensorConfig) {
        return new SolomonSensorContext(solomonSensorConfig, getClickHouseTable(solomonSensorConfig.getTableName()));
    }

    @Deprecated
    private StatfaceReportConfig convertToStatfaceConfig(GraphiteMetricConfig config) throws ConfigValidationException {
        if (config.getType() != MetricType.SIMPLE) {
            throw new ConfigValidationException("Only simple type allowed for STATFACE storage");
        }
        if (!config.getSplits().isEmpty()) {
            throw new ConfigValidationException("For split use statface report section");
        }
        StatfaceReportConfig.Field field = new StatfaceReportConfig.Field(
            "value", config.getMetricField(), null, StatfaceField.ViewType.Float, DEFAULT_STATFACE_PRECISION
        );
        StatfaceReportConfig reportConfig = new StatfaceReportConfig(
            config.getTableName(), config.getMetricName(), config.getTitle(), config.getPeriod(),
            config.getFilter(), Collections.singletonList(field)
        );
        reportConfig.setTable(config.getTable());
        reportConfig.setSubAggregate(config.getSubAggregate());
        return reportConfig;
    }

    private Collection<MonitoringContext> createMonitoringContexts(GraphiteMetricConfig metricConfig)
        throws ConfigValidationException {

        List<MonitoringContext> monitoringContexts = new ArrayList<>();
        for (MonitoringConfig monitoringConfig : metricConfig.getMonitorings()) {
            monitoringContexts.add(createMonitoringContext(monitoringConfig, metricConfig));
        }
        return monitoringContexts;
    }

    private MonitoringContext createMonitoringContext(
        MonitoringConfig monitoringConfig, GraphiteMetricConfig metricConfig
    ) throws ConfigValidationException {
        String fieldName = null;
        if (metricConfig.getType().isQuantile()) {
            String quantile = monitoringConfig.getQuantile();
            if (quantile == null) {
                throw new ConfigValidationException(
                    "'quantile' param is not set for monitoring: " + metricConfig + ", but metric is quantile"
                );
            }
            if (!metricConfig.getQuantiles().contains(quantile)) {
                throw new ConfigValidationException(
                    "Invalid quantile '" + quantile + "'for monitoring: " + monitoringConfig +
                        ", valid options:" + metricConfig.getQuantiles()
                );
            }
            fieldName = quantile;
        }

        if (monitoringConfig.getRange() != null) {
            return new RangeMonitoringContext(metricConfig, monitoringConfig, monitoringService, fieldName);
        }
        if (monitoringConfig.getKronos() != null) {
            return new KronosMonitoringContext(metricConfig, monitoringConfig, monitoringService, fieldName);
        }
        throw new IllegalStateException("Unsupported monitoring type for: " + monitoringConfig);
    }

    private Collection<DashboardContext> createDashboardContexts(GraphiteMetricConfig metricConfig) {
        List<DashboardConfig> dashboardConfigs = metricConfig.getDashboards();
        if (dashboardConfigs == null || dashboardConfigs.isEmpty()) {
            return null;
        }
        List<DashboardContext> dashboardContexts = new ArrayList<>();
        for (DashboardConfig dashboardConfig : dashboardConfigs) {
            String id = dashboardPrefix + dashboardConfig.getId();
            DashboardQueries dashboardQueries = QueryBuilder.buildDashboardQueries(dashboardConfig, metricConfig);
            Set<String> tags = new HashSet<>();
            tags.addAll(dashboardConfig.getTags());
            tags.addAll(dashboardDefaultTags);

            dashboardContexts.add(new DashboardContext(
                id, dashboardConfig, dashboardGraphiteDataSource, dashboardQueries, metricConfig, tags
            ));
        }
        return dashboardContexts;
    }

    private ClickHouseTable getClickHouseTable(String name) {
        synchronized (clickHouseTables) {
            final String tableName = name.toLowerCase();
            return clickHouseTables.computeIfAbsent(tableName, k -> ClickHouseTable.create(tableName, defaultDatabase));
        }
    }

    private void validateJsonSchema(String data) throws IOException, ConfigValidationException {
        JsonNode dataNode = getNodeData(data);

        final JsonValidator validator = JsonSchemaFactory.byDefault().getValidator();
        ProcessingReport report = validator.validateUnchecked(VALIDATION_SCHEMA, dataNode);
        if (!report.isSuccess()) {
            throw new ConfigValidationException(JacksonUtils.prettyPrint(((AsJson) report).asJson()));
        }
    }

    private ConfigFile processConfigFile(ConfigFile configFile) {
        try {
            parseAndCheck(configFile);
        } catch (ConfigValidationException e) {
            log.error(
                "Failed to validate config file: " + configFile.getName() + ". Problems:\n" + e.getMessage(), e);
            configFile.setValid(false);
        } catch (Exception e) {
            log.error("Failed to process config file: " + configFile.getName(), e);
            configFile.setValid(false);
        }

        return configFile;
    }

    public void parseAndCheck(ConfigFile configFile) throws IOException, ConfigValidationException {
        configFile.setKnownModificationTime(configFile.getFile().lastModified());
        parseFile(configFile);

        List<MetricContext> metricContexts = new ArrayList<>();
        List<DashboardContext> dashboardContexts = new ArrayList<>();
        List<MonitoringContext> monitoringContexts = new ArrayList<>();

        configFile.getGraphiteMetricConfigs().stream()
            .map(this::createMetricContext)
            .forEach(context -> {
                metricContexts.add(context);
                dashboardContexts.addAll(context.getDashboardContexts());
                monitoringContexts.addAll(context.getMonitoringContexts());
            });

        configFile.getStatfaceReportConfigs().stream()
            .map(this::createMetricContext)
            .forEach(metricContexts::add);

        configFile.getSolomonSensorConfigs().stream()
            .map(this::createMetricContext)
            .forEach(metricContexts::add);

        validateIdUniqueness(metricContexts, MetricContext::getId, "metrics", configFile.getName());
        validateIdUniqueness(dashboardContexts, DashboardContext::getId, "dashboards", configFile.getName());
        validateIdUniqueness(monitoringContexts, MonitoringContext::getName, "monitorings", configFile.getName());

        configFile.setValid(true);
        configFile.setMetricContexts(metricContexts);
        configFile.setDashboardContexts(dashboardContexts);
        configFile.setMonitoringContexts(monitoringContexts);
        log.info("Processed file: {}, found {} metricContexts", configFile.getName(), metricContexts.size());
    }

    private static <T> void validateIdUniqueness(Collection<T> items, Function<T, String> idExtractor,
                                                 String itemName, String fileName) throws ConfigValidationException {
        List<String> nonUniqueIds = getNonUniqueStrings(items.stream().map(idExtractor).collect(Collectors.toList()));
        if (!nonUniqueIds.isEmpty()) {
            throw new ConfigValidationException(String.format(
                "File %s has multiple %s with the same id:\n%s",
                fileName, itemName, String.join("\n", nonUniqueIds)
            ));
        }
    }

    private static List<String> getNonUniqueStrings(List<String> strings) {
        Set<String> set = new HashSet<>();
        List<String> result = new ArrayList<>();
        for (String s : strings) {
            if (!set.add(s)) {
                result.add(s);
            }
        }
        return result;
    }

    public void parseFile(ConfigFile configFile) throws IOException, ConfigValidationException {
        String data = FileUtils.readFileToString(configFile.getFile());

        validateJsonSchema(data);

        JsonObject configObject = gson.fromJson(data, JsonObject.class);

        List<StatfaceReportConfig> statfaceReportConfigs = parseStatfaceReports(configObject);
        List<GraphiteMetricConfig> graphiteMetricConfigs = parseMetrics(configObject);
        List<SolomonSensorConfig> solomonSensorConfigs = parseSolomonSensors(configObject);

        ListMultimap<String, GraphiteMetricConfig> nameToMetric =
            Multimaps.index(graphiteMetricConfigs, GraphiteMetricConfig::getMetricName);

        processDashboards(parseDashboards(configObject), nameToMetric);
        processMonitorings(parseMonitorings(configObject), nameToMetric);

        configFile.setGraphiteMetricConfigs(graphiteMetricConfigs);
        configFile.setStatfaceReportConfigs(statfaceReportConfigs);
        configFile.setSolomonSensorConfigs(solomonSensorConfigs);
    }

    private GlobalConfig parseGlobalConfig(JsonObject configObject) throws ConfigValidationException {
        return new GlobalConfig(
            Optional.ofNullable(configObject.get("tableName")).map(JsonElement::getAsString).orElse(null),
            parseSplits(configObject.getAsJsonObject("splits")),
            Optional.ofNullable(configObject.get("defaultFilter")).map(JsonElement::getAsString).orElse(null)
        );
    }

    private void processMonitorings(
        List<MonitoringConfig> monitorings, ListMultimap<String, GraphiteMetricConfig> nameToMetric
    ) throws ConfigValidationException {
        if (monitorings == null) {
            return;
        }
        for (MonitoringConfig monitoringConfig : monitorings) {
            List<GraphiteMetricConfig> metricConfigs = nameToMetric.get(monitoringConfig.getMetric());
            selectMetricConfigForMonitoring(metricConfigs, monitoringConfig).addMonitoring(monitoringConfig);
        }
    }

    private GraphiteMetricConfig selectMetricConfigForMonitoring(
        List<GraphiteMetricConfig> metricConfigs, MonitoringConfig monitoring
    ) throws ConfigValidationException {
        if (metricConfigs == null || metricConfigs.isEmpty()) {
            throw new ConfigValidationException("Monitoring for non-existing metric:" + monitoring.getMetric());
        }
        metricConfigs = sortMetricByPeriod(metricConfigs);
        if (monitoring.getKronos() == null) {
            if (metricConfigs.get(0).getPeriod().ordinal() >= MetricPeriod.DAY.ordinal()) {
                throw new ConfigValidationException(
                    "Monitoring period must be not greater than HOUR: " + monitoring.getMetric()
                );
            }
            //Выбираем приоритетно ONE_MIN
            for (GraphiteMetricConfig metricConfig : metricConfigs) {
                if (metricConfig.getPeriod() == MetricPeriod.ONE_MIN) {
                    return metricConfig;
                }
            }
            return metricConfigs.get(0);
        }
        //Выбираем FIVE_MIN(приоритетно) или ONE_MIN для Кроноса
        for (int i = metricConfigs.size() - 1; i >= 0; i--) {
            GraphiteMetricConfig metricConfig = metricConfigs.get(i);
            if (metricConfig.getPeriod() == MetricPeriod.FIVE_MIN || metricConfig.getPeriod() == MetricPeriod.ONE_MIN) {
                return metricConfig;
            }
        }
        throw new ConfigValidationException(
            "Kronos monitoring must be for ONE_MIN or FIVE_MIN (preferred) period for metric: " + monitoring.getMetric()
        );
    }


    private void processDashboards(
        List<DashboardConfig> dashboards, ListMultimap<String, GraphiteMetricConfig> nameToMetric
    ) throws ConfigValidationException {

        if (dashboards == null) {
            return;
        }

        for (DashboardConfig dashboard : dashboards) {
            List<GraphiteMetricConfig> metricConfigs = nameToMetric.get(dashboard.getMetric());
            if (metricConfigs.isEmpty()) {
                throw new ConfigValidationException("Dashboard for non-existing metric:" + dashboard.getMetric());
            }
            sortMetricByPeriod(metricConfigs);
            metricConfigs.get(0).addDashboard(dashboard);
        }
    }


    private List<MonitoringConfig> parseMonitorings(JsonObject configObject) {
        JsonArray monitoringArray = configObject.getAsJsonArray("monitorings");
        if (monitoringArray == null) {
            return null;
        }
        List<MonitoringConfig> monitoringConfigs = new ArrayList<>();
        for (JsonElement monitoringElement : monitoringArray) {
            MonitoringConfig config = gson.fromJson(monitoringElement, MonitoringConfig.class);
            monitoringConfigs.add(config);
        }
        return monitoringConfigs;
    }


    private List<GraphiteMetricConfig> sortMetricByPeriod(List<GraphiteMetricConfig> metrics) {
        List<GraphiteMetricConfig> sorted = new ArrayList<>(metrics);
        Collections.sort(sorted, (m1, m2) -> Integer.compare(m1.getPeriod().ordinal(), m2.getPeriod().ordinal()));
        return sorted;
    }

    private List<DashboardConfig> parseDashboards(JsonObject configObject) {
        JsonArray dashboardArray = configObject.getAsJsonArray("dashboards");
        if (dashboardArray == null) {
            return null;
        }
        List<DashboardConfig> dashboardConfigs = new ArrayList<>(dashboardArray.size());
        for (int i = 0; i < dashboardArray.size(); i++) {
            JsonElement dashboardElement = dashboardArray.get(i);
            dashboardConfigs.add(gson.fromJson(dashboardElement, DashboardConfig.class));
        }
        return dashboardConfigs;
    }

    private Map<String, MetricSplit> parseSplits(JsonObject splitObject) throws ConfigValidationException {
        if (splitObject == null) {
            return Collections.emptyMap();
        }
        Map<String, MetricSplit> splits = new HashMap<>();
        for (Map.Entry<String, JsonElement> nameAndField : splitObject.entrySet()) {
            String name = nameAndField.getKey();
            splits.put(name, new MetricSplit(name, nameAndField.getValue().getAsString()));
        }
        return splits;
    }

    private static List<MetricSplit> getSplitsFromName(String name, List<MetricSplit> localSplits,
                                                       Map<String, MetricSplit> globalSplits) {
        if (globalSplits == null) {
            return Collections.emptyList();
        }
        List<MetricSplit> splits = new ArrayList<>();
        for (String split : StringTemplate.getVariablesFromString(name)) {
            MetricSplit s = globalSplits.get(split);
            if (s != null) {
                splits.add(s);
            }
        }
        if (!splits.isEmpty() && localSplits != null) {
            splits.removeAll(localSplits);
        }
        return splits;
    }

    private List<StatfaceReportConfig> parseStatfaceReports(JsonObject configObject) {
        GlobalConfig globalConfig = parseGlobalConfig(configObject);
        JsonArray statfaceReportArray = configObject.getAsJsonArray("statfaceReports");

        if (statfaceReportArray == null) {
            return Collections.emptyList();
        }
        List<StatfaceReportConfig> reportConfigs = new ArrayList<>();
        for (JsonElement reportElement : statfaceReportArray) {
            StatfaceReportConfig reportConfig = gson.fromJson(reportElement, StatfaceReportConfig.class);
            for (StatfaceReportConfig spawnConfig : spawnConfigs(globalConfig, reportConfig, reportElement)) {
                configValidator.validateStatfaceReportConfig(spawnConfig);
                reportConfigs.add(spawnConfig);
            }
        }
        return reportConfigs;

    }

    private void applyGlobalConfig(GlobalConfig globalConfig,
                                   AbstractMetricConfig config) throws ConfigValidationException {
        if (config.getTableName() == null) {
            if (globalConfig.getTableName() == null) {
                throw new ConfigValidationException("No table specified for config:" + config);
            }
            config.setTableName(globalConfig.getTableName());
        }
        config.setTable(getClickHouseTable(config.getTableName()));

        if (config.getFilter() == null) {
            config.setFilter(globalConfig.getFilter());
        }
    }

    private <T extends AbstractMetricConfig<T>> List<T> spawnConfigs(GlobalConfig globalConfig, T config,
                                                                     JsonElement configElement)
        throws ConfigValidationException {

        applyGlobalConfig(globalConfig, config);

        if (config.getPeriod() != null) {
            return Collections.singletonList(config);
        }
        JsonArray periodArray = configElement.getAsJsonObject().getAsJsonArray("periodArray");
        if (periodArray == null) {
            throw new ConfigValidationException("Unknown 'period' or 'periodArray' for " + config);
        }
        List<MetricPeriod> periods = new ArrayList<>();
        for (JsonElement periodElement : periodArray) {
            periods.add(MetricPeriod.valueOf(periodElement.getAsString()));
        }
        List<T> configs = new ArrayList<>();

        for (MetricPeriod period : periods) {
            T periodConfig = config.copy();
            periodConfig.setPeriod(period);
            configs.add(periodConfig);
        }
        return configs;
    }


    private List<GraphiteMetricConfig> parseMetrics(JsonObject configObject) {
        GlobalConfig globalConfig = parseGlobalConfig(configObject);
        JsonArray metricArray = configObject.getAsJsonArray("metrics");

        if (metricArray == null) {
            return Collections.emptyList();
        }
        List<GraphiteMetricConfig> metrics = new ArrayList<>();
        for (JsonElement metricElement : metricArray) {
            metrics.addAll(parseMetricElement(globalConfig, metricElement));
        }
        return metrics;
    }

    private List<GraphiteMetricConfig> parseMetricElement(GlobalConfig globalConfig,
                                                          JsonElement metricElement) throws ConfigValidationException {

        GraphiteMetricConfig metricConfig = gson.fromJson(metricElement, GraphiteMetricConfig.class);
        metricConfig.setMetricPrefix(graphiteMetricPrefix);
        metricConfig.setGlobalMetricPrefix(graphiteGlobalMetricPrefix);
        List<GraphiteMetricConfig> spawnConfigs = spawnConfigs(globalConfig, metricConfig, metricElement);
        spawnConfigs = spawnSplitConfigs(globalConfig, spawnConfigs, metricElement);
        for (GraphiteMetricConfig spawnConfig : spawnConfigs) {
            configValidator.validateMetricConfig(spawnConfig);
        }
        return spawnConfigs;
    }


    private static List<GraphiteMetricConfig> spawnSplitConfigs(
        GlobalConfig globalConfig, List<GraphiteMetricConfig> configs, JsonElement metricElement
    ) throws ConfigValidationException {

        List<GraphiteMetricConfig> result = new ArrayList<>();
        for (GraphiteMetricConfig config : configs) {
            if (config.getMetricName() != null) {
                result.add(config);
                continue;
            }
            JsonArray metricNameArray = metricElement.getAsJsonObject().getAsJsonArray("metricNameArray");
            if (metricNameArray == null) {
                throw new ConfigValidationException("Unknown 'metricName' or 'metricNameArray' for " + config);
            }
            for (JsonElement nameElement : metricNameArray) {
                String name = nameElement.getAsString();
                GraphiteMetricConfig newConfig = config.copy();
                newConfig.setMetricName(name);
                result.add(newConfig);
            }
        }
        for (GraphiteMetricConfig config : result) {
            config.addSplits(getSplitsFromName(config.getMetricName(), config.getSplits(), globalConfig.getSplits()));
        }
        return result;
    }

    private List<SolomonSensorConfig> parseSolomonSensors(JsonObject configObject) {
        GlobalConfig globalConfig = parseGlobalConfig(configObject);

        Map<String, String> commonSolomonLabels = MoreObjects.firstNonNull(
            gson.fromJson(
                configObject.get("commonSolomonLabels"),
                TypeToken.getParameterized(Map.class, String.class, String.class).getType()
            ),
            Collections.emptyMap()
        );

        Iterable<JsonElement> solomonSensors = MoreObjects.firstNonNull(
            configObject.getAsJsonArray("solomonSensors"),
            Collections.emptyList()
        );

        return StreamSupport.stream(solomonSensors.spliterator(), false)
            .flatMap(configElement ->
                spawnAndValidateSolomonConfigs(globalConfig, commonSolomonLabels, configElement).stream())
            .collect(Collectors.toList());
    }

    private List<SolomonSensorConfig> spawnAndValidateSolomonConfigs(GlobalConfig globalConfig,
                                                                     Map<String, String> commonSolomonLabels,
                                                                     JsonElement configElement) {
        List<SolomonSensorConfig> configs =
            Stream.of(gson.fromJson(configElement, SolomonSensorConfig.class))
                .flatMap(config -> spawnConfigs(globalConfig, config, configElement).stream())
                .flatMap(config -> spawnConfigsForLabelsArrayField(config, configElement).stream())
                .collect(Collectors.toList());
        for (SolomonSensorConfig config : configs) {
            config.setLabels(createSolomonLabels(config, commonSolomonLabels));
            config.setSplits(getSplitsFromLabels(config.getLabels(), globalConfig.getSplits()));
            SolomonSensorConfigValidator.validate(config);
        }
        return configs;
    }

    private Map<String, String> createSolomonLabels(SolomonSensorConfig config, Map<String, String> commonLabels) {
        Map<String, String> labels = new HashMap<>(config.getLabels());

        // Добавляем commonLabels к labels. Значения из labels имеют приоритет.
        commonLabels.forEach(labels::putIfAbsent);

        // st/MARKETINFRA-4145. Тестовые окружения Кликфита пишут в прод Соломона в отдельный проект.
        // Проект из конфига дописываем в лейбл service чтобы в тестинге было столько же шардов, сколько в проде.
        // Сохранять количество шардов нужно потому что есть ограничение на количество сенсоров в секунду на шард.
        if (!Strings.isNullOrEmpty(solomonProjectOverride)) {
            labels.put("service", labels.get("project") + "--" + labels.get("service"));
            labels.put("project", solomonProjectOverride);
        }

        labels.put("period", config.getPeriod().getGraphiteName());

        return labels;
    }

    private List<SolomonSensorConfig> spawnConfigsForLabelsArrayField(SolomonSensorConfig config,
                                                                      JsonElement configElement) {
        if (config.getLabels() != null) {
            return Collections.singletonList(config);
        }

        List<Map<String, String>> allLabels =
            gson.fromJson(
                configElement.getAsJsonObject().getAsJsonArray("labelsArray"),
                new TypeToken<List<Map<String, String>>>() {
                }.getType()
            );

        return allLabels.stream()
            .map(labels -> {
                SolomonSensorConfig configWithLabels = config.copy();
                configWithLabels.setLabels(labels);
                return configWithLabels;
            })
            .collect(Collectors.toList());
    }

    private static List<MetricSplit> getSplitsFromLabels(Map<String, String> labels,
                                                         Map<String, MetricSplit> globalSplits) {
        return labels.values().stream()
            .flatMap(labelValue -> getSplitsFromName(labelValue, null, globalSplits).stream())
            .distinct()
            .collect(Collectors.toList());
    }

    @Required
    public void setConfigDir(String configDir) {
        File dir = new File(configDir);
        if (!dir.isDirectory() || !dir.canRead()) {
            throw new IllegalArgumentException("Wrong config dir: " + configDir);
        }
        this.configDir = dir;
    }

    @Required
    public void setConfigValidator(ConfigValidator configValidator) {
        this.configValidator = configValidator;
    }

    @Required
    public void setDashboardGraphiteDataSource(String dashboardGraphiteDataSource) {
        this.dashboardGraphiteDataSource = dashboardGraphiteDataSource;
    }

    @Required
    public void setMonitoringService(MonitoringService monitoringService) {
        this.monitoringService = monitoringService;
    }

    @Required
    public void setMonitoring(ComplicatedMonitoring monitoring) {
        this.monitoring = monitoring;
    }

    public void setDashboardDefaultTags(String dashboardDefaultTags) {
        String[] splits = dashboardDefaultTags.split(",");
        this.dashboardDefaultTags = new HashSet<>();
        for (String split : splits) {
            this.dashboardDefaultTags.add(split.trim());
        }
    }

    public void setDashboardPrefix(String dashboardPrefix) {
        this.dashboardPrefix = dashboardPrefix;
    }

    public void setGraphiteMetricPrefix(String graphiteMetricPrefix) {
        this.graphiteMetricPrefix = graphiteMetricPrefix;
    }

    public void setGraphiteGlobalMetricPrefix(String graphiteGlobalMetricPrefix) {
        this.graphiteGlobalMetricPrefix = graphiteGlobalMetricPrefix;
    }

    public void setSolomonProjectOverride(String solomonProjectOverride) {
        this.solomonProjectOverride = solomonProjectOverride;
    }

    public void setFailedConfigsPercentToCrit(double failedConfigsPercentToCrit) {
        this.failedConfigsPercentToCrit = failedConfigsPercentToCrit;
    }

    public void setFailedMetricGroupsPercentToCrit(double failedMetricGroupsPercentToCrit) {
        this.failedMetricGroupsPercentToCrit = failedMetricGroupsPercentToCrit;
    }

    public void setFileReadersThreadCount(int fileReadersThreadCount) {
        this.fileReadersThreadCount = fileReadersThreadCount;
    }

    public void setFailedConfigsRetryMinutes(int failedConfigsRetryMinutes) {
        this.failedConfigsRetryMinutes = failedConfigsRetryMinutes;
    }

    @Required
    public void setDefaultDatabase(String defaultDatabase) {
        this.defaultDatabase = defaultDatabase;
    }

    @Required
    public void setMetricContextGroupValidator(MetricContextGroupValidator metricContextGroupValidator) {
        this.metricContextGroupValidator = metricContextGroupValidator;
    }

    @VisibleForTesting
    File getConfigDir() {
        return configDir;
    }
}
