package ru.yandex.market.clickphite.config.metric;

import ru.yandex.market.clickphite.config.dashboard.DashboardConfig;
import ru.yandex.market.clickphite.config.monitoring.MonitoringConfig;
import ru.yandex.market.clickphite.metric.MetricStorage;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 11/12/14
 */
public class GraphiteMetricConfig extends AbstractMetricConfig<GraphiteMetricConfig> {

    private static final List<String> DEFAULT_QUANTILES = Arrays.asList(
        "0.5", "0.6", "0.7", "0.80", "0.90", "0.95", "0.97", "0.99", "0.995", "0.997", "0.999", "1"
    );

    private String metricName;
    private String title;
    private String metricField;
    private List<MetricSplit> splits;
    private MetricType type = MetricType.SIMPLE;
    private MetricStorage storage = MetricStorage.GRAPHITE;
    private List<DashboardConfig> dashboards = new ArrayList<>();
    private List<MonitoringConfig> monitorings = new ArrayList<>();
    private List<String> quantiles;
    private String metricPrefix = "";
    private String globalMetricPrefix = "";

    public GraphiteMetricConfig() {
    }

    private GraphiteMetricConfig(GraphiteMetricConfig other) {
        super(other);
        this.metricName = other.metricName;
        this.title = other.title;
        this.metricField = other.metricField;
        this.splits = other.splits;
        this.type = other.type;
        this.storage = other.storage;
        this.quantiles = other.quantiles;
        this.metricPrefix = other.metricPrefix;
        this.globalMetricPrefix = other.globalMetricPrefix;
    }

    public GraphiteMetricConfig(GraphiteMetricConfig metricConfig, String overrideMetricName,
                                List<MetricSplit> overrideSplits) {
        this(metricConfig);
        splits = overrideSplits;
        metricName = overrideMetricName;
    }

    @Override
    public GraphiteMetricConfig copy() {
        return new GraphiteMetricConfig(this);
    }

    public String getGraphiteName() {
        return globalMetricPrefix + period.getGraphiteName() + "." + metricPrefix + metricName;
    }

    public String getMetricName() {
        return metricName;
    }

    public String getMetricField() {
        return metricField;
    }

    @Override
    public List<? extends MetricField> getFields() {
        return Collections.singletonList(
            new MetricField() {
                @Override
                public String getName() {
                    return "value";
                }

                @Override
                public String getField() {
                    return metricField;
                }

                @Override
                public MetricType getType() {
                    return type;
                }

                @Override
                public List<String> getQuantiles() {
                    return GraphiteMetricConfig.this.getQuantiles();
                }
            }
        );
    }

    public List<MetricSplit> getSplits() {
        return splits != null ? splits : Collections.emptyList();
    }

    public String getTitle() {
        return title;
    }

    @Override
    public MetricType getType() {
        return type;
    }

    public MetricStorage getStorage() {
        return storage;
    }

    @Override
    public List<String> getQuantiles() {
        return quantiles != null ? quantiles : DEFAULT_QUANTILES;
    }

    public List<String> getOriginalQuantiles() {
        return quantiles;
    }

    public List<DashboardConfig> getDashboards() {
        return dashboards;
    }

    public void addDashboard(DashboardConfig dashboard) {
        dashboards.add(dashboard);
    }

    public List<MonitoringConfig> getMonitorings() {
        return monitorings;
    }

    public void addMonitoring(MonitoringConfig monitoring) {
        monitorings.add(monitoring);
    }

    public void addSplits(List<MetricSplit> addSplits) {
        if (addSplits == null || addSplits.isEmpty()) {
            return;
        }
        if (splits == null || splits.isEmpty()) {
            splits = addSplits;
        } else {
            splits.addAll(addSplits);
        }
    }

    public void setQuantiles(List<String> quantiles) {
        this.quantiles = quantiles;
    }

    public void setMetricName(String metricName) {
        this.metricName = metricName;
    }

    public void setType(MetricType type) {
        this.type = type;
    }

    @Override
    public String toString() {
        return "MetricConfig{" +
            "metricName='" + metricName + '\'' +
            ", type=" + type +
            ", storage=" + storage +
            ", period=" + period +
            '}';
    }

    public Object[] keyArray() {
        return new Object[]{
            tableName, metricName, title, metricField, filter, splits == null || splits.isEmpty() ? null : splits,
            period, type, storage, dashboards
        };
    }

    public String getMetricPrefix() {
        return metricPrefix;
    }

    public void setMetricPrefix(String metricPrefix) {
        this.metricPrefix = metricPrefix;
    }

    public void setGlobalMetricPrefix(String globalMetricPrefix) {
        this.globalMetricPrefix = globalMetricPrefix;
    }

    @Override
    public int hashCode() {
        return Arrays.hashCode(keyArray());
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (!(obj instanceof GraphiteMetricConfig)) {
            return false;
        }

        GraphiteMetricConfig that = (GraphiteMetricConfig) obj;
        return Arrays.equals(this.keyArray(), that.keyArray());
    }
}
