package ru.yandex.market.clickphite.config.metric;

import ru.yandex.market.clickphite.DateTimeUtils;
import ru.yandex.market.statface.StatfacePeriod;

import java.util.Date;
import java.util.concurrent.TimeUnit;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 17/01/15
 */
public enum MetricPeriod implements Comparable<MetricPeriod> {

    ONE_SEC("one_sec", StatfacePeriod.continual, "timestamp", 1, false),
    FIVE_SEC("five_sec", StatfacePeriod.continual, "multiply(intDiv(timestamp, 5), 5)", 5, false),
    ONE_MIN("one_min", StatfacePeriod.minutely, "multiply(intDiv(timestamp, 60), 60)", 1, TimeUnit.MINUTES, false),
    FIVE_MIN("five_min", StatfacePeriod.minutely, "multiply(intDiv(timestamp, 300), 300)", 5, TimeUnit.MINUTES, false),
    HOUR("one_hour", StatfacePeriod.hourly, "multiply(intDiv(timestamp, 3600), 3600)", 1, TimeUnit.HOURS, false),
    DAY("one_day", StatfacePeriod.daily, "toUInt32(toDateTime(date))", 1, TimeUnit.DAYS, true),
    WEEK("one_day", StatfacePeriod.weekly, "toUInt32(toDateTime(toMonday(date)))", 7, TimeUnit.DAYS, true),
    MONTH("one_day", StatfacePeriod.monthly, "toUInt32(toDateTime(toStartOfMonth(date)))", 30, TimeUnit.DAYS, true),
    QUARTER(
        "one_day", StatfacePeriod.quarterly, "toUInt32(toDateTime(toStartOfQuarter(date)))", 91, TimeUnit.DAYS, true
    );

    private final String graphiteName;
    private final StatfacePeriod statfacePeriod;
    private final String clickHouseFunction;
    private final int durationSeconds;
    private final boolean graphiteOffset;

    MetricPeriod(String graphiteName, StatfacePeriod statfacePeriod, String clickHouseFunction,
                 int durationSeconds, boolean graphiteOffset) {
        this.graphiteName = graphiteName;
        this.statfacePeriod = statfacePeriod;
        this.clickHouseFunction = clickHouseFunction;
        this.durationSeconds = durationSeconds;
        this.graphiteOffset = graphiteOffset;
    }

    MetricPeriod(String graphiteName, StatfacePeriod statfacePeriod, String clickHouseFunction,
                 int duration, TimeUnit unit, boolean graphiteOffset) {
        this(graphiteName, statfacePeriod, clickHouseFunction, (int) unit.toSeconds(duration), graphiteOffset);
    }

    public boolean isGraphiteOffset() {
        return graphiteOffset;
    }

    public StatfacePeriod getStatfacePeriod() {
        return statfacePeriod;
    }

    public String getGraphiteName() {
        return graphiteName;
    }

    public String getClickHouseFunction() {
        return clickHouseFunction;
    }

    public int getDurationSeconds() {
        return durationSeconds;
    }

    public int getEndTimeSeconds(int metricDelaySeconds) {
        int currentTimeSeconds = DateTimeUtils.currentTimeSeconds();
        return getEndTimeSeconds(metricDelaySeconds, currentTimeSeconds);
    }

    public int getEndTimeSeconds(int metricDelaySeconds, int currentTimeSeconds) {
        Date end = DateTimeUtils.toPeriodStart(this, currentTimeSeconds - metricDelaySeconds);
        return DateTimeUtils.millisToSeconds(end.getTime());
    }
}
