package ru.yandex.market.clickphite.config.metric;

import ru.yandex.market.clickphite.metric.MetricStorage;
import ru.yandex.market.statface.StatfaceField;
import ru.yandex.market.statface.StatfaceGraph;
import ru.yandex.market.statface.StatfaceReportConfiguration;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 17/02/16
 */
public class StatfaceReportConfig extends AbstractMetricConfig<StatfaceReportConfig> {

    private String title;
    private String report;
    private List<Split> splits;
    private List<Field> fields;
    private List<StatfaceGraph> graphs;

    public StatfaceReportConfig() {
    }

    public StatfaceReportConfig(String tableName, String report, String title, MetricPeriod period,
                                String filter, List<Field> fields) {
        this.tableName = tableName;
        this.title = title;
        this.report = report;
        this.filter = filter;
        this.fields = fields;
        this.period = period;
    }

    private StatfaceReportConfig(StatfaceReportConfig other) {
        super(other);
        this.title = other.title;
        this.report = other.report;
        this.splits = other.splits;
        this.fields = other.fields;
        this.graphs = other.graphs;
    }

    public class Graph {
        private String title;
        private List<String> fields;

        public String getTitle() {
            return title;
        }

        public List<String> getFields() {
            return fields;
        }
    }

    public StatfaceReportConfiguration createStatfaceReportConfiguration() {
        StatfaceReportConfiguration reportConfiguration = new StatfaceReportConfiguration(
            report, title, period.getStatfacePeriod(), getStatfaceFields(), getGraphs()
        );
        reportConfiguration.setYtOnly(true);
        return reportConfiguration;
    }

    public static class Split extends AbstractField {

        private String splitField;
        private boolean tree = false;

        public Split() {
        }

        public Split(String name, String splitField) {
            super(name);
            this.splitField = splitField;
        }


        @Override
        public String getField() {
            return splitField;
        }

        public boolean isTree() {
            return tree;
        }

        @Override
        protected StatfaceField.DataType getDataType() {
            return tree ? StatfaceField.DataType.tree : StatfaceField.DataType.string;
        }

        @Override
        protected boolean isDimension() {
            return true;
        }
    }

    public static class Field extends AbstractField {

        private String metricField;

        public Field() {
        }

        public Field(String name, String metricField) {
            super(name);
            this.metricField = metricField;
        }

        public Field(String name, String metricField, String title,
                     StatfaceField.ViewType viewType, Integer precision) {
            super(name, title, viewType, precision);
            this.metricField = metricField;
        }

        @Override
        public String getField() {
            return metricField;
        }

        @Override
        protected StatfaceField.DataType getDataType() {
            return StatfaceField.DataType.number;
        }

        @Override
        protected boolean isDimension() {
            return false;
        }
    }

    private abstract static class AbstractField implements MetricField {
        private String name;
        private String title;
        private StatfaceField.ViewType viewType;
        private Integer precision;

        private AbstractField() {
        }

        private AbstractField(String name, String title, StatfaceField.ViewType viewType, Integer precision) {
            this.name = name;
            this.title = title;
            this.viewType = viewType;
            this.precision = precision;
        }

        private AbstractField(String name) {
            this.name = name;
        }

        @Override
        public String getName() {
            return name;
        }

        protected abstract StatfaceField.DataType getDataType();

        protected abstract boolean isDimension();

        public StatfaceField toStatFaceField() {
            return new StatfaceField(name, isDimension(), getDataType(), title, viewType, precision);
        }
    }

    public List<StatfaceField> getStatfaceFields() {
        List<StatfaceField> statfaceFields = new ArrayList<>();
        statfaceFields.add(StatfaceField.DATE_FIELD);
        if (splits != null) {
            for (Split split : splits) {
                statfaceFields.add(split.toStatFaceField());
            }
        }
        for (Field field : fields) {
            statfaceFields.add(field.toStatFaceField());
        }
        return statfaceFields;
    }

    @Override
    public List<Field> getFields() {
        return fields;
    }

    @Override
    public List<Split> getSplits() {
        return (splits != null) ? splits : Collections.emptyList();
    }

    @Override
    public StatfaceReportConfig copy() {
        return new StatfaceReportConfig(this);
    }

    public String getTitle() {
        return title;
    }

    public void setTitle(String title) {
        this.title = title;
    }

    public String getReport() {
        return report;
    }

    public void setReport(String report) {
        this.report = report;
    }

    public void setSplits(List<Split> splits) {
        this.splits = splits;
    }

    public void setFields(List<Field> fields) {
        this.fields = fields;
    }

    public List<StatfaceGraph> getGraphs() {
        return graphs;
    }

    public void setGraphs(List<StatfaceGraph> graphs) {
        this.graphs = graphs;
    }

    @Override
    public String toString() {
        return "StatfaceReportConfig{" +
            "title='" + title + '\'' +
            ", report='" + report + '\'' +
            '}';
    }

    @Override
    public MetricStorage getStorage() {
        return MetricStorage.STATFACE;
    }

    @Override
    public MetricType getType() {
        return MetricType.SIMPLE;
    }

    @Override
    public List<String> getQuantiles() {
        throw new RuntimeException(getClass().getSimpleName() + " doesn't support quantiles");
    }
}
