package ru.yandex.market.clickphite.config.validation.config;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.market.clickhouse.ClickhouseTemplate;
import ru.yandex.market.clickphite.QueryBuilder;
import ru.yandex.market.clickphite.metric.MetricContextGroup;

import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @author Anton Sukhonosenko <a href="mailto:algebraic@yandex-team.ru"></a>
 * @date 19.09.17
 */
public class MetricContextGroupValidator {
    private static final Logger log = LogManager.getLogger();

    private int threadCount = 32;
    private int validationRetryCount = 10;
    private ClickhouseTemplate clickhouseTemplate;
    private int validationRetryPauseMillis = 100;
    private String localReplicatedTablePostfix = "";

    public List<MetricContextGroupValidationError> checkQueries(List<MetricContextGroup> groups) {
        long beginTimeNanos = System.nanoTime();

        ExecutorService executorService = Executors.newFixedThreadPool(threadCount);

        Collection<Future<Optional<MetricContextGroupValidationError>>> errorFutures = groups.stream()
            .map(group -> executorService.submit(() -> checkQuery(group)))
            .collect(Collectors.toList());

        executorService.shutdown();

        List<MetricContextGroupValidationError> validationErrors = errorFutures.stream()
            .map(future -> {
                try {
                    return future.get();
                } catch (InterruptedException | ExecutionException e) {
                    throw new RuntimeException(e);
                }
            })
            .filter(Optional::isPresent)
            .map(Optional::get)
            .collect(Collectors.toList());

        log.info(
            "Config validation took {} milliseconds",
            TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - beginTimeNanos)
        );

        return validationErrors;
    }

    private Optional<MetricContextGroupValidationError> checkQuery(MetricContextGroup group) {
        for (int i = 1; i <= validationRetryCount; i++) {
            long startTimeNanos = System.nanoTime();
            String query = QueryBuilder.buildMetricQueryTemplate(
                group, QueryBuilder.UNREAL_PERIOD, localReplicatedTablePostfix, "now()"
            );

            try {
                clickhouseTemplate.query(
                    query,
                    rs -> {
                        //ignored
                    }
                );

                log.info(
                    "Validation query succeeded in {} ms, query = {}",
                    TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startTimeNanos), query
                );
                return Optional.empty();
            } catch (Exception ex) {
                final boolean lastTry = (i == validationRetryCount);
                if (lastTry) {
                    log.error(
                        "Check query attempts exceeded, metric group considered invalid: " + group.getId(),
                        ex
                    );
                    return Optional.of(new MetricContextGroupValidationError(group, ex));
                } else {
                    log.warn(
                        String.format(
                            "Check query attempt (%d) failed in %d millis, query = %s",
                            i, TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startTimeNanos), query
                        ), ex
                    );

                    try {
                        Thread.sleep(validationRetryPauseMillis);
                    } catch (InterruptedException e) {
                        log.info("Interrupted");
                    }
                }
            }
        }

        throw new IllegalStateException("bug in MetricContextGroupValidator.checkQuery");
    }

    public void setThreadCount(int threadCount) {
        this.threadCount = threadCount;
    }

    public void setValidationRetryCount(int validationRetryCount) {
        this.validationRetryCount = validationRetryCount;
    }

    public void setLocalReplicatedTablePostfix(String localReplicatedTablePostfix) {
        this.localReplicatedTablePostfix = localReplicatedTablePostfix;
    }

    @Required
    public void setClickhouseTemplate(ClickhouseTemplate clickhouseTemplate) {
        this.clickhouseTemplate = clickhouseTemplate;
    }

    public void setValidationRetryPauseMillis(int validationRetryPauseMillis) {
        this.validationRetryPauseMillis = validationRetryPauseMillis;
    }
}
