package ru.yandex.market.clickphite.config.validation.context;

import com.google.common.base.Joiner;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import ru.yandex.market.clickphite.config.metric.GraphiteMetricConfig;
import ru.yandex.market.clickphite.config.metric.MetricSplit;
import ru.yandex.market.clickphite.config.metric.StatfaceReportConfig;
import ru.yandex.market.clickphite.metric.MetricStorage;
import ru.yandex.market.clickphite.metric.StringTemplate;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 17/01/15
 */
public class ConfigValidator {
    private static final Logger log = LogManager.getLogger();

    private static final int MAX_SPLIT_COUNT = 7;
    private static final String ESCAPE_SYMBOLS = "~@#\\*%\\^\\s:;\\+,\\?!\\(\\)\\\\/<>'&\\|°\\[\\]`‘®™«»–—€•“’”=±§";
    private static final String ESCAPE_SYMBOLS_PATTERN = ".*[" + ESCAPE_SYMBOLS + "].*";

    private Pattern forbiddenFunctionsPattern = null;
    private Pattern allowedFunctionsPattern = null;

    public ConfigValidator() {
        this(null, null);
    }

    public ConfigValidator(String forbiddenFunctionsConfig, String allowedFunctionsConfig) {
        if (StringUtils.isNotEmpty(forbiddenFunctionsConfig)) {
            List<String> forbiddenFunctions = Arrays.asList(forbiddenFunctionsConfig.split(","));
            forbiddenFunctionsPattern = Pattern.compile("((" + Joiner.on('|').join(forbiddenFunctions) + ")[^(]*)\\(");
        }
        if (StringUtils.isNotEmpty(allowedFunctionsConfig)) {
            List<String> allowedFunctions = Arrays.asList(allowedFunctionsConfig.split(","));
            allowedFunctionsPattern = Pattern.compile("(" + Joiner.on('|').join(allowedFunctions) + ").*");
        }
    }

    /**
     * Рекомендуется использовать значения level в диапазоне 0.01 .. 0.99.
     *
     * @param config
     * @throws ConfigValidationException
     */
    public void validateQuantiles(GraphiteMetricConfig config) throws ConfigValidationException {
        List<String> quantiles = config.getOriginalQuantiles();
        if (quantiles == null || quantiles.isEmpty()) {
            return;
        }

        if (!config.getType().isQuantile()) {
            throw new ConfigValidationException("Forbidden 'quantiles' for type SIMPLE " + quantiles);
        }

        List<String> validQuantiles = new ArrayList<>();
        for (String quantile : quantiles) {
            try {
                Double q = Double.parseDouble(quantile);
                if (q >= 0 && q <= 1) {
                    validQuantiles.add(quantile);
                }
            } catch (NumberFormatException ignored) {
            }
        }
        if (validQuantiles.isEmpty()) {
            throw new ConfigValidationException("Not valid 'quantiles' " + quantiles);
        }

        validQuantiles.sort(Comparator.naturalOrder());
        config.setQuantiles(validQuantiles);
    }

    public void validateMetricConfig(GraphiteMetricConfig config) throws ConfigValidationException {
        validateQuantiles(config);
        validateMetricFunctions(config.getMetricField());
        StringBuilder errorReport = new StringBuilder();

        if (config.getStorage() == MetricStorage.STATFACE && StringUtils.isEmpty(config.getTitle())) {
            errorReport.append("Statface metric should contain title\n");
        }

        checkSplitCount(config, errorReport);
//        checkPatterns(config, errorReport);  // будет проверяться только для метрик графита
//        checkForbiddenSymbols(config, errorReport); //TODO sf

        if (errorReport.length() > 0) {
            throw new ConfigValidationException(errorReport.toString());
        }
    }

    public void validateMetricFunctions(String metricField) throws ConfigValidationException {
        if (forbiddenFunctionsPattern == null) {
            return;
        }

        Matcher forbiddenFunctions = forbiddenFunctionsPattern.matcher(metricField);
        while (forbiddenFunctions.find()) {
            if (allowedFunctionsPattern == null
                || !allowedFunctionsPattern.matcher(forbiddenFunctions.group(1)).find()) {
                throw new ConfigValidationException(
                    String.format("Forbidden function '%s' in '%s", forbiddenFunctions.group(1), metricField)
                );
            }
        }
    }

    public void validateStatfaceReportConfig(StatfaceReportConfig config) throws ConfigValidationException {
        config.getFields().forEach(field ->
            validateMetricFunctions(field.getField())
        );
        // TODO: add validations
    }


    public static void checkPatterns(GraphiteMetricConfig config) throws ConfigValidationException {
        Set<String> splits = new HashSet<>();
        String metricName = config.getMetricName();

        StringBuilder errorReport = new StringBuilder();
        for (MetricSplit split : config.getSplits()) {
            splits.add(split.getName());

            String sVar = split.getSplitVariable();
            if (!metricName.contains(sVar)) {
                errorReport.append("Split '").append(sVar).append("' ")
                    .append("does't exist in metric '").append(metricName).append("'\n");
            }
        }

        checkSplitsForPatterns(splits, StringTemplate.getVariablesFromString(metricName).stream(), errorReport);

        if (errorReport.length() > 0) {
            throw new ConfigValidationException(errorReport.toString());
        }
    }

    public static void checkSplitsForPatterns(Set<String> splits, Stream<String> patterns, StringBuilder errorReport) {
        String missingPatterns = patterns
            .filter(variable -> !splits.contains(variable))
            .distinct()
            .map(s -> '\'' + s + '\'')
            .collect(Collectors.joining(", "));

        if (!missingPatterns.isEmpty()) {
            errorReport.append("Patterns ").append(missingPatterns).append(" doesn't exist in splits\n");
        }
    }

    private void checkForbiddenSymbols(GraphiteMetricConfig config, StringBuilder errorReport) {
        if (config.getMetricName().matches(ESCAPE_SYMBOLS_PATTERN)) {
            errorReport.append("Forbidden symbol in '").append(config.getMetricName()).append("'\n");
        }
        for (MetricSplit split : config.getSplits()) {
            if (split.getName().matches(ESCAPE_SYMBOLS_PATTERN)) {
                errorReport.append("Forbidden symbol in '").append(split.getName()).append("'\n");
            }
        }
    }

    private void checkSplitCount(GraphiteMetricConfig config, StringBuilder errorReport) {
        List<MetricSplit> splits = config.getSplits();
        if (splits != null && splits.size() > MAX_SPLIT_COUNT) {
            errorReport.append("Split count(").append(splits.size()).append(") " +
                "exceeded MAX_SPLIT_COUNT(").append(MAX_SPLIT_COUNT).append(")\n");
        }
    }

}
