package ru.yandex.market.clickphite.config.validation.context;

import ru.yandex.market.clickphite.config.metric.MetricSplit;
import ru.yandex.market.clickphite.config.metric.SolomonSensorConfig;
import ru.yandex.market.clickphite.metric.StringTemplate;

import java.util.Arrays;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * @author Alexander Kedrik <a href="mailto:alkedr@yandex-team.ru"></a>
 * @date 17.08.2018
 */
public class SolomonSensorConfigValidator {
    private static final List<String> REQUIRED_LABELS =
        Arrays.asList("project", "service", "cluster", "sensor", "period");

    private SolomonSensorConfigValidator() {
    }

    public static void validate(SolomonSensorConfig config) throws ConfigValidationException {
        StringBuilder errorReport = new StringBuilder();

        validateSplits(config, errorReport);
        validateRequiredLabel(config, errorReport);

        if (errorReport.length() > 0) {
            throw new ConfigValidationException(errorReport.toString());
        }
    }

    private static void validateSplits(SolomonSensorConfig config, StringBuilder errorReport) {
        Set<String> splits = config.getSplits().stream()
            .map(MetricSplit::getName)
            .collect(Collectors.toSet());
        Stream<String> patterns = config.getLabels().values().stream()
            .flatMap(value -> StringTemplate.getVariablesFromString(value).stream());

        ConfigValidator.checkSplitsForPatterns(splits, patterns, errorReport);
    }

    private static void validateRequiredLabel(SolomonSensorConfig config, StringBuilder errorReport) {
        Set<String> configLabels = config.getLabels().keySet();
        String missingLabels = REQUIRED_LABELS.stream()
            .filter(label -> !configLabels.contains(label))
            .map(s -> '\'' + s + '\'')
            .collect(Collectors.joining(", "));

        if (!missingLabels.isEmpty()) {
            errorReport.append("Labels ").append(missingLabels).append(" is required\n");
        }
    }
}
