package ru.yandex.market.clickphite.dashboard;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import ru.yandex.market.clickphite.ProcessStatus;
import ru.yandex.market.clickphite.config.dashboard.DashboardConfig;
import ru.yandex.market.clickphite.config.metric.GraphiteMetricConfig;
import ru.yandex.market.clickphite.config.metric.MetricSplit;
import ru.yandex.market.clickphite.graphite.Metric;
import ru.yandex.market.dashboard.DashboardGraph;
import ru.yandex.market.dashboard.DashboardRow;
import ru.yandex.market.dashboard.GrafanaDashboard;

import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 15/05/15
 */
public class DashboardContext {

    private static final Logger log = LogManager.getLogger();

    private static final String ZOOMED_POSTFIX = " zoomed";

    private final String id;
    private final DashboardConfig dashboardConfig;
    private final String dashboardDataSource;
    private final DashboardQueries queries;
    private final GraphiteMetricConfig metricConfig;
    private final Set<String> tags;

    private volatile boolean actual = false;
    private final ProcessStatus status = new ProcessStatus();

    private final Set<DashboardMetric> metrics = new LinkedHashSet<DashboardMetric>();

    public DashboardContext(String id, DashboardConfig dashboardConfig, String dashboardDataSource,
                            DashboardQueries queries, GraphiteMetricConfig metricConfig, Set<String> tags) {
        this.id = id;
        this.dashboardConfig = dashboardConfig;
        this.dashboardDataSource = dashboardDataSource;
        this.queries = queries;
        this.metricConfig = metricConfig;
        this.tags = tags;
    }

    public synchronized void notifyMetric(String... metric) {
        if (metrics.add(new DashboardMetric(metric))) {
            actual = false;
        }
    }

    public synchronized void setDashboardMetrics(List<DashboardMetric> metrics) {
        this.metrics.clear();
        this.metrics.addAll(metrics);
        actual = false;
    }

    public synchronized boolean existMetrics() {
        return !this.metrics.isEmpty();
    }

    public synchronized GrafanaDashboard createGrafanaDashboard() {
        List<DashboardRow> rows = new ArrayList<>();

        List<DashboardGraph> graphs = new ArrayList<>();
        for (DashboardMetric metric : metrics) {
            graphs.addAll(createGraphs(metric));
        }

        for (int i = 0; i < graphs.size(); i += dashboardConfig.getGraphsInRow()) {
            List<DashboardGraph> rowGraphs = new ArrayList<>();
            for (int j = i; j < Math.min(graphs.size(), i + dashboardConfig.getGraphsInRow()); j++) {
                rowGraphs.add(graphs.get(j));
            }
            rows.add(new DashboardRow(rowGraphs));
        }
        return new GrafanaDashboard(
            id, dashboardConfig.getTitle(), dashboardDataSource, rows, tags
        );
    }

    private List<DashboardGraph> createGraphs(DashboardMetric split) {
        String title = createTitle(split);
        List<String> metricNames = createMetricsNames(split);

        if (!metricConfig.getType().isQuantile()) {
            return Collections.singletonList(new DashboardGraph(title, metricNames));
        }

        List<DashboardGraph> graphs = new ArrayList<>(2);
        graphs.add(new DashboardGraph(title, metricNames.subList(0, metricNames.size() / 2)));
        graphs.add(
            new DashboardGraph(title + ZOOMED_POSTFIX, metricNames.subList(metricNames.size() / 2, metricNames.size()))
        );
        return graphs;
    }

    public String createTitle(DashboardMetric metric) {
        List<MetricSplit> splits = metricConfig.getSplits();
        StringBuilder titleBuilder = new StringBuilder();

        for (int i = 0; i < splits.size(); i++) {
            MetricSplit split = splits.get(i);
            if (i > 0) {
                titleBuilder.append(", ");
            }
            titleBuilder.append(split.getName()).append(": ").append(metric.getSplit(i));
        }
        return titleBuilder.toString();
    }

    private List<String> createMetricsNames(DashboardMetric metric) {
        String metricName = metricConfig.getGraphiteName();

        List<MetricSplit> splits = metricConfig.getSplits();
        for (int i = 0; i < splits.size(); i++) {
            metricName = metricName.replace(splits.get(i).getSplitVariable(), metric.getSplit(i));
        }
        if (metricConfig.getType().isQuantile()) {
            List<String> quantiles = metricConfig.getQuantiles();
            List<String> metricNames = new ArrayList<>(quantiles.size());
            for (int i = 0; i < quantiles.size(); i++) {
                String name = metricName + "." + Metric.escapeName(quantiles.get(i));
                metricNames.add(name);
            }
            return metricNames;
        } else {
            return Collections.singletonList(metricName);
        }
    }

    public DashboardQueries getQueries() {
        return queries;
    }

    public DashboardConfig getDashboardConfig() {
        return dashboardConfig;
    }

    public GraphiteMetricConfig getMetricConfig() {
        return metricConfig;
    }

    public String getId() {
        return id;
    }

    public ProcessStatus getStatus() {
        return status;
    }

    public boolean isActual() {
        return actual;
    }

    public void setActual(boolean actual) {
        this.actual = actual;
    }

    @Override
    public String toString() {
        return "DashboardContext{" +
            "id='" + id + '\'' +
            '}';
    }
}
