package ru.yandex.market.clickphite.dashboard;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.market.clickhouse.ClickhouseTemplate;
import ru.yandex.market.clickphite.ClickphiteService;
import ru.yandex.market.clickphite.config.ConfigurationService;
import ru.yandex.market.clickphite.config.metric.MetricSplit;
import ru.yandex.market.clickphite.graphite.Metric;
import ru.yandex.market.dashboard.GrafanaDashboardUploader;
import ru.yandex.market.monitoring.ComplicatedMonitoring;
import ru.yandex.market.monitoring.MonitoringUnit;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 13/05/15
 */
public class DashboardService implements Runnable {

    private static final Logger log = LogManager.getLogger();

    private ConfigurationService configurationService;
    private GrafanaDashboardUploader grafanaDashboardUploader;
    private ClickhouseTemplate clickhouseTemplate;
    private ComplicatedMonitoring monitoring;
    private ClickphiteService clickphiteService;

    private int dashboardRebuildIntervalHours = 24;
    private int dashboardCheckIntervalHours = 6;
    private final MonitoringUnit dashboardUnit = new MonitoringUnit("Dashboard");

    public void afterPropertiesSet() throws Exception {
        monitoring.addUnit(dashboardUnit);
        new Thread(this, "DashboardService").start();
    }

    @Override
    public void run() {
        log.info("Starting Dashboard Service");

        while (!Thread.interrupted()) {
            if (clickphiteService.isMaster()) {
                updateDashboards();
            } else {
                dashboardUnit.ok();
            }
            try {
                TimeUnit.HOURS.sleep(dashboardCheckIntervalHours);
            } catch (InterruptedException ignored) {
            }
        }
    }

    public void updateDashboards() {
        List<DashboardContext> dashboards = configurationService.getConfiguration().getDashboardContexts();
        List<DashboardContext> failedDashboards = new ArrayList<>();
        for (DashboardContext dashboard : dashboards) {
            try {
                updateDashboard(dashboard);
            } catch (Exception e) {
                log.error("Failed to update dashboard: " + dashboard.getId(), e);
                log.info("Waiting 1 min...");
                try {
                    Thread.sleep(TimeUnit.MINUTES.toMillis(1));
                } catch (InterruptedException ignored) {
                }
                failedDashboards.add(dashboard);
            }
        }

        if (failedDashboards.isEmpty()) {
            dashboardUnit.ok();
        } else {
            dashboardUnit.warning("Failed to update dashboard(s): " +
                failedDashboards.stream().map(DashboardContext::getId).collect(Collectors.joining(", ")));
        }
    }

    private void updateDashboard(DashboardContext dashboard) throws IOException {
        if (needRebuild(dashboard)) {
            rebuildDashboard(dashboard);
        }
        if (!dashboard.isActual() && dashboard.existMetrics()) {
            uploadDashboard(dashboard);
        }
    }

    private void rebuildDashboard(DashboardContext dashboard) {
        final List<MetricSplit> splits = dashboard.getMetricConfig().getSplits();
        log.info("Rebuilding dashboard: " + dashboard.getId());
        final List<DashboardMetric> metrics = clickhouseTemplate.query(
            dashboard.getQueries().getUpdateQuery(),
            (rs, rowNum) -> {
                String[] values = new String[splits.size()];
                for (int i = 0; i < splits.size(); i++) {
                    MetricSplit split = splits.get(i);
                    String splitValue = Metric.escapeName(rs.getString(split.getName()));
                    values[i] = splitValue;
                }
                return new DashboardMetric(values);
            }
        );
        dashboard.setDashboardMetrics(metrics);
        dashboard.getStatus().setLastProcessTimeMillis(System.currentTimeMillis());
        log.info("Rebuild success. Dashboard: " + dashboard.getId());
    }

    private void uploadDashboard(DashboardContext dashboard) throws IOException {
        log.info("Uploading new version of dashboard: " + dashboard.getId());
        grafanaDashboardUploader.getAndUpload(dashboard.createGrafanaDashboard());
        log.info("Upload success. Dashboard: " + dashboard.getId());
    }

    private boolean needRebuild(DashboardContext dashboardContext) {
        long timePassedMillis = System.currentTimeMillis() - dashboardContext.getStatus().getLastProcessTimeMillis();
        return TimeUnit.MILLISECONDS.toHours(timePassedMillis) >= dashboardRebuildIntervalHours;
    }

    @Required
    public void setClickhouseTemplate(ClickhouseTemplate clickhouseTemplate) {
        this.clickhouseTemplate = clickhouseTemplate;
    }

    @Required
    public void setConfigurationService(ConfigurationService configurationService) {
        this.configurationService = configurationService;
    }

    @Required
    public void setGrafanaDashboardUploader(GrafanaDashboardUploader grafanaDashboardUploader) {
        this.grafanaDashboardUploader = grafanaDashboardUploader;
    }

    @Required
    public void setMonitoring(ComplicatedMonitoring monitoring) {
        this.monitoring = monitoring;
    }

    @Required
    public void setClickphiteService(ClickphiteService clickphiteService) {
        this.clickphiteService = clickphiteService;
    }

    public void setDashboardRebuildIntervalHours(int dashboardRebuildIntervalHours) {
        this.dashboardRebuildIntervalHours = dashboardRebuildIntervalHours;
    }

    public void setDashboardCheckIntervalHours(int dashboardCheckIntervalHours) {
        this.dashboardCheckIntervalHours = dashboardCheckIntervalHours;
    }
}
