package ru.yandex.market.clickphite.dictionary;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.market.clickhouse.ClickHouseSource;
import ru.yandex.market.clickhouse.ClickhouseTemplate;
import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnTypeBase;
import ru.yandex.market.monitoring.ComplicatedMonitoring;
import ru.yandex.market.monitoring.MonitoringUnit;

import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author Denis Khurtin <dkhurtin@yandex-team.ru>
 */
public class ClickhouseService implements InitializingBean {

    private static final Logger log = LogManager.getLogger();

    private final MonitoringUnit hostsMonitoringUnit = new MonitoringUnit("Cluster hosts");

    private ClickhouseTemplate clickhouseTemplate;
    private ClickHouseSource clickHouseSource;
    private ComplicatedMonitoring monitoring;

    private String defaultClickhouseHost = "localhost";

    @Override
    public void afterPropertiesSet() throws Exception {
        monitoring.addUnit(hostsMonitoringUnit);
    }

    public List<String> getClusterHosts() {
        try {
            //language=SQL
            List<String> hosts = clickhouseTemplate.query(
                String.format(
                    "SELECT host_name FROM system.clusters WHERE cluster = '%s'",
                    clickHouseSource.getCluster()
                ),
                (rs, rowNum) -> rs.getString("host_name")
            );
            hostsMonitoringUnit.ok();

            if (hosts.isEmpty()) {
                // Non-distributed mode
                return Collections.singletonList(defaultClickhouseHost);
            }

            return hosts;
        } catch (Exception e) {
            String message = "Can't load cluster hosts";
            log.error(message, e);
            hostsMonitoringUnit.critical(message, e);

            return Collections.emptyList();
        }
    }

    public void createDatabaseIfNotExists(String database, String host) {
        //language=SQL
        clickhouseTemplate.update("CREATE DATABASE IF NOT EXISTS " + database, host);
    }

    public boolean tableExists(String table, String host) {
        //language=SQL
        return clickhouseTemplate.queryForInt("EXISTS TABLE " + table, host) > 0;
    }

    public void createTable(String table, List<Column> columns, TableEngine engine, String host, String engineSpec) {
        String columnsLine = columns.stream()
            .map(c -> c.getName() + " " + c.getType().toClickhouseDDL() +
                    (Strings.isNullOrEmpty(c.getDefaultExpr()) ? "" : " DEFAULT " + c.getDefaultExpr()))
            .collect(Collectors.joining(", "));
        //language=SQL
        clickhouseTemplate.update(
            String.format(
                    "CREATE TABLE %s (%s) ENGINE = %s%s",
                    table,
                    columnsLine,
                    engine.getEngineName(),
                    engineSpec),
                host
        );
    }

    public void dropTable(String table, String host) {
        //language=SQL
        clickhouseTemplate.update(String.format("DROP TABLE IF EXISTS %s", table), host);
    }

    public void doubleRenameTable(String fromTable1, String toTable1, String fromTable2, String toTable2, String host) {
        //language=SQL
        clickhouseTemplate.update(
            String.format("RENAME TABLE %s TO %s, %s TO %s", fromTable1, toTable1, fromTable2, toTable2), host
        );
    }


    public BulkUpdater createBulkUpdater(Dictionary dictionary, String table, int batchSize, String host) {
        return new BulkUpdater(dictionary, table, batchSize, host);
    }


    public class BulkUpdater {
        private final Dictionary dictionary;
        private final String insertSql;
        private final StringBuilder sqlBuilder;
        private final int batchSize;
        private final String host;

        private int size;

        private BulkUpdater(Dictionary dictionary, String table, int batchSize, String host) {
            this.dictionary = dictionary;
            String columnsLine = dictionary.getColumns().stream()
                    .map(Column::getName)
                    .collect(Collectors.joining(", "));
            //language=SQL
            this.insertSql = "INSERT INTO " + table + " (" + columnsLine + ") VALUES\n";
            this.sqlBuilder = new StringBuilder(insertSql.length() + 2 * batchSize * 100);
            this.size = 0;
            this.batchSize = batchSize;
            this.host = host;

            this.sqlBuilder.append(insertSql);
        }

        public synchronized void submit(List<Object> values) {
            List<Column> columns = dictionary.getColumns();
            Preconditions.checkState(
                columns.size() == values.size(),
                "%s != %s",
                columns.size(),
                values.size()
            );

            sqlBuilder.append('(');
            for (int i = 0; i < columns.size(); ++i) {
                Column column = columns.get(i);
                ColumnTypeBase type = column.getType();
                if (i > 0) {
                    sqlBuilder.append(',');
                }
                if (values.get(i) == null) {
                    Preconditions.checkNotNull(
                        column.getDefaultValue(),
                        "Column %s cannot be null. Values: %s",
                        column.getName(),
                        values
                    );
                    sqlBuilder.append(column.getDefaultValue());
                } else {
                    Preconditions.checkState(
                        type.validate(values.get(i)),
                        "Invalid value for column %s (%s): %s. Values: %s",
                        column.getName(),
                        column.getType().name(),
                        values.get(i),
                        values
                    );
                    type.formatItem(values.get(i), sqlBuilder);
                }
            }
            sqlBuilder
                .append(')')
                .append('\n');

            if (++size == batchSize) {
                flush();
            }
        }

        public synchronized void done() {
            flush();
        }

        private void flush() {
            if (size > 0) {
                clickhouseTemplate.update(sqlBuilder.toString(), host);
                sqlBuilder.setLength(insertSql.length());
                size = 0;
            }
        }
    }

    @Required
    public void setClickhouseTemplate(ClickhouseTemplate clickhouseTemplate) {
        this.clickhouseTemplate = clickhouseTemplate;
    }

    @Required
    public void setClickHouseSource(ClickHouseSource clickHouseSource) {
        this.clickHouseSource = clickHouseSource;
    }

    @Required
    public void setMonitoring(ComplicatedMonitoring monitoring) {
        this.monitoring = monitoring;
    }

    @Required
    public void setDefaultClickhouseHost(String defaultClickhouseHost) {
        this.defaultClickhouseHost = defaultClickhouseHost;
    }
}
