package ru.yandex.market.clickphite.dictionary;

import com.google.common.annotations.VisibleForTesting;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.market.clickphite.ClickphiteService;
import ru.yandex.market.monitoring.ComplicatedMonitoring;
import ru.yandex.market.monitoring.MonitoringUnit;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import static java.util.stream.Collectors.joining;

/**
 * @author Tatiana Litvinenko <a href="mailto:tanlit@yandex-team.ru"></a>
 * @date 27.05.2015
 */
public class DictionaryService implements Runnable, InitializingBean, DisposableBean {
    private static final Logger log = LogManager.getLogger();

    private static final int DEFAULT_PERIOD_MILLIS = (int) TimeUnit.MINUTES.toMillis(30);
    private static final int BATCH_SIZE = 100_000;
    private static final int ALLOWED_RETRIES = 3;
    private static final String NEW_POSTFIX = "_new";
    private static final String OLD_POSTFIX = "_old";

    private long periodMillis = DEFAULT_PERIOD_MILLIS;

    private final Map<String, MonitoringUnit> monitoringUnits = new HashMap<>();

    private final Thread dictionaryLoadThread = new Thread(this, "DictionaryService");

    private ClickhouseService clickhouseService;
    private ClickphiteService clickphiteService;
    private ComplicatedMonitoring monitoring;

    private List<DictionaryLoadTask> dictionaries;

    private boolean shouldLoadDictionaries;

    @Override
    public void afterPropertiesSet() throws Exception {
        if (shouldLoadDictionaries) {
            setupMonitoringUnits();
            dictionaryLoadThread.start();
        } else {
            log.info("Property clickphite.dictionaries.shouldLoad is set to false. Cancel loading");
        }
    }

    @VisibleForTesting
    protected void setupMonitoringUnits() {
        dictionaries.stream()
            .map(task -> task.getDictionary().getTable())
            .forEach(dictionary -> {
                    MonitoringUnit monitoringUnit = new MonitoringUnit("Dictionary " + dictionary);
                    monitoring.addUnit(monitoringUnit);
                    monitoringUnits.put(dictionary, monitoringUnit);
                }
            );
    }

    @Override
    public void destroy() {
        dictionaryLoadThread.interrupt();
    }

    @Override
    public void run() {
        log.info("Run loading dictionaries {}", dictionaries);

        while (!Thread.interrupted()) {
            try {
                updateDictionaries();
            } catch (Exception e) {
                log.error("Can't update dictionaries", e);
            }

            try {
                Thread.sleep(periodMillis);
            } catch (InterruptedException ignored) {
            }
        }

        log.info("DictionaryService has finished");
    }

    @VisibleForTesting
    protected void updateDictionaries() {
        if (clickphiteService.isMaster()) {
            log.info("Updating dictionaries " +
                dictionaries.stream().map(task -> task.getDictionary().getTable()).collect(joining(",")));

            List<String> hosts = clickhouseService.getClusterHosts();

            for (DictionaryLoadTask task : dictionaries) {
                updateDictionary(task, hosts);
            }
            log.info("Dictionaries update finished");
        } else {
            for (MonitoringUnit monitoringUnit : monitoringUnits.values()) {
                monitoringUnit.ok();
            }
        }
    }

    private void updateDictionary(DictionaryLoadTask task, List<String> hosts) {
        List<String> failingHosts = new ArrayList<>();
        String dictionaryTable = task.getDictionary().getTable();
        for (String host : hosts) {
            try {
                tryUpdateDictionaryOnHost(task, host);
            } catch (Exception exception) {
                failingHosts.add(host);
                String errorMessage = String.format(
                    "Can't update dictionary '%s' on hosts %s", dictionaryTable,
                    String.join(", ", failingHosts)
                );
                monitoringUnits.get(dictionaryTable).warning(errorMessage, exception);
            }
        }

        if (failingHosts.isEmpty()) {
            monitoringUnits.get(dictionaryTable).ok();
        }
    }

    private void tryUpdateDictionaryOnHost(DictionaryLoadTask task, String host) throws Exception {
        String dictionaryTable = task.getDictionary().getTable();
        log.info("Updating dictionary " + dictionaryTable + " to " + host);

        Exception lastException = null;
        for (int i = 0; i < ALLOWED_RETRIES; i++) {
            try {
                updateDictionaryOnHost(task, host);
                return;
            } catch (Exception e) {
                log.warn(String.format("Can't update dictionary '%s' on host '%s'", dictionaryTable, host), e);
                lastException = e;
            }
        }

        throw lastException;
    }

    private void updateDictionaryOnHost(DictionaryLoadTask task, String host) throws IOException {
        long startTime = System.currentTimeMillis();

        Dictionary dictionary = task.getDictionary();
        String dataTable = preProcessTables(dictionary, host);
        String tmpTable = dataTable + NEW_POSTFIX;

        DictionaryLoader dictionaryLoader = task.getLoader();
        try {
            // Загружаем данные и пишем их во временную таблицу table
            dictionaryLoader.load(dictionary, reader -> {
                ClickhouseService.BulkUpdater updater =
                    clickhouseService.createBulkUpdater(dictionary, tmpTable, BATCH_SIZE, host);

                task.getProcessor().insertData(dictionary, reader, updater::submit);

                updater.done();
            });

            postProcessTables(dataTable, host);

            long workTime = System.currentTimeMillis() - startTime;
            log.info(
                "Dictionary '{}' on host '{}' was updated in {} ms.", dictionary.getTable(), host, workTime
            );
        } finally {
            // Удаляем временные таблицы
            clickhouseService.dropTable(tmpTable, host);
        }
    }

    /*
    Если все успешно, переименовываем (Все таблицы переименовываются под глобальной блокировкой)
     */
    void postProcessTables(String tableName, String host) {
        clickhouseService.dropTable(tableName + OLD_POSTFIX, host);
        clickhouseService.doubleRenameTable(
            tableName, tableName + OLD_POSTFIX, tableName + NEW_POSTFIX, tableName, host
        );
    }

    /**
     * Создает (если не существует) таблицу tableName с указанным движком, в которой будут лежать сами данные.
     * От создания вьюхи поверх исходной таблицы отказались,
     * т.к. не гарантировано что запрос к вьюхе использует индексы исходной таблицы
     * Создает временную таблицу tableName_data_new для обновление данных.
     */
    String preProcessTables(Dictionary dict, String host) {
        clickhouseService.createDatabaseIfNotExists(dict.getDb(), host);

        String dataTable = dict.getDb() + "." + dict.getTable();

        if (!clickhouseService.tableExists(dataTable, host)) {
            clickhouseService.createTable(
                dataTable,
                dict.getAllColumns(),
                dict.getEngine(),
                host,
                dict.getEngineSpec());
        }


        clickhouseService.dropTable(dataTable + NEW_POSTFIX, host);
        clickhouseService.createTable(dataTable + NEW_POSTFIX,
            dict.getAllColumns(),
            dict.getEngine(),
            host,
            dict.getEngineSpec());

        return dataTable;
    }

    @Required
    public void setClickhouseService(ClickhouseService clickhouseService) {
        this.clickhouseService = clickhouseService;
    }

    @Required
    public void setClickphiteService(ClickphiteService clickphiteService) {
        this.clickphiteService = clickphiteService;
    }

    @Required
    public void setMonitoring(ComplicatedMonitoring monitoring) {
        this.monitoring = monitoring;
    }

    @Required
    public void setDictionaries(List<DictionaryLoadTask> dictionaries) {
        this.dictionaries = dictionaries;
    }

    @Required
    public void setShouldLoadDictionaries(boolean shouldLoadDictionaries) {
        this.shouldLoadDictionaries = shouldLoadDictionaries;
    }
}
