package ru.yandex.market.clickphite.dictionary;

import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;

import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

/**
 * Created by astepanel on 19.06.17.
 */
public interface MergeTreeDictionary extends Dictionary {
    TableEngine ENGINE = TableEngine.MERGE_TREE;
    Integer DEFAULT_GRANULARITY = 2048;
    Column FAKE_DATE_COLUMN = new Column("fake_date", ColumnType.Date, "'0000-00-00'");

    /**
     * Движок предполагает что таблица может быть партиционирована по дню.
     * Если колонки с датой для партиционирования в таблице нет, используйте FAKE_DATE_COLUMN.
     * В этом случае, если в таблице есть вычисляемые поля, FAKE_DATE_COLUMN нужно добавить в их перечень
     *
     * @return Колонку, содержащую дату, по которой таблица может быть партиционирована
     */
    Column getDateColumn();

    @Override
    default List<Column> getCalculatedColumns() {
        if (FAKE_DATE_COLUMN.equals(getDateColumn())) {
            return Collections.singletonList(FAKE_DATE_COLUMN);
        }
        return Collections.emptyList();
    }

    /**
     * @return Перечень колонок, входящих в первичный ключ
     */
    List<Column> getPrimaryKey();

    /**
     * @return Гранулярность первичного ключа
     */
    default Integer getGranularity() {
        return DEFAULT_GRANULARITY;
    }

    @Override
    default TableEngine getEngine() {
        return TableEngine.MERGE_TREE;
    }

    @Override
    default String getEngineSpec() {
        String primaryKeyLine = getPrimaryKey().stream()
                .map(Column::getName)
                .collect(Collectors.joining(", "));
        return String.format("(%s, (%s), %d)", getDateColumn().getName(), primaryKeyLine, getGranularity());
    }
}
